<?php

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\User;
use App\Models\Customer;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

echo "╔══════════════════════════════════════════════════════════════╗\n";
echo "║   Currency Rate Access - Setup Verification                 ║\n";
echo "╚══════════════════════════════════════════════════════════════╝\n\n";

$allChecks = true;

// Check 1: Column exists
echo "✓ Check 1: Verifying database column exists...\n";
if (Schema::hasColumn('users', 'currency_rate_access')) {
    echo "  ✓ Column 'currency_rate_access' exists in users table\n\n";
} else {
    echo "  ✗ FAILED: Column 'currency_rate_access' NOT found!\n";
    echo "  → Run: php artisan migrate\n\n";
    $allChecks = false;
}

// Check 2: Model configuration
echo "✓ Check 2: Verifying User model configuration...\n";
$user = new User();
$fillable = $user->getFillable();
$casts = $user->getCasts();

if (in_array('currency_rate_access', $fillable)) {
    echo "  ✓ 'currency_rate_access' is in \$fillable array\n";
} else {
    echo "  ✗ FAILED: 'currency_rate_access' NOT in \$fillable array\n";
    $allChecks = false;
}

if (isset($casts['currency_rate_access']) && $casts['currency_rate_access'] === 'boolean') {
    echo "  ✓ 'currency_rate_access' is cast to boolean\n\n";
} else {
    echo "  ✗ FAILED: 'currency_rate_access' NOT properly cast\n\n";
    $allChecks = false;
}

// Check 3: Test with actual data
echo "✓ Check 3: Testing with actual customers...\n";
$customersWithUsers = Customer::whereNotNull('user_id')
    ->with('user')
    ->take(5)
    ->get();

if ($customersWithUsers->count() > 0) {
    echo "  ✓ Found {$customersWithUsers->count()} customers with user accounts\n";
    echo "\n  Customer List:\n";
    echo "  " . str_repeat("-", 70) . "\n";
    echo "  " . str_pad("ID", 5) . str_pad("Customer Name", 30) . str_pad("User ID", 10) . str_pad("Access", 10) . "\n";
    echo "  " . str_repeat("-", 70) . "\n";
    
    foreach ($customersWithUsers as $customer) {
        $access = $customer->user->currency_rate_access ? 'Enabled' : 'Disabled';
        echo "  " . str_pad($customer->id, 5) 
             . str_pad(substr($customer->customer_name, 0, 28), 30) 
             . str_pad($customer->user_id, 10) 
             . str_pad($access, 10) 
             . "\n";
    }
    echo "  " . str_repeat("-", 70) . "\n\n";
} else {
    echo "  ⚠ WARNING: No customers with user accounts found\n\n";
}

// Check 4: Test update functionality
echo "✓ Check 4: Testing update functionality...\n";
$testCustomer = Customer::whereNotNull('user_id')->with('user')->first();

if ($testCustomer && $testCustomer->user) {
    echo "  Testing with: {$testCustomer->customer_name} (User ID: {$testCustomer->user->id})\n";
    
    $originalValue = $testCustomer->user->currency_rate_access ?? 0;
    echo "  Original value: " . ($originalValue ? 'Enabled' : 'Disabled') . "\n";
    
    // Toggle it
    $newValue = $originalValue ? 0 : 1;
    $testCustomer->user->currency_rate_access = $newValue;
    $saved = $testCustomer->user->save();
    
    if ($saved) {
        echo "  ✓ Update successful\n";
        
        // Verify
        $testCustomer->user->refresh();
        $verifiedValue = $testCustomer->user->currency_rate_access;
        
        if ($verifiedValue == $newValue) {
            echo "  ✓ Value verified in database: " . ($verifiedValue ? 'Enabled' : 'Disabled') . "\n";
            
            // Restore original value
            $testCustomer->user->currency_rate_access = $originalValue;
            $testCustomer->user->save();
            echo "  ✓ Original value restored\n\n";
        } else {
            echo "  ✗ FAILED: Value not updated correctly\n\n";
            $allChecks = false;
        }
    } else {
        echo "  ✗ FAILED: Could not save user model\n\n";
        $allChecks = false;
    }
} else {
    echo "  ⚠ WARNING: No test customer available\n\n";
}

// Check 5: View file exists
echo "✓ Check 5: Verifying Livewire view file...\n";
$viewFile = resource_path('views/livewire/customer-management.blade.php');
if (file_exists($viewFile)) {
    echo "  ✓ View file exists\n";
    $viewContent = file_get_contents($viewFile);
    
    if (strpos($viewContent, 'toggleCurrencyRateAccess') !== false) {
        echo "  ✓ toggleCurrencyRateAccess method referenced in view\n";
    } else {
        echo "  ✗ FAILED: toggleCurrencyRateAccess NOT found in view\n";
        $allChecks = false;
    }
    
    if (strpos($viewContent, 'Currency Rate Access') !== false) {
        echo "  ✓ Column header found in view\n\n";
    } else {
        echo "  ✗ FAILED: Column header NOT found in view\n\n";
        $allChecks = false;
    }
} else {
    echo "  ✗ FAILED: View file not found\n\n";
    $allChecks = false;
}

// Check 6: Component class exists
echo "✓ Check 6: Verifying Livewire component...\n";
$componentFile = app_path('Livewire/CustomerManagement.php');
if (file_exists($componentFile)) {
    echo "  ✓ Component file exists\n";
    $componentContent = file_get_contents($componentFile);
    
    if (strpos($componentContent, 'function toggleCurrencyRateAccess') !== false) {
        echo "  ✓ toggleCurrencyRateAccess method found in component\n\n";
    } else {
        echo "  ✗ FAILED: toggleCurrencyRateAccess method NOT found\n\n";
        $allChecks = false;
    }
} else {
    echo "  ✗ FAILED: Component file not found\n\n";
    $allChecks = false;
}

// Summary
echo "╔══════════════════════════════════════════════════════════════╗\n";
if ($allChecks) {
    echo "║                     ✓ ALL CHECKS PASSED                     ║\n";
    echo "╚══════════════════════════════════════════════════════════════╝\n\n";
    echo "✓ Setup is complete and working!\n";
    echo "→ Visit the Customer Management page to test the checkbox\n";
} else {
    echo "║                   ✗ SOME CHECKS FAILED                      ║\n";
    echo "╚══════════════════════════════════════════════════════════════╝\n\n";
    echo "⚠ Please review the failed checks above\n";
    echo "→ See CURRENCY_RATE_ACCESS_IMPLEMENTATION.md for troubleshooting\n";
}

echo "\n";

