<?php

require_once 'vendor/autoload.php';

use App\Models\Currency;
use App\Models\User;
use App\Jobs\SendCurrencyRateNotificationJob;
use App\Services\NotificationTranslationService;
use Illuminate\Support\Facades\Log;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "=== Testing Specific Issue: Same Old/New Rates ===\n\n";

// Get a currency to test with
$currency = Currency::first();
if (!$currency) {
    echo "❌ No currencies found in database\n";
    exit(1);
}

echo "Testing with currency: {$currency->currency_name}\n";
echo "Current buy rate: {$currency->buy_rate}\n";
echo "Current sell rate: {$currency->sell_rate}\n\n";

$translationService = new NotificationTranslationService();

// Test the exact scenario you reported: same old and new rates
echo "=== Test 1: Same Old and New Rates ===\n";
$sameRate = 1404.0;
$message = $translationService->getCurrencyRateMessage($currency, $sameRate, $sameRate, 'buy_rate');
echo "Message for same rates (1404 -> 1404): $message\n";

// Test with null old rate
echo "\n=== Test 2: Null Old Rate ===\n";
$message = $translationService->getCurrencyRateMessage($currency, $sameRate, null, 'buy_rate');
echo "Message for null old rate: $message\n";

// Test with string vs float comparison
echo "\n=== Test 3: String vs Float Comparison ===\n";
$oldRateString = "1404.00";
$newRateFloat = 1404.0;
$message = $translationService->getCurrencyRateMessage($currency, $newRateFloat, $oldRateString, 'buy_rate');
echo "Message for string old rate vs float new rate: $message\n";

// Test with different precision
echo "\n=== Test 4: Different Precision ===\n";
$oldRate = 1404.000;
$newRate = 1404.00;
$message = $translationService->getCurrencyRateMessage($currency, $newRate, $oldRate, 'buy_rate');
echo "Message for different precision (1404.000 -> 1404.00): $message\n";

// Test the actual Livewire component logic
echo "\n=== Test 5: Livewire Component Logic ===\n";

// Simulate what happens in the Livewire component
$field = 'buy_rate';
$newValue = 1404.0; // Same as current rate

// Get the old rate BEFORE updating (this is the key fix!)
$oldRate = $currency->$field;

echo "Current rate in database: $oldRate\n";
echo "New rate to set: $newValue\n";
echo "Are they equal? " . ($oldRate == $newValue ? 'Yes' : 'No') . "\n";
echo "Are they strictly equal? " . ($oldRate === $newValue ? 'Yes' : 'No') . "\n";

// Test the notification message generation
$message = $translationService->getCurrencyRateMessage($currency, $newValue, $oldRate, $field);
echo "Generated message: $message\n";

// Test with a different rate to see the difference
echo "\n=== Test 6: Different Rate ===\n";
$differentValue = 1500.0;
$message = $translationService->getCurrencyRateMessage($currency, $differentValue, $oldRate, $field);
echo "Generated message for different rate (1404 -> 1500): $message\n";

echo "\n=== Analysis ===\n";
echo "The issue you reported (same old/new rates showing as decrease) should now be fixed.\n";
echo "When rates are the same, the message should show 'نوێکرایەوە' (updated) instead of 'کەمبووەتەوە' (decreased).\n";
echo "The fix ensures proper type conversion and comparison logic.\n\n";

echo "To test in the admin panel:\n";
echo "1. Go to the currency management page\n";
echo "2. Update a currency rate to the SAME value it already has\n";
echo "3. Check the notification - it should show 'نوێکرایەوە' not 'کەمبووەتەوە'\n";
echo "4. Update a currency rate to a DIFFERENT value\n";
echo "5. Check the notification - it should show the correct increase/decrease\n";
