<?php

/**
 * Quick OneSignal Test
 * 
 * This script tests your OneSignal configuration
 */

// Load Laravel environment
require_once 'vendor/autoload.php';
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

echo "🔍 OneSignal Configuration Test\n";
echo "================================\n\n";

// Get configuration
$appId = config('onesignal.app_id');
$restApiKey = config('onesignal.rest_api_key');

echo "Configuration:\n";
echo "App ID: " . ($appId ? $appId : 'NOT SET') . "\n";
echo "REST API Key: " . ($restApiKey ? substr($restApiKey, 0, 10) . '... (length: ' . strlen($restApiKey) . ')' : 'NOT SET') . "\n\n";

if (!$appId || !$restApiKey) {
    echo "❌ Missing configuration. Please check your .env file.\n";
    exit(1);
}

// Test API call
echo "Testing OneSignal API...\n";

try {
    $client = new Client([
        'base_uri' => 'https://onesignal.com/api/v1/',
        'headers' => [
            'Authorization' => 'Basic ' . $restApiKey,
            'Content-Type' => 'application/json',
        ],
    ]);

    // Test 1: Get app info
    echo "1. Testing app info...\n";
    $response = $client->get('apps/' . $appId);
    $data = json_decode($response->getBody()->getContents(), true);
    
    echo "   ✅ App info retrieved successfully!\n";
    echo "   App Name: " . ($data['name'] ?? 'Unknown') . "\n";
    echo "   App ID: " . ($data['id'] ?? 'Unknown') . "\n\n";

    // Test 2: Send test notification
    echo "2. Testing notification send...\n";
    $payload = [
        'app_id' => $appId,
        'include_player_ids' => ['test-player-id-' . time()],
        'contents' => ['en' => 'Test notification from Laravel - ' . now()->format('Y-m-d H:i:s')],
        'data' => ['type' => 'test']
    ];

    $response = $client->post('notifications', [
        'json' => $payload,
    ]);

    $responseData = json_decode($response->getBody()->getContents(), true);
    
    if (isset($responseData['errors'])) {
        echo "   ❌ OneSignal returned errors:\n";
        foreach ($responseData['errors'] as $error) {
            echo "      - " . $error . "\n";
        }
    } else {
        echo "   ✅ Test notification sent successfully!\n";
        echo "   Notification ID: " . ($responseData['id'] ?? 'Unknown') . "\n";
        echo "   Recipients: " . ($responseData['recipients'] ?? 'Unknown') . "\n";
    }

} catch (GuzzleException $e) {
    echo "❌ API Error: " . $e->getMessage() . "\n";
    
    if ($e->hasResponse()) {
        $response = $e->getResponse();
        $body = $response->getBody()->getContents();
        echo "Response Body: " . $body . "\n";
    }
} catch (Exception $e) {
    echo "❌ General Error: " . $e->getMessage() . "\n";
}

echo "\n✅ Test completed!\n";


