<?php

require_once 'vendor/autoload.php';

use App\Services\OneSignalService;
use App\Services\NotificationTranslationService;
use App\Models\User;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

try {
    echo "Testing OneSignal Kurdish text encoding...\n\n";
    
    // Get a user for testing
    $user = User::first();
    if (!$user) {
        echo "No users found in database\n";
        exit(1);
    }
    
    echo "Using user: {$user->email} (ID: {$user->id})\n\n";
    
    // Test the translation service
    $translationService = new NotificationTranslationService();
    $testMessage = $translationService->getTestMessage();
    
    echo "Test message from TranslationService:\n";
    echo "Message: $testMessage\n";
    echo "Length: " . strlen($testMessage) . "\n";
    echo "Bytes: " . implode(', ', array_map('ord', str_split($testMessage))) . "\n";
    echo "JSON encoded: " . json_encode(['message' => $testMessage], JSON_UNESCAPED_UNICODE) . "\n\n";
    
    // Test currency rate message
    $currencyRateMessage = $translationService->getCurrencyRateMessage(
        (object)['currency_name' => 'USD'],
        140900,
        140900
    );
    
    echo "Currency rate message from TranslationService:\n";
    echo "Message: $currencyRateMessage\n";
    echo "Length: " . strlen($currencyRateMessage) . "\n";
    echo "Bytes: " . implode(', ', array_map('ord', str_split($currencyRateMessage))) . "\n";
    echo "JSON encoded: " . json_encode(['message' => $currencyRateMessage], JSON_UNESCAPED_UNICODE) . "\n\n";
    
    // Test OneSignal service (but don't actually send)
    echo "Testing OneSignal payload construction...\n";
    $oneSignalService = new OneSignalService();
    
    // Create the payload manually to see what would be sent
    $payload = [
        'app_id' => config('onesignal.app_id'),
        'include_external_user_ids' => [(string)$user->id],
        'contents' => [
            'en' => $currencyRateMessage,
            'ku' => $currencyRateMessage // Kurdish Sorani support
        ],
        'data' => [
            'type' => 'currency_rate_update',
            'currency_code' => 'USD',
            'currency_name' => 'USD',
            'new_rate' => 140900,
            'old_rate' => 140900,
            'timestamp' => now()->toISOString(),
        ],
        'small_icon' => 'ic_stat_onesignal_default',
        'large_icon' => 'ic_stat_onesignal_default',
    ];
    
    $jsonPayload = json_encode($payload, JSON_UNESCAPED_UNICODE);
    
    echo "OneSignal payload:\n";
    echo "JSON length: " . strlen($jsonPayload) . "\n";
    echo "Contents (en): " . $payload['contents']['en'] . "\n";
    echo "Contents (ku): " . $payload['contents']['ku'] . "\n";
    echo "JSON preview: " . substr($jsonPayload, 0, 200) . "...\n\n";
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}







