<?php

/**
 * OneSignal Configuration Test Script
 * 
 * Run this script to test your OneSignal configuration:
 * php test_onesignal.php
 */

require_once 'vendor/autoload.php';

// Load Laravel environment
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Services\OneSignalService;
use App\Models\User;

echo "🔍 OneSignal Configuration Test\n";
echo "================================\n\n";

// Test 1: Check environment variables
echo "1. Checking Environment Variables:\n";
echo "   ONESIGNAL_APP_ID: " . (config('onesignal.app_id') ? '✅ Set' : '❌ Not Set') . "\n";
echo "   ONESIGNAL_REST_API_KEY: " . (config('onesignal.rest_api_key') ? '✅ Set' : '❌ Not Set') . "\n";
echo "   ONESIGNAL_USER_AUTH_KEY: " . (config('onesignal.user_auth_key') ? '✅ Set' : '❌ Not Set') . "\n\n";

// Test 2: Check database
echo "2. Checking Database:\n";
$usersWithPlayerIds = User::whereNotNull('onesignal_player_id')->count();
$totalUsers = User::count();
echo "   Users with OneSignal Player IDs: {$usersWithPlayerIds}\n";
echo "   Total Users: {$totalUsers}\n";
echo "   Coverage: " . ($totalUsers > 0 ? round(($usersWithPlayerIds / $totalUsers) * 100, 2) : 0) . "%\n\n";

// Test 3: Test OneSignal Service
echo "3. Testing OneSignal Service:\n";
try {
    $oneSignalService = new OneSignalService();
    
    // Test with a dummy player ID
    $testPlayerId = 'test-player-id-' . time();
    $translationService = new \App\Services\NotificationTranslationService();
    $testMessage = $translationService->getTestMessage();
    
    $result = $oneSignalService->sendNotificationToUsers(
        [$testPlayerId],
        $testMessage,
        ['type' => 'test']
    );
    
    if (is_array($result)) {
        if ($result['success'] === true) {
            echo "   ✅ OneSignal service is working correctly!\n";
            echo "   Response: " . json_encode($result['data']) . "\n";
        } else {
            echo "   ❌ OneSignal service error:\n";
            echo "   Error Type: " . ($result['error_type'] ?? 'unknown') . "\n";
            echo "   Error Message: " . ($result['error'] ?? 'Unknown error') . "\n";
            
            if (isset($result['onesignal_errors'])) {
                echo "   OneSignal API Errors: " . json_encode($result['onesignal_errors']) . "\n";
            }
        }
    } else {
        echo "   ❌ OneSignal service returned unexpected response format\n";
    }
} catch (Exception $e) {
    echo "   ❌ Exception occurred: " . $e->getMessage() . "\n";
}

echo "\n4. Recommendations:\n";

if (!config('onesignal.app_id')) {
    echo "   ❌ Add ONESIGNAL_APP_ID to your .env file\n";
}

if (!config('onesignal.rest_api_key')) {
    echo "   ❌ Add ONESIGNAL_REST_API_KEY to your .env file\n";
}

if ($usersWithPlayerIds === 0) {
    echo "   ⚠️  No users have OneSignal player IDs. Test the mobile app to generate player IDs.\n";
}

echo "\n✅ Test completed!\n";


