<?php

require_once 'vendor/autoload.php';

use App\Models\User;
use App\Models\Currency;
use App\Services\OneSignalService;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "Testing Notification Improvements\n";
echo "================================\n\n";

// Get a test user
$user = User::first();
if (!$user) {
    echo "No users found in database.\n";
    exit(1);
}

// Get a currency to test with
$currency = Currency::first();
if (!$currency) {
    echo "No currencies found in database.\n";
    exit(1);
}

echo "Testing with user: {$user->name} (ID: {$user->id})\n";
echo "Testing with currency: {$currency->currency_name}\n\n";

// Enable currency rate notifications for testing
$user->currency_rate_notifications = true;
$user->save();

echo "Currency rate notifications enabled for testing\n\n";

// Test the OneSignal service directly
echo "Testing OneSignal Service with Currency Name:\n";
echo "--------------------------------------------\n";

$oneSignalService = new OneSignalService();

// Test currency rate notification
echo "1. Testing currency rate notification with currency name...\n";
$result = $oneSignalService->sendCurrencyRateNotification($user, $currency, 1.25, 1.20);

if ($result['success']) {
    echo "   ✅ Currency rate notification sent successfully\n";
    echo "   📱 Message should now show '{$currency->currency_name}' instead of currency code\n";
} else {
    echo "   ❌ Currency rate notification failed: {$result['error']}\n";
}

// Restore original setting
$user->currency_rate_notifications = false;
$user->save();

echo "\nCurrency rate notifications disabled again\n";

echo "\nPerformance Improvements:\n";
echo "-------------------------\n";
echo "✅ Notifications now use currency names instead of codes\n";
echo "✅ Currency rate updates are now processed asynchronously via queue\n";
echo "✅ Database queries are optimized (only fetch user IDs, not full user objects)\n";
echo "✅ Notifications are sent in batch instead of individually\n";
echo "✅ Rate updates are no longer blocked by notification sending\n\n";

echo "To test the queue performance:\n";
echo "1. Update a currency rate in the admin panel\n";
echo "2. The rate should update immediately (no delay)\n";
echo "3. Notifications will be sent in the background via queue\n";
echo "4. Check the logs to see the notification job execution\n\n";

echo "Test completed!\n";
