<?php

/**
 * Test Kurdish Text Helper
 * 
 * Run with: php test_kurdish_helper.php
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Helpers\KurdishTextHelper;

echo "\n";
echo "==========================================\n";
echo " Testing Kurdish Text Helper\n";
echo "==========================================\n\n";

// Test 1: Detect corrupted text
echo "Test 1: Detect Corrupted Text\n";
echo "------------------------------\n";
$corruptedText = '???? ???';
$cleanText = 'ئاسۆ کوی';

echo "Input: '{$corruptedText}'\n";
echo "Is corrupted: " . (KurdishTextHelper::isCorrupted($corruptedText) ? 'YES' : 'NO') . "\n";
echo "Input: '{$cleanText}'\n";
echo "Is corrupted: " . (KurdishTextHelper::isCorrupted($cleanText) ? 'YES' : 'NO') . "\n\n";

// Test 2: Fix corrupted text
echo "Test 2: Fix Corrupted Text\n";
echo "------------------------------\n";
$fixed = KurdishTextHelper::fixCorruptedText($corruptedText);
echo "Input: '{$corruptedText}'\n";
echo "Fixed: '{$fixed}'\n";
echo "Expected: '{$cleanText}'\n";
echo "Match: " . ($fixed === $cleanText ? '✓ YES' : '✗ NO') . "\n\n";

// Test 3: Process method (combined fix + ensure UTF-8)
echo "Test 3: Process Method\n";
echo "------------------------------\n";
$processed = KurdishTextHelper::process($corruptedText);
echo "Input: '{$corruptedText}'\n";
echo "Processed: '{$processed}'\n";
echo "Expected: '{$cleanText}'\n";
echo "Match: " . ($processed === $cleanText ? '✓ YES' : '✗ NO') . "\n\n";

// Test 4: Test with actual database transaction
echo "Test 4: Test with Database Transaction\n";
echo "------------------------------\n";

use App\Models\AccountTransaction;

$transaction = AccountTransaction::find(129518);

if ($transaction) {
    echo "Transaction 129518 found\n";
    echo "Raw note from DB: '{$transaction->note}'\n";
    
    $fixedNote = KurdishTextHelper::process($transaction->note);
    echo "After processing: '{$fixedNote}'\n";
    echo "Expected: '{$cleanText}'\n";
    echo "Match: " . ($fixedNote === $cleanText ? '✓ YES' : '✗ NO') . "\n\n";
} else {
    echo "Transaction 129518 not found\n\n";
}

// Test 5: Test unknown corrupted text
echo "Test 5: Unknown Corrupted Text\n";
echo "------------------------------\n";
$unknownCorrupted = '??? ????';
$fixedUnknown = KurdishTextHelper::process($unknownCorrupted);
echo "Input: '{$unknownCorrupted}'\n";
echo "Fixed: " . ($fixedUnknown === null ? 'NULL (unfixable)' : "'{$fixedUnknown}'") . "\n";
echo "Expected behavior: Should return NULL for unknown corrupted text\n\n";

// Test 6: Test clean Kurdish text (should pass through unchanged)
echo "Test 6: Clean Kurdish Text\n";
echo "------------------------------\n";
$cleanKurdish = 'ڕاکێشان';
$processedClean = KurdishTextHelper::process($cleanKurdish);
echo "Input: '{$cleanKurdish}'\n";
echo "Processed: '{$processedClean}'\n";
echo "Match: " . ($processedClean === $cleanKurdish ? '✓ YES (unchanged)' : '✗ NO (changed)') . "\n\n";

echo "==========================================\n";
echo " Test Complete\n";
echo "==========================================\n\n";

echo "Summary:\n";
echo "- Kurdish text helper can detect corrupted text\n";
echo "- Known corrupted patterns are fixed using config mappings\n";
echo "- Unknown corrupted text returns NULL\n";
echo "- Clean Kurdish text passes through unchanged\n\n";

echo "Next Steps:\n";
echo "1. Test the API endpoint: GET /api/mobile/transactions\n";
echo "2. The API should automatically fix corrupted text on-the-fly\n";
echo "3. Add more mappings to config/kurdish_mappings.php as needed\n\n";



