<?php

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

echo "\n";
echo "========================================\n";
echo " Testing Kurdish Text Helper\n";
echo "========================================\n\n";

// Test if helper class exists
if (class_exists('App\Helpers\KurdishTextHelper')) {
    echo "✓ KurdishTextHelper class is loaded!\n\n";
    
    // Test the fix
    $corrupted = '???? ???';
    $expected = 'ئاسۆ کوی';
    
    try {
        $fixed = \App\Helpers\KurdishTextHelper::process($corrupted);
        
        echo "Input (corrupted): '{$corrupted}'\n";
        echo "Output (fixed):    '{$fixed}'\n";
        echo "Expected:          '{$expected}'\n\n";
        
        if ($fixed === $expected) {
            echo "✓ SUCCESS! Helper is working correctly!\n\n";
            echo "The API should now return correct Kurdish text.\n";
            echo "Test in Postman: GET /api/mobile/transactions\n";
        } else {
            echo "✗ MISMATCH! Helper found but not fixing correctly.\n\n";
            echo "Debugging info:\n";
            echo "- Fixed value: " . ($fixed === null ? 'NULL' : "'{$fixed}'") . "\n";
            echo "- Length: " . strlen($fixed ?? '') . " bytes\n";
            echo "- Encoding: " . mb_detect_encoding($fixed ?? '', ['UTF-8', 'ASCII'], true) . "\n\n";
            
            echo "Check config/kurdish_mappings.php for correct mapping.\n";
        }
    } catch (\Exception $e) {
        echo "✗ ERROR running helper:\n";
        echo $e->getMessage() . "\n\n";
        echo "Stack trace:\n";
        echo $e->getTraceAsString() . "\n";
    }
} else {
    echo "✗ KurdishTextHelper class NOT found!\n\n";
    echo "Troubleshooting steps:\n";
    echo "1. Run: composer dump-autoload\n";
    echo "2. Check file exists: app/Helpers/KurdishTextHelper.php\n";
    echo "3. Check namespace: App\\Helpers\n";
    echo "4. Clear Laravel cache: php artisan cache:clear\n";
}

echo "\n";
echo "========================================\n";
echo " Config Check\n";
echo "========================================\n\n";

try {
    $mappings = config('kurdish_mappings.mappings');
    
    if ($mappings && is_array($mappings)) {
        echo "✓ Config file loaded successfully!\n\n";
        echo "Current mappings:\n";
        foreach ($mappings as $corrupt => $correct) {
            echo "  '{$corrupt}' => '{$correct}'\n";
        }
    } else {
        echo "✗ Config not loaded or empty!\n";
        echo "1. Check file exists: config/kurdish_mappings.php\n";
        echo "2. Run: php artisan config:clear\n";
    }
} catch (\Exception $e) {
    echo "✗ Error loading config:\n";
    echo $e->getMessage() . "\n";
}

echo "\n";



