<?php

require_once 'vendor/autoload.php';

use App\Models\Currency;
use App\Models\User;
use App\Jobs\SendCurrencyRateNotificationJob;
use App\Services\NotificationTranslationService;
use Illuminate\Support\Facades\Log;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "=== Currency Rate Notification Fix Test ===\n\n";

// Get a currency to test with
$currency = Currency::first();
if (!$currency) {
    echo "❌ No currencies found in database\n";
    exit(1);
}

echo "Testing with currency: {$currency->currency_name}\n";
echo "Current buy rate: {$currency->buy_rate}\n";
echo "Current sell rate: {$currency->sell_rate}\n\n";

// Test 1: Test the NotificationTranslationService directly
echo "=== Test 1: NotificationTranslationService ===\n";
$translationService = new NotificationTranslationService();

// Test with different old and new rates
$oldBuyRate = 100.0;
$newBuyRate = 105.0;
$message = $translationService->getCurrencyRateMessage($currency, $newBuyRate, $oldBuyRate, 'buy_rate');
echo "Message for buy rate change (100 -> 105): $message\n";

$oldSellRate = 110.0;
$newSellRate = 108.0;
$message = $translationService->getCurrencyRateMessage($currency, $newSellRate, $oldSellRate, 'sell_rate');
echo "Message for sell rate change (110 -> 108): $message\n";

// Test with same rates
$message = $translationService->getCurrencyRateMessage($currency, $newBuyRate, $newBuyRate, 'buy_rate');
echo "Message for no change (105 -> 105): $message\n\n";

// Test 2: Test the SendCurrencyRateNotificationJob
echo "=== Test 2: SendCurrencyRateNotificationJob ===\n";

// Get a user to test with
$user = User::first();
if (!$user) {
    echo "❌ No users found in database\n";
    exit(1);
}

echo "Testing with user: {$user->name} (ID: {$user->id})\n";

// Test the job with different rates
try {
    $job = new SendCurrencyRateNotificationJob($currency, 'buy_rate', 105.0, 100.0, 'Test User');
    echo "✅ Job created successfully with old rate: 100.0, new rate: 105.0\n";
    
    // Test the job with same rates
    $job2 = new SendCurrencyRateNotificationJob($currency, 'sell_rate', 110.0, 110.0, 'Test User');
    echo "✅ Job created successfully with same rates: 110.0 -> 110.0\n";
    
} catch (Exception $e) {
    echo "❌ Error creating job: " . $e->getMessage() . "\n";
}

// Test 3: Simulate the Livewire component logic
echo "\n=== Test 3: Simulate Livewire Component Logic ===\n";

// Simulate the updateRate method logic
$field = 'buy_rate';
$newValue = 120.0;

// Get the old rate BEFORE updating (this is the key fix!)
$oldRate = $currency->$field;

echo "Old rate: $oldRate\n";
echo "New rate: $newValue\n";
echo "Rates equal: " . ($oldRate == $newValue ? 'Yes' : 'No') . "\n";

// Test the notification message generation
$message = $translationService->getCurrencyRateMessage($currency, $newValue, $oldRate, $field);
echo "Generated message: $message\n";

// Test 4: Check if user has notification preferences enabled
echo "\n=== Test 4: User Notification Preferences ===\n";
echo "User notifications enabled: " . ($user->notifications_enabled ? 'Yes' : 'No') . "\n";
echo "Currency rate notifications: " . ($user->currency_rate_notifications ? 'Yes' : 'No') . "\n";

if (!$user->notifications_enabled || !$user->currency_rate_notifications) {
    echo "⚠️  User doesn't have currency rate notifications enabled\n";
    echo "To enable: UPDATE users SET notifications_enabled = 1, currency_rate_notifications = 1 WHERE id = {$user->id};\n";
}

echo "\n=== Test Complete ===\n";
echo "The fix ensures that:\n";
echo "1. Old rate is captured BEFORE updating the currency\n";
echo "2. Proper type conversion for rate comparison\n";
echo "3. Correct message generation based on rate changes\n";
echo "4. Only users with enabled preferences receive notifications\n\n";

echo "When currency rates are updated in the admin panel, only users with currency_rate_notifications = 1 will receive notifications.\n";