<?php

require_once 'vendor/autoload.php';

use Illuminate\Support\Facades\DB;
use App\Models\User;
use App\Models\Currency;
use App\Jobs\SendCurrencyRateNotificationJob;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "Testing Currency Rate Notification Filtering\n";
echo "==========================================\n\n";

// Get all users and their notification settings
$users = User::all();

echo "Current users and their notification settings:\n";
echo "=============================================\n";
foreach ($users as $user) {
    echo "User ID: {$user->id}\n";
    echo "  Name: {$user->name}\n";
    echo "  Email: {$user->email}\n";
    echo "  Notifications Enabled: " . ($user->notifications_enabled ? 'YES' : 'NO') . "\n";
    echo "  Currency Rate Notifications: " . ($user->currency_rate_notifications ? 'YES' : 'NO') . "\n";
    echo "  Currency Rate Access: " . ($user->currency_rate_access ? 'YES' : 'NO') . "\n";
    echo "  Will Receive Currency Notifications: " . 
         ($user->notifications_enabled && $user->currency_rate_notifications && $user->currency_rate_access ? 'YES' : 'NO') . "\n";
    echo "  ---\n";
}

echo "\nTesting Notification Filtering Query:\n";
echo "=====================================\n";

// Test the same query used in SendCurrencyRateNotificationJob
$eligibleUsers = User::where('notifications_enabled', true)
                    ->where('currency_rate_notifications', true)
                    ->where('currency_rate_access', true)
                    ->get();

echo "Users who will receive currency rate notifications:\n";
if ($eligibleUsers->count() > 0) {
    foreach ($eligibleUsers as $user) {
        echo "- User ID: {$user->id}, Name: {$user->name}, Email: {$user->email}\n";
    }
} else {
    echo "No users are eligible to receive currency rate notifications.\n";
}

echo "\nNotification Filtering Logic:\n";
echo "============================\n";
echo "A user will receive currency rate notifications ONLY if ALL of these are true:\n";
echo "1. notifications_enabled = true\n";
echo "2. currency_rate_notifications = true\n";
echo "3. currency_rate_access = true\n\n";

echo "Test Scenarios:\n";
echo "===============\n";

// Test different scenarios
$testScenarios = [
    ['notifications_enabled' => true, 'currency_rate_notifications' => true, 'currency_rate_access' => true, 'expected' => 'YES'],
    ['notifications_enabled' => true, 'currency_rate_notifications' => true, 'currency_rate_access' => false, 'expected' => 'NO'],
    ['notifications_enabled' => true, 'currency_rate_notifications' => false, 'currency_rate_access' => true, 'expected' => 'NO'],
    ['notifications_enabled' => false, 'currency_rate_notifications' => true, 'currency_rate_access' => true, 'expected' => 'NO'],
    ['notifications_enabled' => false, 'currency_rate_notifications' => false, 'currency_rate_access' => false, 'expected' => 'NO'],
];

echo "Scenario Testing:\n";
foreach ($testScenarios as $index => $scenario) {
    $willReceive = $scenario['notifications_enabled'] && 
                   $scenario['currency_rate_notifications'] && 
                   $scenario['currency_rate_access'];
    
    $result = $willReceive ? 'YES' : 'NO';
    $status = $result === $scenario['expected'] ? '✓ PASS' : '✗ FAIL';
    
    echo "Scenario " . ($index + 1) . ": {$status}\n";
    echo "  Notifications Enabled: " . ($scenario['notifications_enabled'] ? 'YES' : 'NO') . "\n";
    echo "  Currency Rate Notifications: " . ($scenario['currency_rate_notifications'] ? 'YES' : 'NO') . "\n";
    echo "  Currency Rate Access: " . ($scenario['currency_rate_access'] ? 'YES' : 'NO') . "\n";
    echo "  Will Receive: {$result} (Expected: {$scenario['expected']})\n";
    echo "  ---\n";
}

echo "\nRecommendations for Testing:\n";
echo "============================\n";
if ($users->count() > 0) {
    $firstUser = $users->first();
    echo "1. Enable currency rate access for user {$firstUser->id}:\n";
    echo "   UPDATE users SET currency_rate_access = 1 WHERE id = {$firstUser->id};\n\n";
    
    echo "2. Test with a currency rate update (trigger the notification job)\n\n";
    
    echo "3. Check the logs to verify only eligible users receive notifications\n\n";
    
    echo "4. Disable currency rate access for user {$firstUser->id}:\n";
    echo "   UPDATE users SET currency_rate_access = 0 WHERE id = {$firstUser->id};\n\n";
    
    echo "5. Test again and verify no notifications are sent\n\n";
}

echo "Files Modified for Notification Filtering:\n";
echo "==========================================\n";
echo "1. app/Jobs/SendCurrencyRateNotificationJob.php\n";
echo "   - Added currency_rate_access filter to user query\n";
echo "   - Updated log messages to reflect new filtering\n\n";

echo "2. app/Services/OneSignalService.php\n";
echo "   - Updated sendCurrencyRateNotification method\n";
echo "   - Added currency_rate_access check\n\n";

echo "Test completed successfully!\n";
echo "The notification system now properly filters currency rate notifications based on user permissions.\n";

