<?php

require_once 'vendor/autoload.php';

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\User;
use App\Models\Account;
use App\Models\AccountTransaction;
use App\Http\Controllers\Api\CustomerController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

try {
    echo "Testing Account Statement API\n";
    echo "============================\n\n";
    
    // Get first user
    $user = User::first();
    if (!$user) {
        echo "No users found in database\n";
        exit(1);
    }
    
    echo "Testing with user: {$user->email}\n\n";
    
    // Get user's first account
    $account = $user->customer->accounts()->first();
    if (!$account) {
        echo "No accounts found for user\n";
        exit(1);
    }
    
    echo "Testing with account: {$account->id} ({$account->currency->currency_name})\n\n";
    
    // Create some test transactions
    echo "Creating test transactions...\n";
    
    // Create a deposit transaction
    $depositTransaction = AccountTransaction::create([
        'from_account_id' => null,
        'to_account_id' => $account->id,
        'user_id' => $user->id,
        'transaction_type' => 'deposit',
        'transaction_amount' => 1000.00,
        'transaction_date' => now()->subDays(5),
        'transaction_time' => now()->subDays(5)->format('H:i:s'),
        'description' => 'Test deposit transaction',
    ]);
    echo "✅ Created deposit transaction: {$depositTransaction->id}\n";
    
    // Create a withdrawal transaction
    $withdrawalTransaction = AccountTransaction::create([
        'from_account_id' => $account->id,
        'to_account_id' => null,
        'user_id' => $user->id,
        'transaction_type' => 'withdrawal',
        'transaction_amount' => 250.00,
        'transaction_date' => now()->subDays(3),
        'transaction_time' => now()->subDays(3)->format('H:i:s'),
        'description' => 'Test withdrawal transaction',
    ]);
    echo "✅ Created withdrawal transaction: {$withdrawalTransaction->id}\n";
    
    // Create another deposit transaction
    $depositTransaction2 = AccountTransaction::create([
        'from_account_id' => null,
        'to_account_id' => $account->id,
        'user_id' => $user->id,
        'transaction_type' => 'deposit',
        'transaction_amount' => 500.00,
        'transaction_date' => now()->subDays(1),
        'transaction_time' => now()->subDays(1)->format('H:i:s'),
        'description' => 'Test deposit transaction 2',
    ]);
    echo "✅ Created deposit transaction 2: {$depositTransaction2->id}\n\n";
    
    // Test the API endpoint
    echo "Testing API endpoint...\n";
    Auth::login($user);
    
    $controller = new CustomerController();
    $request = new Request([
        'account_id' => $account->id,
        'start_date' => now()->subDays(10)->format('Y-m-d'),
        'end_date' => now()->format('Y-m-d'),
    ]);
    
    $response = $controller->accountStatement($request);
    $responseData = json_decode($response->getContent(), true);
    
    echo "API Response Status: " . ($responseData['success'] ? 'SUCCESS' : 'FAILED') . "\n";
    
    if ($responseData['success'] && isset($responseData['data']['summary'])) {
        $summary = $responseData['data']['summary'];
        echo "Statement Summary:\n";
        echo "- Total Transactions: {$summary['total_transactions']}\n";
        echo "- Total Deposits: {$summary['total_deposits']}\n";
        echo "- Total Withdrawals: {$summary['total_withdrawals']}\n";
        echo "- Current Balance: {$summary['current_balance']}\n\n";
        
        echo "Transactions Found: " . count($responseData['data']['transactions']) . "\n";
        
        foreach ($responseData['data']['transactions'] as $transaction) {
            echo "- Transaction {$transaction['id']}: {$transaction['transaction_type']} - {$transaction['transaction_amount']} ({$transaction['transaction_date']})\n";
        }
    } else {
        echo "❌ API call failed or returned unexpected data\n";
        echo "Response: " . json_encode($responseData, JSON_PRETTY_PRINT) . "\n";
    }
    
    echo "\n✅ Account statement API test completed!\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}
