<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\TransactionController;
use App\Http\Controllers\Api\AccountController;
use App\Http\Controllers\Api\CustomerController;
use App\Http\Controllers\Api\CurrencyController;
use App\Http\Controllers\Api\OneSignalController;
use App\Http\Controllers\Api\NotificationPreferencesController;
use App\Http\Controllers\Api\NotificationController;
use App\Http\Controllers\Api\AppVersionController;
use App\Http\Controllers\Api\StatisticsController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// Public routes
Route::prefix('mobile')->group(function () {
    Route::post('login', [AuthController::class, 'login']);
    Route::post('register', [AuthController::class, 'register']);
    
    // Public currency rates (no auth required)
    Route::get('currency-rates', [CurrencyController::class, 'rates']);
    
    // App version check (no auth required)
    Route::get('app-version', [AppVersionController::class, 'getLatestVersion']);
    
    // Test endpoint
    Route::get('test', function () {
        return response()->json([
            'success' => true,
            'message' => 'Test endpoint working',
            'timestamp' => now()
        ]);
    });
    
    // Test Kurdish text encoding endpoint
    Route::post('test-kurdish', function(Request $request) {
        $kurdishText = $request->input('text', 'ڕاکێشان');
        
        // Test database storage
        $transaction = new \App\Models\AccountTransaction();
        $transaction->note = $kurdishText;
        $transaction->person_name = $kurdishText;
        $transaction->transaction_type = 'test';
        $transaction->transaction_amount = 100;
        $transaction->transaction_date = now();
        $transaction->transaction_time = now()->format('H:i:s');
        $transaction->from_account_id = 1;
        $transaction->to_account_id = 1;
        $transaction->user_id = 1;
        $transaction->status = 'approved';
        $transaction->save();
        
        // Test retrieval
        $retrieved = \App\Models\AccountTransaction::latest()->first();
        
        return response()->json([
            'original' => $kurdishText,
            'stored_note' => $retrieved->note,
            'stored_person_name' => $retrieved->person_name,
            'note_matches' => $kurdishText === $retrieved->note,
            'person_name_matches' => $kurdishText === $retrieved->person_name,
            'note_encoding' => mb_detect_encoding($retrieved->note),
            'person_name_encoding' => mb_detect_encoding($retrieved->person_name),
        ], 200, [], JSON_UNESCAPED_UNICODE);
    });
    
    // Protected routes
    Route::middleware('auth:sanctum')->group(function () {
        // Authentication routes
        Route::get('user', [AuthController::class, 'user']);
        Route::post('logout', [AuthController::class, 'logout']);
        Route::post('refresh', [AuthController::class, 'refresh']);
        Route::post('update-password', [AuthController::class, 'updatePassword']);
        Route::post('update-language', [AuthController::class, 'updateLanguage']);
        Route::post('update-profile-picture', [AuthController::class, 'updateProfilePicture']);
        Route::delete('delete-profile-picture', [AuthController::class, 'deleteProfilePicture']);
        
        // Transaction routes
            Route::get('transactions/export', [TransactionController::class, 'export']);
            Route::get('transactions/{id}/details', [TransactionController::class, 'show']);
            Route::apiResource('transactions', TransactionController::class);
        
        // Account routes
        Route::get('accounts', [AccountController::class, 'index']);
        Route::get('accounts/{id}', [AccountController::class, 'show']);
        Route::get('accounts/{id}/balance', [AccountController::class, 'balance']);
        Route::get('currencies', [AccountController::class, 'currencies']);
        
        // Customer routes
        Route::get('profile', [CustomerController::class, 'profile']);
        Route::put('profile', [CustomerController::class, 'update']);
        Route::get('customer/account-statement', [CustomerController::class, 'accountStatement']);
        
        // OneSignal routes
        Route::post('onesignal/external-id', [OneSignalController::class, 'updateExternalId']);
        Route::post('onesignal/subscription-id', [OneSignalController::class, 'updateSubscriptionId']);
        Route::post('onesignal/player-id', [OneSignalController::class, 'updatePlayerId']);
        Route::get('onesignal/player-id', [OneSignalController::class, 'getPlayerId']);
        Route::post('onesignal/test', [OneSignalController::class, 'testNotification']);
        
        // Notification preferences routes
        Route::get('notification-preferences', [NotificationPreferencesController::class, 'getPreferences']);
        Route::put('notification-preferences', [NotificationPreferencesController::class, 'updatePreferences']);
        Route::post('notification-preferences/toggle-all', [NotificationPreferencesController::class, 'toggleAll']);
        
        // Notification routes
        Route::get('notifications', [NotificationController::class, 'index']);
        Route::post('notifications', [NotificationController::class, 'store']);
        Route::get('notifications/{id}', [NotificationController::class, 'show']);
        Route::put('notifications/{id}/read', [NotificationController::class, 'markAsRead']);
        Route::put('notifications/read-all', [NotificationController::class, 'markAllAsRead']);
        Route::delete('notifications/clear-all', [NotificationController::class, 'clearAll']);
        Route::delete('notifications/{id}', [NotificationController::class, 'destroy']);
        Route::get('notifications/unread-count', [NotificationController::class, 'unreadCount']);
        
        // Statistics routes
        Route::get('statistics/transactions', [StatisticsController::class, 'transactionStats']);
        Route::get('statistics/currency-rates/{currency}', [StatisticsController::class, 'currencyRateHistory']);
    });
});

// Legacy route for backward compatibility
Route::middleware('auth:sanctum')->get('/user', function (Request $request) {
    return $request->user();
});
