<?php

// Load Laravel environment
require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "=== Mass Restore User Roles ===\n\n";

try {
    // Check if Spatie Permission is being used
    if (!class_exists('\Spatie\Permission\Models\Role')) {
        echo "Error: Spatie Permission package not detected.\n";
        exit(1);
    }
    
    // Make sure roles exist
    $requiredRoles = ['super-admin', 'admin', 'supervisor', 'cashier', 'customer'];
    foreach ($requiredRoles as $roleName) {
        if (!\Spatie\Permission\Models\Role::where('name', $roleName)->exists()) {
            \Spatie\Permission\Models\Role::create(['name' => $roleName, 'guard_name' => 'web']);
            echo "Created missing role: {$roleName}\n";
        }
    }
    
    // Special users - manually assign roles to specific emails
    $specialUsers = [
        'admin@admin.com' => 'super-admin',
        'admin@Berat.com' => 'super-admin'
        // Add more special cases if needed
    ];
    
    // Define patterns for role assignment
    $patterns = [
        'supervisor' => ['supervisor', 'manager'],
        'cashier' => ['cashier', 'teller', 'sarraf'],
        'admin' => ['admin'],
        // Add more patterns if needed
    ];
    
    // Get all users
    $users = \App\Models\User::all();
    echo "Found " . $users->count() . " users.\n\n";
    
    // Counters
    $assigned = 0;
    $specificAssigned = 0;
    $patternAssigned = 0;
    $relationAssigned = 0;
    $defaultAssigned = 0;
    
    // Process each user
    foreach ($users as $user) {
        $currentRoles = $user->roles()->pluck('name')->toArray();
        $email = strtolower($user->email);
        $name = strtolower($user->name);
        $roleAssigned = false;
        
        echo "Processing user: ID: {$user->id}, Name: {$user->name}, Email: {$user->email}\n";
        
        // 1. Check specific user assignments
        foreach ($specialUsers as $userEmail => $roleName) {
            if ($email == strtolower($userEmail)) {
                echo "  Assigning specific role ({$roleName}) to {$user->email}\n";
                $user->syncRoles([$roleName]);
                $roleAssigned = true;
                $specificAssigned++;
                break;
            }
        }
        
        if ($roleAssigned) continue;
        
        // 2. Check for patterns in email or name
        foreach ($patterns as $roleName => $keywords) {
            foreach ($keywords as $keyword) {
                if (
                    strpos($email, $keyword) !== false || 
                    strpos($name, $keyword) !== false
                ) {
                    echo "  Assigning role ({$roleName}) based on pattern match in name/email\n";
                    $user->syncRoles([$roleName]);
                    $roleAssigned = true;
                    $patternAssigned++;
                    break 2;
                }
            }
        }
        
        if ($roleAssigned) continue;
        
        // 3. Check related records
        try {
            // Check for cashier record
            if (\Illuminate\Support\Facades\DB::table('cashiers')->where('user_id', $user->id)->exists()) {
                echo "  Assigning cashier role based on cashiers table\n";
                $user->syncRoles(['cashier']);
                $roleAssigned = true;
                $relationAssigned++;
            }
        } catch (\Exception $e) {
            // Table might not exist, just continue
        }
        
        if ($roleAssigned) continue;
        
        // 4. Default to customer for everyone else
        echo "  Assigning default customer role\n";
        $user->syncRoles(['customer']);
        $defaultAssigned++;
        $assigned++;
    }
    
    // Final report
    echo "\n=== Role Assignment Complete ===\n";
    echo "Total users processed: " . $users->count() . "\n";
    echo "Specifically assigned roles: {$specificAssigned}\n";
    echo "Pattern-based assignments: {$patternAssigned}\n";
    echo "Relation-based assignments: {$relationAssigned}\n";  
    echo "Default customer assignments: {$defaultAssigned}\n";
    
    // Check if any users still have no roles
    $usersWithoutRoles = \App\Models\User::whereDoesntHave('roles')->count();
    if ($usersWithoutRoles > 0) {
        echo "\nWARNING: {$usersWithoutRoles} users still have no roles!\n";
    } else {
        echo "\nSuccess: All users now have at least one role assigned.\n";
    }
    
    echo "\nDone!\n";
    
} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}