<?php

/**
 * Quick Fix for Transaction 129518
 * 
 * This is a standalone script that directly updates the corrupted transaction.
 * Run with: php quick_fix_transaction_129518.php
 */

require __DIR__.'/vendor/autoload.php';

// Bootstrap Laravel
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;

echo "\n";
echo "==========================================\n";
echo " Quick Fix for Transaction 129518\n";
echo "==========================================\n\n";

try {
    // Set UTF-8 encoding
    DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');
    echo "✓ Database connection established\n";
    
    // Get current data
    echo "\nCurrent data:\n";
    $current = DB::table('account_transactions')
        ->where('id', 129518)
        ->first();
    
    if (!$current) {
        echo "✗ Transaction 129518 not found!\n";
        exit(1);
    }
    
    echo "  ID: {$current->id}\n";
    echo "  Note: {$current->note}\n";
    echo "  Person Name: " . ($current->person_name ?? 'NULL') . "\n";
    
    // Update with correct Kurdish text
    $correctText = 'ئاسۆ کوی';
    
    echo "\nUpdating transaction...\n";
    $updated = DB::table('account_transactions')
        ->where('id', 129518)
        ->update([
            'note' => $correctText,
            'updated_at' => now()
        ]);
    
    if ($updated) {
        echo "✓ Transaction updated successfully!\n";
        
        // Verify the update
        echo "\nVerifying update:\n";
        $verified = DB::table('account_transactions')
            ->where('id', 129518)
            ->first();
        
        echo "  ID: {$verified->id}\n";
        echo "  Note: {$verified->note}\n";
        echo "  Expected: {$correctText}\n";
        echo "  Match: " . ($verified->note === $correctText ? '✓ YES' : '✗ NO') . "\n";
        
        if ($verified->note === $correctText) {
            echo "\n";
            echo "==========================================\n";
            echo " ✓ SUCCESS!\n";
            echo "==========================================\n";
            echo "\nThe transaction has been fixed.\n";
            echo "Test the API to confirm the fix:\n";
            echo "GET /api/mobile/transactions\n\n";
        } else {
            echo "\n✗ WARNING: Text doesn't match exactly.\n";
            echo "  This might be a database encoding issue.\n";
        }
    } else {
        echo "✗ Update failed!\n";
    }
    
    // Show other corrupted transactions
    echo "\n==========================================\n";
    echo " Other Corrupted Transactions\n";
    echo "==========================================\n\n";
    
    $corrupted = DB::table('account_transactions')
        ->where(function($query) {
            $query->where('note', 'like', '%?%')
                  ->orWhere('person_name', 'like', '%?%');
        })
        ->where('id', '!=', 129518)
        ->select('id', 'transaction_type', 'note', 'person_name')
        ->orderBy('id', 'desc')
        ->limit(10)
        ->get();
    
    if ($corrupted->count() > 0) {
        echo "Found {$corrupted->count()} other corrupted transactions:\n\n";
        echo str_pad("ID", 10) . str_pad("Type", 15) . str_pad("Note", 30) . "Person Name\n";
        echo str_repeat("-", 80) . "\n";
        
        foreach ($corrupted as $trans) {
            echo str_pad($trans->id, 10);
            echo str_pad($trans->transaction_type, 15);
            echo str_pad(substr($trans->note ?? '', 0, 28), 30);
            echo substr($trans->person_name ?? '', 0, 20) . "\n";
        }
        
        echo "\nTo fix these, you need to know the original Kurdish text.\n";
        echo "Then use this pattern in Laravel Tinker:\n\n";
        echo "DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');\n";
        echo "DB::table('account_transactions')->where('id', TRANSACTION_ID)->update(['note' => 'ORIGINAL_TEXT']);\n";
    } else {
        echo "No other corrupted transactions found.\n";
    }
    
    echo "\n";
    
} catch (\Exception $e) {
    echo "\n✗ ERROR: " . $e->getMessage() . "\n";
    echo "\nStack trace:\n";
    echo $e->getTraceAsString() . "\n";
    exit(1);
}



