<?php

/**
 * Quick Test for Kurdish Text Fix
 * Access via: http://your-domain.com/test-kurdish-fix.php
 */

require __DIR__.'/../vendor/autoload.php';

$app = require_once __DIR__.'/../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

header('Content-Type: text/html; charset=UTF-8');

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Kurdish Text Fix Test</title>
    <style>
        body { 
            font-family: Arial, sans-serif; 
            margin: 40px; 
            background: #f5f5f5;
        }
        .container { 
            max-width: 800px; 
            margin: 0 auto; 
            background: white; 
            padding: 30px; 
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 { color: #333; border-bottom: 3px solid #4CAF50; padding-bottom: 10px; }
        .success { 
            background: #d4edda; 
            color: #155724; 
            padding: 15px; 
            border-radius: 5px;
            border-left: 4px solid #28a745;
            margin: 15px 0;
        }
        .error { 
            background: #f8d7da; 
            color: #721c24; 
            padding: 15px; 
            border-radius: 5px;
            border-left: 4px solid #dc3545;
            margin: 15px 0;
        }
        .info { 
            background: #d1ecf1; 
            color: #0c5460; 
            padding: 15px; 
            border-radius: 5px;
            border-left: 4px solid #17a2b8;
            margin: 15px 0;
        }
        pre { 
            background: #f8f9fa; 
            padding: 15px; 
            border-radius: 5px;
            overflow-x: auto;
            border: 1px solid #dee2e6;
        }
        .kurdish-text { 
            font-size: 24px; 
            font-weight: bold; 
            color: #007bff;
            padding: 10px;
            background: #e7f3ff;
            border-radius: 5px;
            display: inline-block;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #4CAF50;
            color: white;
        }
        tr:hover {
            background-color: #f5f5f5;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Kurdish Text Fix Test</h1>
        
        <?php
        try {
            use App\Models\AccountTransaction;
            
            echo '<div class="info"><strong>Testing Transaction 129518...</strong></div>';
            
            $transaction = AccountTransaction::find(129518);
            
            if ($transaction) {
                $expectedText = 'ئاسۆ کوی';
                $actualText = $transaction->note;
                $isFixed = ($actualText === $expectedText);
                
                echo '<table>';
                echo '<tr><th>Property</th><th>Value</th></tr>';
                echo '<tr><td><strong>Transaction ID</strong></td><td>' . $transaction->id . '</td></tr>';
                echo '<tr><td><strong>Transaction Type</strong></td><td>' . $transaction->transaction_type . '</td></tr>';
                echo '<tr><td><strong>Amount</strong></td><td>' . $transaction->transaction_amount . '</td></tr>';
                echo '<tr><td><strong>Status</strong></td><td>' . $transaction->status . '</td></tr>';
                echo '</table>';
                
                echo '<h2>📝 Note Field Test</h2>';
                echo '<table>';
                echo '<tr><th>Aspect</th><th>Result</th></tr>';
                echo '<tr><td><strong>Actual Note Value</strong></td><td><span class="kurdish-text">' . htmlspecialchars($actualText) . '</span></td></tr>';
                echo '<tr><td><strong>Expected Value</strong></td><td><span class="kurdish-text">' . htmlspecialchars($expectedText) . '</span></td></tr>';
                echo '<tr><td><strong>Match</strong></td><td>' . ($isFixed ? '<strong style="color: green;">✓ YES</strong>' : '<strong style="color: red;">✗ NO</strong>') . '</td></tr>';
                echo '<tr><td><strong>Character Count</strong></td><td>' . mb_strlen($actualText) . ' characters</td></tr>';
                echo '<tr><td><strong>Byte Count</strong></td><td>' . strlen($actualText) . ' bytes</td></tr>';
                echo '</table>';
                
                if ($isFixed) {
                    echo '<div class="success">';
                    echo '<h3>✓ SUCCESS!</h3>';
                    echo '<p><strong>The Kurdish text fix is working correctly!</strong></p>';
                    echo '<p>The note field now shows: <span class="kurdish-text">' . htmlspecialchars($actualText) . '</span></p>';
                    echo '<p>Your API should now return correct Kurdish text.</p>';
                    echo '</div>';
                    
                    echo '<h3>📱 Test the API</h3>';
                    echo '<div class="info">';
                    echo '<p>Use Postman or curl to test:</p>';
                    echo '<pre>GET https://LA DERMA.innovation-pulsehub.com/api/mobile/transactions</pre>';
                    echo '<p>You should see:</p>';
                    echo '<pre>{
    "id": 129518,
    "note": "ئاسۆ کوی"
}</pre>';
                    echo '</div>';
                } else {
                    echo '<div class="error">';
                    echo '<h3>✗ Not Working Yet</h3>';
                    echo '<p><strong>Current value:</strong> <code>' . htmlspecialchars($actualText) . '</code></p>';
                    echo '<p><strong>Expected value:</strong> <code>' . htmlspecialchars($expectedText) . '</code></p>';
                    echo '<p><strong>Debug Info:</strong></p>';
                    echo '<ul>';
                    echo '<li>Value hex: ' . bin2hex($actualText) . '</li>';
                    echo '<li>Encoding: ' . mb_detect_encoding($actualText, ['UTF-8', 'ASCII', 'Windows-1256'], true) . '</li>';
                    echo '</ul>';
                    echo '<p><strong>Try:</strong></p>';
                    echo '<ol>';
                    echo '<li>Clear cache: <code>php artisan cache:clear</code></li>';
                    echo '<li>Restart web server</li>';
                    echo '<li>Check model accessor in <code>app/Models/AccountTransaction.php</code></li>';
                    echo '</ol>';
                    echo '</div>';
                }
                
                // Test JSON encoding
                echo '<h3>🔍 JSON Encoding Test</h3>';
                $jsonData = [
                    'id' => $transaction->id,
                    'note' => $transaction->note,
                    'person_name' => $transaction->person_name,
                ];
                $jsonOutput = json_encode($jsonData, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
                echo '<pre>' . htmlspecialchars($jsonOutput) . '</pre>';
                
            } else {
                echo '<div class="error">';
                echo '<h3>✗ Transaction Not Found</h3>';
                echo '<p>Transaction 129518 does not exist in the database.</p>';
                echo '</div>';
            }
            
        } catch (\Exception $e) {
            echo '<div class="error">';
            echo '<h3>✗ Error</h3>';
            echo '<p><strong>Message:</strong> ' . htmlspecialchars($e->getMessage()) . '</p>';
            echo '<p><strong>File:</strong> ' . $e->getFile() . ':' . $e->getLine() . '</p>';
            echo '<details><summary>Stack Trace</summary><pre>' . htmlspecialchars($e->getTraceAsString()) . '</pre></details>';
            echo '</div>';
        }
        ?>
        
        <hr style="margin: 30px 0;">
        <p style="text-align: center; color: #666;">
            <small>Generated at: <?php echo date('Y-m-d H:i:s'); ?></small>
        </p>
    </div>
</body>
</html>



