<?php

// Load Laravel environment
require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(\Illuminate\Contracts\Console\Kernel::class)->bootstrap();

echo "=== User Role Management Tool ===\n\n";

try {
    // Check if Spatie Permission is being used
    if (!class_exists('\Spatie\Permission\Models\Role')) {
        echo "Error: Spatie Permission package not detected.\n";
        exit(1);
    }
    
    // Get all available roles
    $roles = \Spatie\Permission\Models\Role::all();
    
    if ($roles->isEmpty()) {
        echo "Error: No roles found in database.\n";
        exit(1);
    }
    
    echo "Available roles:\n";
    foreach ($roles as $index => $role) {
        echo ($index + 1) . ". " . $role->name . "\n";
    }
    echo "\n";
    
    // Display menu
    while (true) {
        echo "=== Menu ===\n";
        echo "1. List all users and their current roles\n";
        echo "2. Assign role to specific user by ID\n";
        echo "3. Assign role to specific user by email\n";
        echo "4. List users with no roles\n";
        echo "5. Assign super-admin role to admin@admin.com\n";
        echo "6. Exit\n";
        echo "Choice: ";
        
        $choice = trim(fgets(STDIN));
        
        if ($choice == "1") {
            // List all users
            $users = \App\Models\User::all();
            echo "\n=== Users and their roles ===\n";
            
            foreach ($users as $user) {
                $userRoles = $user->roles()->pluck('name')->toArray();
                echo "ID: {$user->id}, Name: {$user->name}, Email: {$user->email}\n";
                echo "   Roles: " . (empty($userRoles) ? "None" : implode(", ", $userRoles)) . "\n";
            }
            echo "\n";
            
        } elseif ($choice == "2") {
            // Assign role by user ID
            echo "Enter user ID: ";
            $userId = trim(fgets(STDIN));
            
            $user = \App\Models\User::find($userId);
            if (!$user) {
                echo "User with ID {$userId} not found.\n\n";
                continue;
            }
            
            echo "User found: {$user->name} ({$user->email})\n";
            echo "Current roles: " . implode(", ", $user->roles()->pluck('name')->toArray()) . "\n";
            echo "Choose role to assign:\n";
            
            foreach ($roles as $index => $role) {
                echo ($index + 1) . ". " . $role->name . "\n";
            }
            
            echo "Role number: ";
            $roleIndex = trim(fgets(STDIN)) - 1;
            
            if (!isset($roles[$roleIndex])) {
                echo "Invalid role selection.\n\n";
                continue;
            }
            
            echo "Replace existing roles? (y/n): ";
            $replace = strtolower(trim(fgets(STDIN))) == 'y';
            
            if ($replace) {
                $user->syncRoles([$roles[$roleIndex]->name]);
            } else {
                $user->assignRole($roles[$roleIndex]->name);
            }
            
            echo "Role '{$roles[$roleIndex]->name}' " . ($replace ? "set as only role" : "added") . 
                 " for user {$user->name}.\n\n";
            
        } elseif ($choice == "3") {
            // Assign role by email
            echo "Enter user email: ";
            $email = trim(fgets(STDIN));
            
            $user = \App\Models\User::where('email', $email)->first();
            if (!$user) {
                echo "User with email {$email} not found.\n\n";
                continue;
            }
            
            echo "User found: {$user->name} (ID: {$user->id})\n";
            echo "Current roles: " . implode(", ", $user->roles()->pluck('name')->toArray()) . "\n";
            echo "Choose role to assign:\n";
            
            foreach ($roles as $index => $role) {
                echo ($index + 1) . ". " . $role->name . "\n";
            }
            
            echo "Role number: ";
            $roleIndex = trim(fgets(STDIN)) - 1;
            
            if (!isset($roles[$roleIndex])) {
                echo "Invalid role selection.\n\n";
                continue;
            }
            
            echo "Replace existing roles? (y/n): ";
            $replace = strtolower(trim(fgets(STDIN))) == 'y';
            
            if ($replace) {
                $user->syncRoles([$roles[$roleIndex]->name]);
            } else {
                $user->assignRole($roles[$roleIndex]->name);
            }
            
            echo "Role '{$roles[$roleIndex]->name}' " . ($replace ? "set as only role" : "added") . 
                 " for user {$user->name}.\n\n";
                 
        } elseif ($choice == "4") {
            // List users with no roles
            $usersWithoutRoles = \App\Models\User::whereDoesntHave('roles')->get();
            
            if ($usersWithoutRoles->isEmpty()) {
                echo "All users have at least one role assigned.\n\n";
                continue;
            }
            
            echo "\n=== Users with no roles ===\n";
            foreach ($usersWithoutRoles as $user) {
                echo "ID: {$user->id}, Name: {$user->name}, Email: {$user->email}\n";
            }
            echo "\n";
            
        } elseif ($choice == "5") {
            // Quick fix for admin@admin.com
            $adminUser = \App\Models\User::where('email', 'admin@admin.com')->first();
            
            if (!$adminUser) {
                echo "User with email admin@admin.com not found.\n\n";
                continue;
            }
            
            $superAdminRole = \Spatie\Permission\Models\Role::where('name', 'super-admin')->first();
            if (!$superAdminRole) {
                echo "super-admin role not found.\n\n";
                continue;
            }
            
            $adminUser->syncRoles(['super-admin']);
            
            // Sync all permissions
            $permissions = \Spatie\Permission\Models\Permission::all();
            if ($permissions->isNotEmpty()) {
                $superAdminRole->syncPermissions($permissions);
            }
            
            echo "Assigned super-admin role to {$adminUser->name} (admin@admin.com) and synced all permissions.\n\n";
            
        } elseif ($choice == "6") {
            echo "Exiting...\n";
            break;
        } else {
            echo "Invalid choice. Please try again.\n\n";
        }
    }
    
} catch (\Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
}