<?php

/**
 * Fix Specific Corrupted Kurdish Text in Database
 * 
 * Run this script with: php fix_specific_kurdish_text.php
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use App\Models\AccountTransaction;
use Illuminate\Support\Facades\DB;

echo "========================================\n";
echo "Kurdish Text Fix Script\n";
echo "========================================\n\n";

// Set the database connection to use UTF-8
DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');

// First, fix transaction ID 129518 with the known text
echo "Fixing transaction ID 129518...\n";
$transaction = AccountTransaction::find(129518);

if ($transaction) {
    echo "Current note: " . $transaction->note . "\n";
    echo "Current person_name: " . ($transaction->person_name ?? 'NULL') . "\n";
    
    // Update with the correct Kurdish text
    $correctText = 'ئاسۆ کوی';
    $transaction->note = $correctText;
    
    // Disable model events temporarily to avoid double encoding
    AccountTransaction::flushEventListeners();
    AccountTransaction::boot();
    
    $transaction->save();
    
    echo "Updated note to: {$correctText}\n";
    echo "✓ Transaction 129518 fixed!\n\n";
} else {
    echo "✗ Transaction 129518 not found!\n\n";
}

// Find all transactions with corrupted text (containing ? characters)
echo "Searching for other corrupted transactions...\n";
$corruptedTransactions = AccountTransaction::where(function($query) {
    $query->where('note', 'like', '%?%')
          ->orWhere('person_name', 'like', '%?%');
})->where('id', '!=', 129518) // Exclude the one we just fixed
  ->get();

echo "Found " . $corruptedTransactions->count() . " other transactions with corrupted text.\n\n";

if ($corruptedTransactions->count() > 0) {
    echo "Corrupted transactions:\n";
    echo "ID\tType\tNote\tPerson Name\n";
    echo "----------------------------------------\n";
    
    foreach ($corruptedTransactions as $trans) {
        echo $trans->id . "\t";
        echo $trans->transaction_type . "\t";
        echo ($trans->note ?? 'NULL') . "\t";
        echo ($trans->person_name ?? 'NULL') . "\n";
    }
    
    echo "\n";
    echo "Note: These transactions need manual correction.\n";
    echo "If you know the original text, update them using:\n";
    echo "UPDATE account_transactions SET note = 'original_text' WHERE id = transaction_id;\n";
}

// Verify the fix
echo "\n========================================\n";
echo "Verification\n";
echo "========================================\n";

$verified = AccountTransaction::find(129518);
if ($verified) {
    echo "Transaction 129518:\n";
    echo "Note: " . $verified->note . "\n";
    echo "Expected: ئاسۆ کوی\n";
    echo "Match: " . ($verified->note === 'ئاسۆ کوی' ? '✓ YES' : '✗ NO') . "\n";
}

echo "\n✓ Script completed!\n";



