<?php

/**
 * Fix Kurdish Text Encoding Script
 * 
 * This script fixes existing corrupted Kurdish text in the database
 * Run this with: php artisan tinker
 * Then copy and paste the code below
 */

use App\Models\AccountTransaction;
use Illuminate\Support\Facades\DB;

echo "Starting Kurdish text encoding fix...\n";

// Get all transactions with corrupted Kurdish text (containing ? characters)
$corruptedTransactions = AccountTransaction::where(function($query) {
    $query->where('note', 'like', '%?%')
          ->orWhere('person_name', 'like', '%?%');
})->get();

echo "Found " . $corruptedTransactions->count() . " transactions with potentially corrupted text.\n";

$fixedCount = 0;
$skippedCount = 0;

foreach ($corruptedTransactions as $transaction) {
    $updated = false;
    
    // Try to fix note field
    if ($transaction->note && strpos($transaction->note, '?') !== false) {
        // Try different encoding conversions
        $encodings = ['Windows-1256', 'ISO-8859-6', 'UTF-8'];
        
        foreach ($encodings as $encoding) {
            $converted = mb_convert_encoding($transaction->note, 'UTF-8', $encoding);
            if ($converted !== $transaction->note && strpos($converted, '?') === false) {
                $transaction->note = $converted;
                $updated = true;
                echo "Fixed note for transaction ID {$transaction->id}: {$converted}\n";
                break;
            }
        }
    }
    
    // Try to fix person_name field
    if ($transaction->person_name && strpos($transaction->person_name, '?') !== false) {
        // Try different encoding conversions
        $encodings = ['Windows-1256', 'ISO-8859-6', 'UTF-8'];
        
        foreach ($encodings as $encoding) {
            $converted = mb_convert_encoding($transaction->person_name, 'UTF-8', $encoding);
            if ($converted !== $transaction->person_name && strpos($converted, '?') === false) {
                $transaction->person_name = $converted;
                $updated = true;
                echo "Fixed person_name for transaction ID {$transaction->id}: {$converted}\n";
                break;
            }
        }
    }
    
    if ($updated) {
        $transaction->save();
        $fixedCount++;
    } else {
        $skippedCount++;
    }
}

echo "\nFix completed!\n";
echo "Fixed: {$fixedCount} transactions\n";
echo "Skipped: {$skippedCount} transactions\n";

// Test with a sample Kurdish text
echo "\nTesting with sample Kurdish text...\n";
$testText = 'ڕاکێشان';
$testTransaction = new AccountTransaction();
$testTransaction->note = $testText;
$testTransaction->person_name = $testText;
$testTransaction->transaction_type = 'test';
$testTransaction->transaction_amount = 100;
$testTransaction->transaction_date = now();
$testTransaction->transaction_time = now()->format('H:i:s');
$testTransaction->from_account_id = 1;
$testTransaction->to_account_id = 1;
$testTransaction->user_id = 1;
$testTransaction->status = 'approved';
$testTransaction->save();

$retrieved = AccountTransaction::find($testTransaction->id);
echo "Original: {$testText}\n";
echo "Stored note: {$retrieved->note}\n";
echo "Stored person_name: {$retrieved->person_name}\n";
echo "Note matches: " . ($testText === $retrieved->note ? 'YES' : 'NO') . "\n";
echo "Person name matches: " . ($testText === $retrieved->person_name ? 'YES' : 'NO') . "\n";

// Clean up test transaction
$testTransaction->delete();
echo "Test transaction cleaned up.\n";


