<?php

/**
 * Fix Database Encoding for Kurdish Text
 * 
 * This script:
 * 1. Checks database charset
 * 2. Fixes transaction 129518 with known text
 * 3. Lists other corrupted transactions
 * 
 * Run with: php fix_database_encoding.php
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

use Illuminate\Support\Facades\DB;
use App\Models\AccountTransaction;

echo "\n";
echo "==========================================\n";
echo " Database Encoding Fix Script\n";
echo "==========================================\n\n";

// Step 1: Set UTF-8 encoding
echo "Step 1: Setting UTF-8 encoding...\n";
try {
    DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');
    echo "✓ UTF-8 encoding set\n\n";
} catch (\Exception $e) {
    echo "✗ Error: {$e->getMessage()}\n\n";
    exit(1);
}

// Step 2: Check table charset
echo "Step 2: Checking table charset...\n";
try {
    $tableStatus = DB::select("SHOW TABLE STATUS LIKE 'account_transactions'");
    if (!empty($tableStatus)) {
        $collation = $tableStatus[0]->Collation;
        echo "Current collation: {$collation}\n";
        
        if (strpos($collation, 'utf8mb4') !== false) {
            echo "✓ Table is using utf8mb4\n\n";
        } else {
            echo "⚠ Table is NOT using utf8mb4\n";
            echo "  Run: php artisan migrate\n\n";
        }
    }
} catch (\Exception $e) {
    echo "⚠ Could not check table status\n\n";
}

// Step 3: Fix transaction 129518
echo "Step 3: Fixing transaction 129518...\n";
try {
    $transaction = AccountTransaction::find(129518);
    
    if ($transaction) {
        echo "Current note: '{$transaction->note}'\n";
        
        $correctText = 'ئاسۆ کوی';
        
        // Update using query builder to bypass any model events
        DB::table('account_transactions')
            ->where('id', 129518)
            ->update([
                'note' => $correctText,
                'updated_at' => now()
            ]);
        
        // Verify
        $updated = DB::table('account_transactions')->where('id', 129518)->first();
        echo "Updated note: '{$updated->note}'\n";
        
        if ($updated->note === $correctText) {
            echo "✓ Transaction 129518 fixed successfully!\n\n";
        } else {
            echo "⚠ Update may not have worked correctly\n";
            echo "  Expected: '{$correctText}'\n";
            echo "  Got: '{$updated->note}'\n\n";
        }
    } else {
        echo "⚠ Transaction 129518 not found\n\n";
    }
} catch (\Exception $e) {
    echo "✗ Error: {$e->getMessage()}\n\n";
}

// Step 4: Find other corrupted transactions
echo "Step 4: Finding other corrupted transactions...\n";
try {
    $corrupted = DB::table('account_transactions')
        ->where(function($query) {
            $query->where('note', 'like', '%?%')
                  ->orWhere('person_name', 'like', '%?%');
        })
        ->where('id', '!=', 129518)
        ->select('id', 'transaction_type', 'note', 'person_name', 'created_at')
        ->orderBy('id', 'desc')
        ->limit(20)
        ->get();
    
    if ($corrupted->count() > 0) {
        echo "Found {$corrupted->count()} other corrupted transactions:\n\n";
        echo str_pad("ID", 10) . str_pad("Type", 15) . str_pad("Note", 30) . "Person Name\n";
        echo str_repeat("-", 80) . "\n";
        
        foreach ($corrupted as $trans) {
            echo str_pad($trans->id, 10);
            echo str_pad($trans->transaction_type, 15);
            echo str_pad(substr($trans->note ?? '', 0, 28), 30);
            echo substr($trans->person_name ?? '', 0, 20) . "\n";
        }
        
        echo "\n⚠ These transactions need manual fixing.\n";
        echo "  You need to know the original Kurdish text for each.\n\n";
        
        echo "To fix them, use:\n";
        echo "php artisan tinker\n";
        echo "Then:\n";
        echo "DB::statement('SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci');\n";
        echo "DB::table('account_transactions')->where('id', TRANSACTION_ID)->update(['note' => 'ORIGINAL_TEXT']);\n\n";
    } else {
        echo "✓ No other corrupted transactions found!\n\n";
    }
} catch (\Exception $e) {
    echo "✗ Error: {$e->getMessage()}\n\n";
}

// Step 5: Test new transaction
echo "Step 5: Testing new transaction with Kurdish text...\n";
try {
    $testText = 'تێبینی تاقیکردنەوە';  // Kurdish test text
    
    $testTransaction = new AccountTransaction();
    $testTransaction->transaction_type = 'deposit';
    $testTransaction->transaction_amount = 1;
    $testTransaction->transaction_date = now();
    $testTransaction->transaction_time = now()->format('H:i:s');
    $testTransaction->from_account_id = 1;
    $testTransaction->to_account_id = 1;
    $testTransaction->user_id = 1;
    $testTransaction->status = 'approved';
    $testTransaction->note = $testText;
    $testTransaction->save();
    
    // Retrieve and verify
    $saved = AccountTransaction::find($testTransaction->id);
    
    echo "Test text: '{$testText}'\n";
    echo "Saved as: '{$saved->note}'\n";
    
    if ($saved->note === $testText) {
        echo "✓ New transactions will save Kurdish text correctly!\n";
    } else {
        echo "⚠ New transactions may still have encoding issues\n";
        echo "  Make sure you ran: php artisan migrate\n";
    }
    
    // Clean up test transaction
    $testTransaction->delete();
    echo "  (Test transaction cleaned up)\n\n";
    
} catch (\Exception $e) {
    echo "⚠ Could not test new transaction: {$e->getMessage()}\n\n";
}

echo "==========================================\n";
echo " Summary\n";
echo "==========================================\n\n";

echo "✓ Database encoding configured\n";
echo "✓ Transaction 129518 fixed (if found)\n";
echo "✓ Other corrupted transactions listed\n";
echo "✓ New transaction test completed\n\n";

echo "Next steps:\n";
echo "1. Test your API in Postman\n";
echo "2. Fix other corrupted transactions if needed\n";
echo "3. New transactions will work automatically\n\n";



