<?php

require_once 'vendor/autoload.php';

use Illuminate\Support\Facades\DB;
use App\Models\Notification;

// Bootstrap Laravel
$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

try {
    echo "Fixing corrupted Kurdish text in notifications...\n\n";
    
    // Find all notifications with corrupted Kurdish text
    $corruptedNotifications = Notification::where('body', 'like', '%?????%')
        ->orWhere('body', 'like', '%?%USD%')
        ->get();
    
    echo "Found {$corruptedNotifications->count()} corrupted notifications\n\n";
    
    $fixedCount = 0;
    
    foreach ($corruptedNotifications as $notification) {
        echo "Processing notification ID: {$notification->id}\n";
        echo "Original body: {$notification->body}\n";
        
        // Try to extract information from the data field
        $data = $notification->data;
        if (is_string($data)) {
            $data = json_decode($data, true);
        }
        
        $newBody = null;
        
        // Check if it's a currency rate notification
        if (isset($data['type']) && $data['type'] === 'currency_rate_update') {
            $currencyName = $data['currency_name'] ?? 'USD';
            $oldRate = $data['old_rate'] ?? null;
            $newRate = $data['value'] ?? null;
            
            if ($oldRate !== null && $newRate !== null) {
                $change = $newRate > $oldRate ? 'زیادبووە' : 'کەمبووەتەوە';
                $newBody = "ڕێژەی دراوی $currencyName لە $oldRate $change بۆ $newRate";
            } else {
                $newBody = "ڕێژەی دراوی $currencyName نوێکرایەوە";
            }
        } elseif ($notification->type === 'test') {
            $newBody = "تێستکردنی نوێ بۆ بەکارهێنەر - " . $notification->created_at->format('Y-m-d H:i:s');
        } else {
            $newBody = "ئاگادارکردنەوەی نوێ - " . $notification->created_at->format('Y-m-d H:i:s');
        }
        
        if ($newBody) {
            echo "New body: $newBody\n";
            
            // Update the notification
            $notification->body = $newBody;
            $notification->save();
            
            echo "✅ Updated notification ID: {$notification->id}\n";
            $fixedCount++;
        } else {
            echo "❌ Could not reconstruct body for notification ID: {$notification->id}\n";
        }
        
        echo "\n";
    }
    
    echo "=== Summary ===\n";
    echo "Total corrupted notifications found: {$corruptedNotifications->count()}\n";
    echo "Successfully fixed: $fixedCount\n";
    echo "Failed to fix: " . ($corruptedNotifications->count() - $fixedCount) . "\n";
    
} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    echo "Trace: " . $e->getTraceAsString() . "\n";
}







