<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Currency;
use App\Models\CurrencyRateHistory;
use Carbon\Carbon;

class CurrencyRateHistorySeeder extends Seeder
{
    /**
     * Run the database seeds.
     * This will populate 30 days of sample rate history
     */
    public function run(): void
    {
        $this->command->info('Seeding currency rate history...');

        $currencies = Currency::all();

        if ($currencies->isEmpty()) {
            $this->command->warn('No currencies found. Please seed currencies first.');
            return;
        }

        foreach ($currencies as $currency) {
            if (!$currency->buy_rate || !$currency->sell_rate) {
                continue;
            }

            $baseBuyRate = (float) $currency->buy_rate;
            $baseSellRate = (float) $currency->sell_rate;

            // Generate 30 days of history with small random variations
            for ($i = 30; $i >= 0; $i--) {
                $date = Carbon::now()->subDays($i)->startOfDay();
                
                // Add small random variation (±2%)
                $buyVariation = $baseBuyRate * (mt_rand(-200, 200) / 10000);
                $sellVariation = $baseSellRate * (mt_rand(-200, 200) / 10000);
                
                CurrencyRateHistory::create([
                    'currency_id' => $currency->id,
                    'buy_rate' => round($baseBuyRate + $buyVariation, 2),
                    'sell_rate' => round($baseSellRate + $sellVariation, 2),
                    'recorded_at' => $date,
                ]);
            }

            $this->command->info("✓ Created 31 days of history for {$currency->currency_name}");
        }

        $this->command->info('✅ Currency rate history seeded successfully!');
    }
}

