<?php

/**
 * Check notification setup and identify potential issues
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use App\Models\User;
use App\Models\Account;
use App\Models\Customer;
use App\Models\AccountTransaction;
use Illuminate\Support\Facades\DB;

echo "═══════════════════════════════════════════════════════════════\n";
echo "   NOTIFICATION SETUP CHECK\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

// Check 1: Users with notification settings
echo "1. USER NOTIFICATION SETTINGS:\n";
echo "   " . str_repeat("─", 60) . "\n";
$totalUsers = User::count();
$usersWithNotificationsEnabled = User::where('notifications_enabled', true)->count();
$usersWithTransactionNotif = User::where('transaction_notifications', true)->count();
$usersWithCurrencyRateNotif = User::where('currency_rate_notifications', true)->count();

echo "   Total Users: {$totalUsers}\n";
echo "   Notifications Enabled: {$usersWithNotificationsEnabled}\n";
echo "   Transaction Notifications: {$usersWithTransactionNotif}\n";
echo "   Currency Rate Notifications: {$usersWithCurrencyRateNotif}\n\n";

// Check 2: Users with OneSignal player IDs
echo "2. ONESIGNAL INTEGRATION:\n";
echo "   " . str_repeat("─", 60) . "\n";
$usersWithPlayerIds = User::whereNotNull('onesignal_player_id')->count();
$usersWithExternalIds = User::whereNotNull('onesignal_external_id')->count();

echo "   Users with OneSignal Player ID: {$usersWithPlayerIds}\n";
echo "   Users with OneSignal External ID: {$usersWithExternalIds}\n\n";

// Check 3: Account → Customer → User relationship chain
echo "3. ACCOUNT → CUSTOMER → USER RELATIONSHIP:\n";
echo "   " . str_repeat("─", 60) . "\n";
$totalAccounts = Account::count();
$accountsWithCustomers = Account::whereNotNull('customer_id')->count();
$customersWithUsers = Customer::whereNotNull('user_id')->count();
$totalCustomers = Customer::count();

echo "   Total Accounts: {$totalAccounts}\n";
echo "   Accounts with Customers: {$accountsWithCustomers}\n";
echo "   Total Customers: {$totalCustomers}\n";
echo "   Customers with Users: {$customersWithUsers}\n\n";

// Check 4: Accounts that would receive transaction notifications
$accountsWithNotifications = Account::whereHas('customer.user', function($query) {
    $query->where('notifications_enabled', true)
          ->where('transaction_notifications', true);
})->count();

echo "   Accounts eligible for transaction notifications: {$accountsWithNotifications}\n\n";

// Check 5: Sample broken chain (accounts without valid user relationship)
echo "4. POTENTIAL ISSUES:\n";
echo "   " . str_repeat("─", 60) . "\n";

$brokenAccounts = Account::whereDoesntHave('customer.user')->limit(5)->get();
if ($brokenAccounts->count() > 0) {
    echo "   ⚠️  Found " . Account::whereDoesntHave('customer.user')->count() . " accounts without valid user relationship:\n";
    foreach ($brokenAccounts as $account) {
        echo "      - Account #{$account->id} ({$account->account_title})";
        if ($account->customer_id) {
            $customer = Customer::find($account->customer_id);
            if ($customer) {
                echo " → Customer: {$customer->customer_name} (user_id: " . ($customer->user_id ?? 'NULL') . ")";
            } else {
                echo " → Customer ID {$account->customer_id} not found";
            }
        } else {
            echo " → No customer assigned";
        }
        echo "\n";
    }
    echo "\n";
} else {
    echo "   ✅ All accounts have valid user relationships\n\n";
}

// Check 6: Recent transactions without notifications
$recentTransactions = AccountTransaction::with(['from_account.customer.user'])
    ->orderBy('created_at', 'desc')
    ->limit(5)
    ->get();

echo "5. RECENT TRANSACTIONS (Last 5):\n";
echo "   " . str_repeat("─", 60) . "\n";
foreach ($recentTransactions as $trans) {
    echo "   Transaction #{$trans->id} - {$trans->transaction_type} - {$trans->transaction_amount}\n";
    echo "      From Account: #{$trans->from_account_id}";
    
    if ($trans->from_account) {
        echo " ({$trans->from_account->account_title})";
        
        if ($trans->from_account->customer) {
            echo " → Customer: {$trans->from_account->customer->customer_name}";
            
            if ($trans->from_account->customer->user) {
                $user = $trans->from_account->customer->user;
                echo " → User: {$user->name}";
                echo "\n      Notifications: " . ($user->notifications_enabled ? '✓' : '✗');
                echo " | Transaction Notif: " . ($user->transaction_notifications ? '✓' : '✗');
                echo " | Language: " . ($user->language_preference ?? 'NULL');
                echo " | Player ID: " . ($user->onesignal_player_id ? '✓' : '✗');
            } else {
                echo " → ⚠️  NO USER";
            }
        } else {
            echo " → ⚠️  NO CUSTOMER";
        }
    } else {
        echo " ⚠️  ACCOUNT NOT FOUND";
    }
    echo "\n\n";
}

echo "═══════════════════════════════════════════════════════════════\n";
echo "   CHECK COMPLETE\n";
echo "═══════════════════════════════════════════════════════════════\n";













