<?php

/**
 * Check if mobile app is calling OneSignal endpoints
 * Analyzes recent API calls and provides recommendations
 */

require __DIR__.'/vendor/autoload.php';

$app = require_once __DIR__.'/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

use Illuminate\Support\Facades\DB;

echo "═══════════════════════════════════════════════════════════════\n";
echo "   MOBILE APP INTEGRATION CHECK\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

// Check if we can find any log entries about OneSignal
$logFile = storage_path('logs/laravel.log');

if (!file_exists($logFile)) {
    echo "⚠️  Log file not found: {$logFile}\n";
    echo "Cannot check for OneSignal API calls.\n";
    exit(1);
}

echo "Analyzing Laravel logs for OneSignal activity...\n";
echo "Log file: {$logFile}\n\n";

// Read last 5000 lines of log file
$lines = [];
$file = new SplFileObject($logFile, 'r');
$file->seek(PHP_INT_MAX);
$totalLines = $file->key();

$startLine = max(0, $totalLines - 5000);
$file->seek($startLine);

while (!$file->eof()) {
    $lines[] = $file->fgets();
}

$onesignalCalls = 0;
$playerIdUpdates = 0;
$externalIdUpdates = 0;
$subscriptionIdUpdates = 0;
$testNotifications = 0;
$errors = [];

foreach ($lines as $line) {
    if (stripos($line, 'onesignal') !== false) {
        $onesignalCalls++;
        
        if (stripos($line, 'updatePlayerId') !== false) {
            $playerIdUpdates++;
        }
        if (stripos($line, 'updateExternalId') !== false) {
            $externalIdUpdates++;
        }
        if (stripos($line, 'updateSubscriptionId') !== false) {
            $subscriptionIdUpdates++;
        }
        if (stripos($line, 'testNotification') !== false) {
            $testNotifications++;
        }
        if (stripos($line, 'error') !== false || stripos($line, 'failed') !== false) {
            $errors[] = trim(substr($line, 0, 200));
        }
    }
}

echo "Results from last ~5000 log lines:\n";
echo "─────────────────────────────────────────────────────────────\n";
echo "Total OneSignal-related log entries: {$onesignalCalls}\n";
echo "Player ID updates: {$playerIdUpdates}\n";
echo "External ID updates: {$externalIdUpdates}\n";
echo "Subscription ID updates: {$subscriptionIdUpdates}\n";
echo "Test notifications: {$testNotifications}\n";
echo "Errors/Failures: " . count($errors) . "\n\n";

if (count($errors) > 0) {
    echo "Recent Errors (showing last 5):\n";
    echo "─────────────────────────────────────────────────────────────\n";
    foreach (array_slice($errors, -5) as $error) {
        echo "  • {$error}\n";
    }
    echo "\n";
}

// Diagnosis
echo "═══════════════════════════════════════════════════════════════\n";
echo "   DIAGNOSIS\n";
echo "═══════════════════════════════════════════════════════════════\n\n";

if ($onesignalCalls === 0) {
    echo "❌ CRITICAL: No OneSignal API calls found in recent logs!\n\n";
    echo "This means the mobile app is NOT calling the OneSignal endpoints.\n\n";
    echo "MOBILE APP ISSUES TO CHECK:\n";
    echo "─────────────────────────────────────────────────────────────\n";
    echo "1. Is OneSignal SDK initialized in the mobile app?\n";
    echo "2. Is the app calling these endpoints after login:\n";
    echo "   - POST /api/mobile/onesignal/player-id\n";
    echo "   - POST /api/mobile/onesignal/external-id\n";
    echo "3. Check mobile app code for OneSignal integration\n";
    echo "4. Check if app has notification permissions enabled\n\n";
} elseif ($playerIdUpdates === 0 && $subscriptionIdUpdates === 0) {
    echo "⚠️  WARNING: OneSignal calls found, but no Player ID updates!\n\n";
    echo "The app is calling OneSignal endpoints, but not updating Player IDs.\n";
    echo "Check mobile app implementation of Player ID registration.\n\n";
} else {
    echo "✅ Good! Mobile app is calling OneSignal endpoints.\n\n";
    echo "Player ID updates found: {$playerIdUpdates}\n";
    echo "But 0 users have Player IDs in database.\n\n";
    echo "This suggests the updates are failing silently.\n";
    echo "Check the error messages above for clues.\n\n";
}

echo "RECOMMENDED ACTIONS:\n";
echo "─────────────────────────────────────────────────────────────\n";
echo "1. Review mobile app OneSignal integration code\n";
echo "2. Ensure app calls Player ID endpoint on successful login\n";
echo "3. Test with a specific user:\n";
echo "   - Ask them to log out completely\n";
echo "   - Clear app cache/data\n";
echo "   - Log back in\n";
echo "   - Check: php check_user_onesignal.php [their_email]\n\n";
echo "4. Monitor logs in real-time during user login:\n";
echo "   tail -f storage/logs/laravel.log | grep -i onesignal\n\n";

echo "═══════════════════════════════════════════════════════════════\n";













