<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Log;
use App\Services\NotificationTranslationService;

class OneSignalService
{
    private $appId;
    private $restApiKey;
    private $client;
    private $translationService;

    public function __construct()
    {
        $this->appId = config('onesignal.app_id');
        $this->restApiKey = config('onesignal.rest_api_key');
        $this->translationService = new NotificationTranslationService();
        
        // Log configuration for debugging
        Log::info('OneSignal Service initialized', [
            'app_id' => $this->appId ? 'Set (length: ' . strlen($this->appId) . ')' : 'Not Set',
            'rest_api_key' => $this->restApiKey ? 'Set (length: ' . strlen($this->restApiKey) . ')' : 'Not Set',
            'rest_api_key_preview' => $this->restApiKey ? substr($this->restApiKey, 0, 10) . '...' : 'Not Set',
        ]);
        
        $this->client = new Client([
            'base_uri' => 'https://onesignal.com/api/v1/',
            'headers' => [
                'Authorization' => 'Basic ' . $this->restApiKey,
                'Content-Type' => 'application/json; charset=utf-8',
                'Accept' => 'application/json; charset=utf-8',
            ],
        ]);
    }

    /**
     * Send notification to specific users by their user IDs (external user IDs)
     * Note: User ID and External ID are the same in OneSignal
     */
    public function sendNotificationToUsers(array $userIds, string $message, array $data = [], $language = 'en')
    {
        // Validate configuration first
        if (!$this->appId) {
            Log::error('OneSignal App ID not configured');
            return [
                'success' => false,
                'error' => 'OneSignal App ID not configured. Please check your .env file.',
                'error_type' => 'configuration'
            ];
        }

        if (!$this->restApiKey) {
            Log::error('OneSignal REST API Key not configured');
            return [
                'success' => false,
                'error' => 'OneSignal REST API Key not configured. Please check your .env file.',
                'error_type' => 'configuration'
            ];
        }

        if (empty($userIds)) {
            Log::error('No user IDs provided for notification');
            return [
                'success' => false,
                'error' => 'No user IDs provided for notification.',
                'error_type' => 'validation'
            ];
        }

        // Convert user IDs to strings (OneSignal expects strings)
        // User ID and External ID are the same in OneSignal
        $externalUserIds = array_map('strval', $userIds);

        try {
            // Prepare content for different languages
            $contents = $this->prepareMultiLanguageContent($message, $language);
            
            $payload = [
                'app_id' => $this->appId,
                'include_external_user_ids' => $externalUserIds, // Use external user IDs instead of player IDs
                'contents' => $contents,
                'data' => $data,
                'small_icon' => 'ic_stat_onesignal_default',
                'large_icon' => 'ic_stat_onesignal_default',
            ];

            Log::info('Sending OneSignal notification', [
                'app_id' => $this->appId,
                'app_id_length' => strlen($this->appId ?? ''),
                'rest_api_key_length' => strlen($this->restApiKey ?? ''),
                'rest_api_key_preview' => $this->restApiKey ? substr($this->restApiKey, 0, 10) . '...' : 'Not Set',
                'user_ids' => $externalUserIds,
                'message' => $message,
                'payload' => $payload,
                'authorization_header' => 'Basic ' . $this->restApiKey
            ]);

            // Ensure proper UTF-8 encoding for the payload
            $jsonPayload = json_encode($payload, JSON_UNESCAPED_UNICODE);
            
            // Debug the message being sent
            Log::info('OneSignal payload with Kurdish text', [
                'message' => $message,
                'message_length' => strlen($message),
                'message_bytes' => array_map('ord', str_split($message)),
                'json_payload' => $jsonPayload,
                'json_length' => strlen($jsonPayload)
            ]);
            
            $response = $this->client->post('notifications', [
                'body' => $jsonPayload,
                'headers' => [
                    'Content-Type' => 'application/json; charset=utf-8',
                ],
            ]);

            $responseData = json_decode($response->getBody()->getContents(), true);
            
            Log::info('OneSignal notification sent successfully', [
                'user_ids' => $externalUserIds,
                'message' => $message,
                'response' => $responseData,
                'status_code' => $response->getStatusCode()
            ]);

            // Check if OneSignal returned an error in the response
            if (isset($responseData['errors']) && !empty($responseData['errors'])) {
                Log::error('OneSignal API returned errors', [
                    'errors' => $responseData['errors'],
                    'user_ids' => $externalUserIds
                ]);
                
                return [
                    'success' => false,
                    'error' => 'OneSignal API Error: ' . implode(', ', $responseData['errors']),
                    'error_type' => 'onesignal_api',
                    'onesignal_errors' => $responseData['errors']
                ];
            }

            return [
                'success' => true,
                'data' => $responseData,
                'recipients' => count($externalUserIds)
            ];

        } catch (GuzzleException $e) {
            $errorMessage = $e->getMessage();
            $errorType = 'network';
            
            // Parse specific error types
            if (strpos($errorMessage, '401') !== false) {
                $errorMessage = 'OneSignal API authentication failed. Please check your REST API Key.';
                $errorType = 'authentication';
            } elseif (strpos($errorMessage, '400') !== false) {
                $errorMessage = 'Invalid request to OneSignal API. Please check your payload.';
                $errorType = 'validation';
            } elseif (strpos($errorMessage, 'timeout') !== false) {
                $errorMessage = 'OneSignal API request timed out. Please try again.';
                $errorType = 'timeout';
            }

            Log::error('OneSignal notification failed', [
                'error' => $errorMessage,
                'error_type' => $errorType,
                'user_ids' => $externalUserIds,
                'message' => $message,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'error' => $errorMessage,
                'error_type' => $errorType,
                'exception' => $e->getMessage()
            ];
        }
    }

    /**
     * Send notification to all users
     */
    public function sendNotificationToAll(string $message, array $data = [])
    {
        // Validate configuration first
        if (!$this->appId) {
            Log::error('OneSignal App ID not configured');
            return [
                'success' => false,
                'error' => 'OneSignal App ID not configured. Please check your .env file.',
                'error_type' => 'configuration'
            ];
        }

        if (!$this->restApiKey) {
            Log::error('OneSignal REST API Key not configured');
            return [
                'success' => false,
                'error' => 'OneSignal REST API Key not configured. Please check your .env file.',
                'error_type' => 'configuration'
            ];
        }

        try {
            $payload = [
                'app_id' => $this->appId,
                'included_segments' => ['All'],
                'contents' => [
                    'en' => $message,
                    'ku' => $message // Kurdish Sorani support
                ],
                'data' => $data,
                'small_icon' => 'ic_stat_onesignal_default',
                'large_icon' => 'ic_stat_onesignal_default',
            ];

            Log::info('Sending OneSignal broadcast notification', [
                'app_id' => $this->appId,
                'message' => $message,
                'payload' => $payload
            ]);

            // Ensure proper UTF-8 encoding for the payload
            $jsonPayload = json_encode($payload, JSON_UNESCAPED_UNICODE);
            
            $response = $this->client->post('notifications', [
                'body' => $jsonPayload,
                'headers' => [
                    'Content-Type' => 'application/json; charset=utf-8',
                ],
            ]);

            $responseData = json_decode($response->getBody()->getContents(), true);
            
            Log::info('OneSignal broadcast notification sent successfully', [
                'message' => $message,
                'response' => $responseData,
                'status_code' => $response->getStatusCode()
            ]);

            // Check if OneSignal returned an error in the response
            if (isset($responseData['errors']) && !empty($responseData['errors'])) {
                Log::error('OneSignal API returned errors for broadcast', [
                    'errors' => $responseData['errors'],
                    'message' => $message
                ]);
                
                return [
                    'success' => false,
                    'error' => 'OneSignal API Error: ' . implode(', ', $responseData['errors']),
                    'error_type' => 'onesignal_api',
                    'onesignal_errors' => $responseData['errors']
                ];
            }

            return [
                'success' => true,
                'data' => $responseData,
                'recipients' => 'all_users'
            ];

        } catch (GuzzleException $e) {
            $errorMessage = $e->getMessage();
            $errorType = 'network';
            
            // Parse specific error types
            if (strpos($errorMessage, '401') !== false) {
                $errorMessage = 'OneSignal API authentication failed. Please check your REST API Key.';
                $errorType = 'authentication';
            } elseif (strpos($errorMessage, '400') !== false) {
                $errorMessage = 'Invalid request to OneSignal API. Please check your payload.';
                $errorType = 'validation';
            } elseif (strpos($errorMessage, 'timeout') !== false) {
                $errorMessage = 'OneSignal API request timed out. Please try again.';
                $errorType = 'timeout';
            }

            Log::error('OneSignal broadcast notification failed', [
                'error' => $errorMessage,
                'error_type' => $errorType,
                'message' => $message,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'error' => $errorMessage,
                'error_type' => $errorType,
                'exception' => $e->getMessage()
            ];
        }
    }

    /**
     * Send transaction notification
     * Uses user ID directly as External ID (they are the same in OneSignal)
     */
    public function sendTransactionNotification($user, $transaction, $type = 'transaction_completed')
    {
        // Check if user has notifications enabled and transaction notifications enabled
        if (!$user->notifications_enabled || !$user->transaction_notifications) {
            Log::info('Skipping transaction notification - user has disabled notifications', [
                'user_id' => $user->id,
                'notifications_enabled' => $user->notifications_enabled,
                'transaction_notifications' => $user->transaction_notifications,
                'transaction_id' => $transaction->id,
                'type' => $type
            ]);
            return [
                'success' => false,
                'error' => 'User has disabled transaction notifications',
                'error_type' => 'user_preference'
            ];
        }

        // Use user ID directly as External ID (they are the same in OneSignal)
        $userId = $user->id;
        $userLanguage = $user->language_preference ?? 'en';
        
        Log::info('Sending transaction notification using user ID as External ID', [
            'user_id' => $userId,
            'transaction_id' => $transaction->id,
            'type' => $type,
            'user_language' => $userLanguage
        ]);

        $message = $this->getTransactionMessage($transaction, $type, $userLanguage);
        $data = $this->getTransactionData($transaction);

        // Store transaction notification in database with localized title
        try {
            $title = $this->getLocalizedTitle('transaction', $userLanguage);
            \App\Models\Notification::create([
                'type' => 'transaction',
                'title' => $title,
                'body' => $message,
                'user_id' => $userId,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to store transaction notification in database', [
                'user_id' => $userId,
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
            ]);
        }

        return $this->sendNotificationToUsers([$userId], $message, $data, $userLanguage);
    }

    /**
     * Send transaction notification by finding user through account relationship
     */
    public function sendTransactionNotificationByAccount($transaction, $type = 'transaction_completed')
    {
        // Get the user through Account → Customer → User relationship
        $fromAccount = $transaction->from_account;
        
        if (!$fromAccount || !$fromAccount->customer || !$fromAccount->customer->user) {
            Log::warning('Transaction has no associated user through account relationship', [
                'transaction_id' => $transaction->id,
                'from_account_id' => $transaction->from_account_id,
            ]);
            return false;
        }

        $user = $fromAccount->customer->user;
        return $this->sendTransactionNotification($user, $transaction, $type);
    }

    /**
     * Get transaction message based on type and language
     */
    private function getTransactionMessage($transaction, $type, $language = 'en')
    {
        return $this->translationService->getTransactionMessage($transaction, $type, $language);
    }

    /**
     * Get transaction data for notification
     */
    private function getTransactionData($transaction)
    {
        return [
            'type' => 'transaction',
            'transaction_id' => $transaction->id,
            'amount' => $transaction->transaction_amount,
            'currency' => isset($transaction->from_currency_code->code) ? $transaction->from_currency_code->code : 'USD',
            'transaction_type' => $transaction->transaction_type,
            'date' => $transaction->transaction_date,
        ];
    }

    /**
     * Send currency rate update notification
     */
    public function sendCurrencyRateNotification($user, $currency, $newRate, $oldRate = null)
    {
        // Check if user has notifications enabled, currency rate notifications enabled, and currency rate access
        if (!$user->notifications_enabled || !$user->currency_rate_notifications || !$user->currency_rate_access) {
            Log::info('Skipping currency rate notification - user has disabled notifications or no access', [
                'user_id' => $user->id,
                'notifications_enabled' => $user->notifications_enabled,
                'currency_rate_notifications' => $user->currency_rate_notifications,
                'currency_rate_access' => $user->currency_rate_access,
                'currency_code' => $currency->code ?? 'unknown'
            ]);
            return [
                'success' => false,
                'error' => 'User has disabled currency rate notifications or no access',
                'error_type' => 'user_preference'
            ];
        }

        $userId = $user->id;
        $userLanguage = $user->language_preference ?? 'en';
        $message = $this->getCurrencyRateMessage($currency, $newRate, $oldRate, $userLanguage);
        $data = $this->getCurrencyRateData($currency, $newRate, $oldRate);

        Log::info('Sending currency rate notification', [
            'user_id' => $userId,
            'currency_code' => $currency->code ?? 'unknown',
            'new_rate' => $newRate,
            'old_rate' => $oldRate,
            'user_language' => $userLanguage
        ]);

        return $this->sendNotificationToUsers([$userId], $message, $data, $userLanguage);
    }

    /**
     * Send account update notification
     */
    public function sendAccountUpdateNotification($user, $account, $updateType = 'balance_update')
    {
        // Check if user has notifications enabled and account update notifications enabled
        if (!$user->notifications_enabled || !$user->account_update_notifications) {
            Log::info('Skipping account update notification - user has disabled notifications', [
                'user_id' => $user->id,
                'notifications_enabled' => $user->notifications_enabled,
                'account_update_notifications' => $user->account_update_notifications,
                'account_id' => $account->id ?? 'unknown'
            ]);
            return [
                'success' => false,
                'error' => 'User has disabled account update notifications',
                'error_type' => 'user_preference'
            ];
        }

        $userId = $user->id;
        $message = $this->getAccountUpdateMessage($account, $updateType);
        $data = $this->getAccountUpdateData($account, $updateType);

        Log::info('Sending account update notification', [
            'user_id' => $userId,
            'account_id' => $account->id ?? 'unknown',
            'update_type' => $updateType
        ]);

        // Store account update notification in database
        try {
            \App\Models\Notification::create([
                'type' => 'account_update',
                'title' => 'نوێکاری حساب',
                'body' => $message,
                'user_id' => $userId,
                'data' => $data,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to store account update notification in database', [
                'user_id' => $userId,
                'account_id' => $account->id ?? 'unknown',
                'error' => $e->getMessage(),
            ]);
        }

        return $this->sendNotificationToUsers([$userId], $message, $data);
    }

    /**
     * Get currency rate message
     */
    private function getCurrencyRateMessage($currency, $newRate, $oldRate = null, $language = 'en')
    {
        return $this->translationService->getCurrencyRateMessage($currency, $newRate, $oldRate, null, $language);
    }

    /**
     * Get currency rate data for notification
     */
    private function getCurrencyRateData($currency, $newRate, $oldRate = null)
    {
        return [
            'type' => 'currency_rate',
            'currency_code' => $currency->code ?? 'unknown',
            'currency_name' => $currency->currency_name ?? $currency->name ?? 'Unknown Currency',
            'new_rate' => $newRate,
            'old_rate' => $oldRate,
            'timestamp' => now()->toISOString(),
        ];
    }

    /**
     * Get account update message
     */
    private function getAccountUpdateMessage($account, $updateType)
    {
        return $this->translationService->getAccountUpdateMessage($account, $updateType);
    }

    /**
     * Get account update data for notification
     */
    private function getAccountUpdateData($account, $updateType)
    {
        return [
            'type' => 'account_update',
            'account_id' => $account->id ?? null,
            'account_name' => $account->name ?? 'Unknown Account',
            'update_type' => $updateType,
            'balance' => $account->balance ?? null,
            'currency' => $account->currency_code ?? null,
            'timestamp' => now()->toISOString(),
        ];
    }

    /**
     * Send notification in Kurdish Sorani
     */
    public function sendKurdishNotification(array $userIds, string $message, array $data = [])
    {
        return $this->sendNotificationToUsers($userIds, $message, $data);
    }

    /**
     * Send welcome notification in Kurdish
     */
    public function sendWelcomeNotification($user)
    {
        if (!$user->notifications_enabled) {
            return [
                'success' => false,
                'error' => 'User has disabled notifications',
                'error_type' => 'user_preference'
            ];
        }

        $message = $this->translationService->getWelcomeMessage($user->name);
        $data = [
            'type' => 'welcome',
            'user_id' => $user->id,
            'timestamp' => now()->toISOString(),
        ];

        return $this->sendNotificationToUsers([$user->id], $message, $data);
    }

    /**
     * Send payment received notification in Kurdish
     */
    public function sendPaymentReceivedNotification($user, $amount, $currency = 'دینار')
    {
        if (!$user->notifications_enabled) {
            return [
                'success' => false,
                'error' => 'User has disabled notifications',
                'error_type' => 'user_preference'
            ];
        }

        $message = $this->translationService->getPaymentReceivedMessage($amount, $currency);
        $data = [
            'type' => 'payment_received',
            'amount' => $amount,
            'currency' => $currency,
            'user_id' => $user->id,
            'timestamp' => now()->toISOString(),
        ];

        return $this->sendNotificationToUsers([$user->id], $message, $data);
    }

    /**
     * Send low balance warning notification in Kurdish
     */
    public function sendLowBalanceWarningNotification($user, $balance, $currency = 'دینار')
    {
        if (!$user->notifications_enabled) {
            return [
                'success' => false,
                'error' => 'User has disabled notifications',
                'error_type' => 'user_preference'
            ];
        }

        $message = $this->translationService->getLowBalanceWarningMessage($balance, $currency);
        $data = [
            'type' => 'low_balance_warning',
            'balance' => $balance,
            'currency' => $currency,
            'user_id' => $user->id,
            'timestamp' => now()->toISOString(),
        ];

        return $this->sendNotificationToUsers([$user->id], $message, $data);
    }

    /**
     * Send security alert notification in Kurdish
     */
    public function sendSecurityAlertNotification($user)
    {
        if (!$user->notifications_enabled) {
            return [
                'success' => false,
                'error' => 'User has disabled notifications',
                'error_type' => 'user_preference'
            ];
        }

        $message = $this->translationService->getSecurityAlertMessage();
        $data = [
            'type' => 'security_alert',
            'user_id' => $user->id,
            'timestamp' => now()->toISOString(),
        ];

        return $this->sendNotificationToUsers([$user->id], $message, $data);
    }

    /**
     * Prepare multi-language content for OneSignal
     */
    private function prepareMultiLanguageContent($message, $primaryLanguage = 'en')
    {
        $contents = [];
        
        // Always include English as fallback
        $contents['en'] = $message;
        
        // Add primary language if it's not English
        if ($primaryLanguage !== 'en') {
            $contents[$primaryLanguage] = $message;
        }
        
        // Add Kurdish support for backward compatibility
        if ($primaryLanguage !== 'ckb') {
            // For now, use the same message for Kurdish
            // In the future, this could be translated separately
            $contents['ku'] = $message;
        }
        
        return $contents;
    }

    /**
     * Get localized title for notification types
     */
    private function getLocalizedTitle($type, $language = 'en')
    {
        switch ($type) {
            case 'transaction':
                switch ($language) {
                    case 'ar':
                        return 'تحديث المعاملة';
                    case 'ckb':
                        return 'نوێکاری مامەڵە';
                    default: // English
                        return 'Transaction Update';
                }
            case 'currency_rate':
                switch ($language) {
                    case 'ar':
                        return 'تحديث سعر العملة';
                    case 'ckb':
                        return 'نوێکاری نرخ';
                    default: // English
                        return 'Currency Rate Update';
                }
            case 'account_update':
                switch ($language) {
                    case 'ar':
                        return 'تحديث الحساب';
                    case 'ckb':
                        return 'نوێکاری حساب';
                    default: // English
                        return 'Account Update';
                }
            default:
                switch ($language) {
                    case 'ar':
                        return 'إشعار';
                    case 'ckb':
                        return 'ئاگادارکردنەوە';
                    default: // English
                        return 'Notification';
                }
        }
    }
}
