<?php

namespace App\Models;

use Laravel\Sanctum\HasApiTokens;
use App\Models\Scopes\Searchable;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Notifications\Notifiable;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Laravel\Fortify\TwoFactorAuthenticatable;

class User extends Authenticatable
{
    use HasRoles;
    use Notifiable;
    use HasFactory;
    use Searchable;
    use HasApiTokens;
    use LogsActivity;
    use TwoFactorAuthenticatable;

    protected $fillable = [
        'name', 
        'email', 
        'password', 
        'onesignal_player_id', 
        'onesignal_external_id',
        'notifications_enabled',
        'transaction_notifications',
        'currency_rate_notifications',
        'account_update_notifications',
        'currency_rate_access',
        'language_preference',
        'profile_picture'
    ];

    protected $searchableFields = ['*'];

    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'notifications_enabled' => 'boolean',
        'transaction_notifications' => 'boolean',
        'currency_rate_notifications' => 'boolean',
        'account_update_notifications' => 'boolean',
        'currency_rate_access' => 'boolean',
        'language_preference' => 'string',
    ];

    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    public function patientHistories()
    {
        return $this->hasMany(PatientHistory::class, 'created_by');
    }

    /**
     * Get the notifications for the user.
     */
    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }

    public function isSuperAdmin(): bool
    {
        return $this->hasRole('super-admin');
    }

    public function getDescriptionForEvent(string $modelName): string
    {
        return " {$this->title} User was  {$modelName}";
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(string $modelName) => "User has been {$modelName}"
            );
    }
}
