<?php

namespace App\Models;

use App\Models\Scopes\Searchable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class InvoiceItem extends Model
{
    use HasFactory;
    use Searchable;
    use LogsActivity;

    protected $fillable = [
        'invoice_id',
        'subcategory_id',
        'product_id',
        'product_batch_id',
        'quantity',
        'unit_price',
        'total_price',
        'notes',
    ];

    protected $searchableFields = ['*'];

    protected $casts = [
        'quantity' => 'integer',
        'unit_price' => 'decimal:2',
        'total_price' => 'decimal:2',
    ];

    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    public function subcategory()
    {
        return $this->belongsTo(Subcategory::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function productBatch()
    {
        return $this->belongsTo(ProductBatch::class);
    }

    /**
     * Calculate total price
     */
    public function calculateTotal()
    {
        $this->total_price = $this->quantity * $this->unit_price;
        $this->save();
    }

    protected static function boot()
    {
        parent::boot();

        // When creating/updating, calculate total
        static::saving(function ($item) {
            $item->total_price = $item->quantity * $item->unit_price;
        });

        // When saving, update invoice total
        static::saved(function ($item) {
            $item->invoice->calculateTotal();
        });

        // When deleting, update invoice total
        static::deleted(function ($item) {
            $item->invoice->calculateTotal();
        });
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(string $eventName) => "InvoiceItem has been {$eventName}"
            );
    }
}
