<?php

namespace App\Models;

use App\Models\Scopes\Searchable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Builder;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class AccountTransaction extends Model
{
    use HasFactory;
    use Searchable;
    use LogsActivity;

    protected $fillable = [
        'from_account_id',
        'to_account_id',
        'transaction_type',
        'transaction_amount',
        'transaction_date',
        'transaction_time',
        'note',
        'get_fee',
        'person_name',
        'user_id',
        'transaction_id',
        'transaction_group_id', // Add this

        'sender_name',
        'receiver_name',
        'sender_phone',
        'receiver_phone',

        'exchange_rate',  
        'transaction_category',
        'status',
        'rejected_by',
        'approved_by',
        'approved_at',
        'rejected_at',
        'is_primary', // Added here
        'moneyTransferOnly', // <-- Add here
    ];

    protected $searchableFields = ['*'];

    protected $table = 'account_transactions';

    protected $casts = [
        'transaction_date' => 'date',
    ];

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // Define the polymorphic relationship
    public function fees()
    {
        return $this->morphMany(TransactionFee::class, 'feeable');
    }

    public function transaction()
    {
        return $this->belongsTo(Transaction::class, 'transaction_id');
    }

    public function transactionFee()
    {
        return $this->hasOne(TransactionFee::class, 'transaction_id');
    }
    
    public function files()
    {
        return $this->morphMany(File::class, 'fileable');
    }
    
    public function account()
    {
        return $this->belongsTo(Account::class, 'from_account_id');
    }
    public function from_account()
    {
        return $this->belongsTo(Account::class, 'from_account_id');
    }
    public function to_account()
    {
        return $this->belongsTo(Account::class, 'to_account_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function getDescriptionForEvent(string $modelName): string
    {
        return " {$this->title} AccountTransaction was  {$modelName}";
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(
                    string $modelName
                ) => "AccountTransaction has been {$modelName}"
            );
    }

    protected static function booted()
    {
        // Add global scope to sanitize UTF-8 data
        static::addGlobalScope('utf8_sanitize', function (Builder $builder) {
            // This scope will be applied to all queries
            // The actual sanitization happens in the Livewire component
        });

        // Ensure proper UTF-8 encoding for Kurdish text
        static::creating(function ($model) {
            if ($model->note) {
                $model->note = mb_convert_encoding($model->note, 'UTF-8', 'auto');
            }
            if ($model->person_name) {
                $model->person_name = mb_convert_encoding($model->person_name, 'UTF-8', 'auto');
            }
        });
        
        static::updating(function ($model) {
            if ($model->note) {
                $model->note = mb_convert_encoding($model->note, 'UTF-8', 'auto');
            }
            if ($model->person_name) {
                $model->person_name = mb_convert_encoding($model->person_name, 'UTF-8', 'auto');
            }
        });
    }
}
