<?php

namespace App\Models;

use App\Models\Scopes\Searchable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;
use Spatie\Searchable\SearchResult;
use Spatie\Searchable\Searchable as searching;

class Account extends Model implements searching
{
    use HasFactory;
    use Searchable;
    use LogsActivity;

    protected $fillable = [
        'account_title',
        'account_number',
        'balance',
        'customer_id',
        'currency_id',
        'account_type',
    ];

    protected $searchableFields = ['*'];


    public function getSearchResult(): SearchResult
    {
        // Load related customer, user, and currency data
        $this->load(['customer.user', 'currency']);
        
        // Retrieve the customer and currency models
        $customer = $this->customer;
        $currency = $this->currency;
    
        // Set the default URL to redirect for non-cashbox accounts
        $url = route('accounts.show', $this->id);
    
        // Initialize variable to store the displayed name (either customer name or username)
        $displayName = '';
        $userName = $customer->user->name ?? 'N/A';
    
        // Check if the customer exists
        if ($customer) {
            // Check the customer name condition
            if (in_array($customer->customer_name, ['Cashiers', 'Bank (Cashboxes)'])) {
                // For "Cashiers" or "Bank (Cashboxes)", use the username
                $displayName = $userName;
                // Redirect to 'show-cashbox' route for cashbox accounts
                $url = route('show-cashbox', $this->id);
            } else {
                // For other customers, use the customer name
                $displayName = $customer->customer_name;
            }
        }
    
        // Prepare currency title for display
        $currencyTitle = $currency ? $currency->currency_name : 'Unknown Currency';
    
        // Build the HTML for the search result, including customer name or username, and currency title
        $html = '<ul>';
        $html .= '<li><b>' . __('word.AccountNumber') . ': ' . $this->account_title . '</b></li>'; // Using __() for translation
        $html .= '<li>'. __('word.Balance') .' : ' . $this->balance . ' (' . $currencyTitle . ')</li>';
        $html .= '<li>' . ($displayName ? ''. __('word.Name') .': ' . $displayName : '') . '</li>';
        $html .= '</ul>';
    
        // Return a new SearchResult with the HTML, URL, and make the user name searchable
        return new SearchResult(
            $this,
            $html,
            $url,
            $userName // Allows searching by username
        );
    }
    
    

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function calculateTotalBalance()
    {
        // Get the balance from approved transactions
        $approvedBalance = $this->calculateBalance();
        
        // Get the balance from pending transactions
        $pendingBalance = $this->calculateBalancePending();
        
        // Return the sum of both balances
        return $approvedBalance + $pendingBalance;
    }

    /**
     * Optimized balance calculation using database aggregation
     * This method is much faster for bulk operations
     */
    public function calculateTotalBalanceOptimized()
    {
        $accountId = $this->id;
        
        // Calculate approved balance using database aggregation
        $approvedBalance = AccountTransaction::where('from_account_id', $accountId)
            ->where('status', 'approved')
            ->selectRaw('
                SUM(CASE WHEN transaction_type = "deposit" THEN ROUND(transaction_amount) ELSE 0 END) -
                SUM(CASE WHEN transaction_type = "withdrawal" THEN ROUND(transaction_amount) ELSE 0 END) -
                COALESCE((
                    SELECT SUM(tf.fee_amount) 
                    FROM transaction_fees tf 
                    WHERE tf.feeable_id = account_transactions.id 
                    AND tf.feeable_type = "App\\Models\\AccountTransaction"
                ), 0) +
                SUM(CASE WHEN transaction_type = "transfer" AND to_account_id = ? THEN ROUND(get_fee) ELSE 0 END)
            ', [$accountId])
            ->value('balance') ?? 0;

        // Calculate pending balance using database aggregation
        $pendingBalance = AccountTransaction::where('from_account_id', $accountId)
            ->where('moneyTransferOnly', 1)
            ->where('status', 'pending')
            ->selectRaw('
                SUM(CASE WHEN transaction_type = "deposit" THEN ROUND(transaction_amount) ELSE 0 END) -
                SUM(CASE WHEN transaction_type = "withdrawal" THEN ROUND(transaction_amount) ELSE 0 END) -
                COALESCE((
                    SELECT SUM(tf.fee_amount) 
                    FROM transaction_fees tf 
                    WHERE tf.feeable_id = account_transactions.id 
                    AND tf.feeable_type = "App\\Models\\AccountTransaction"
                ), 0) +
                SUM(CASE WHEN transaction_type = "transfer" AND to_account_id = ? THEN ROUND(get_fee) ELSE 0 END)
            ', [$accountId])
            ->value('balance') ?? 0;

        return $approvedBalance + $pendingBalance;
    }

    /**
     * Calculate balances for multiple accounts efficiently using a single query
     * This is much faster than calling calculateTotalBalance() for each account individually
     */
    public static function calculateBalancesForAccounts($accountIds)
    {
        if (empty($accountIds)) {
            return collect();
        }

        // Get approved balances
        $approvedBalances = AccountTransaction::whereIn('from_account_id', $accountIds)
            ->where('status', 'approved')
            ->selectRaw('
                from_account_id,
                SUM(CASE WHEN transaction_type = "deposit" THEN ROUND(transaction_amount) ELSE 0 END) -
                SUM(CASE WHEN transaction_type = "withdrawal" THEN ROUND(transaction_amount) ELSE 0 END) +
                SUM(CASE WHEN transaction_type = "transfer" AND to_account_id = from_account_id THEN ROUND(get_fee) ELSE 0 END) as balance
            ')
            ->groupBy('from_account_id')
            ->pluck('balance', 'from_account_id');

        // Get pending balances
        $pendingBalances = AccountTransaction::whereIn('from_account_id', $accountIds)
            ->where('moneyTransferOnly', 1)
            ->where('status', 'pending')
            ->selectRaw('
                from_account_id,
                SUM(CASE WHEN transaction_type = "deposit" THEN ROUND(transaction_amount) ELSE 0 END) -
                SUM(CASE WHEN transaction_type = "withdrawal" THEN ROUND(transaction_amount) ELSE 0 END) +
                SUM(CASE WHEN transaction_type = "transfer" AND to_account_id = from_account_id THEN ROUND(get_fee) ELSE 0 END) as balance
            ')
            ->groupBy('from_account_id')
            ->pluck('balance', 'from_account_id');

        // Combine balances
        $balances = collect();
        foreach ($accountIds as $accountId) {
            $approvedBalance = $approvedBalances->get($accountId, 0);
            $pendingBalance = $pendingBalances->get($accountId, 0);
            $balances->put($accountId, $approvedBalance + $pendingBalance);
        }

        return $balances;
    }

     public function calculateBalancePending()
    {
        // Total deposits into the account (rounded individually)
        $totalDeposits = $this->pending_transactions()
            ->where('transaction_type', 'deposit')
            ->where('status', 'pending')
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->transaction_amount); // 👈 round individually
            });

        // Total withdrawals from the account (rounded individually)
        $totalWithdrawals = $this->pending_transactions()
            ->where('transaction_type', 'withdrawal')
            ->where('status', 'pending')
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->transaction_amount); // 👈 round individually
            });

        // Calculate fees deducted from the account (already fine, it's fee_amount sums)
        $totalFees = $this->pending_transactions()
            ->with('fees')
            ->get()
            ->sum(function ($transaction) {
                return $transaction->fees->sum('fee_amount');
            });

        // Total received fees for to_account (you can round get_fee too if needed)
        $totalReceivedFees = $this->pending_transactions()
            ->where('transaction_type', 'transfer')
            ->where('status', 'pending')
            ->where('to_account_id', $this->id)
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->get_fee); // 👈 safe rounding
            });

        // Final balance calculation
        $balance = $totalDeposits - $totalWithdrawals - $totalFees + $totalReceivedFees;

        return $balance;
    }

    public function calculateBalance()
    {
        // Total deposits into the account (rounded individually)
        $totalDeposits = $this->transactions()
            ->where('transaction_type', 'deposit')
            ->where('status', 'approved')
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->transaction_amount); // 👈 round individually
            });

        // Total withdrawals from the account (rounded individually)
        $totalWithdrawals = $this->transactions()
            ->where('transaction_type', 'withdrawal')
            ->where('status', 'approved')
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->transaction_amount); // 👈 round individually
            });

        // Calculate fees deducted from the account (already fine, it's fee_amount sums)
        $totalFees = $this->transactions()
            ->with('fees')
            ->get()
            ->sum(function ($transaction) {
                return $transaction->fees->sum('fee_amount');
            });

        // Total received fees for to_account (you can round get_fee too if needed)
        $totalReceivedFees = $this->transactions()
            ->where('transaction_type', 'transfer')
            ->where('status', 'approved')
            ->where('to_account_id', $this->id)
            ->get()
            ->sum(function ($transaction) {
                return round($transaction->get_fee); // 👈 safe rounding
            });

        // Final balance calculation
        $balance = $totalDeposits - $totalWithdrawals - $totalFees + $totalReceivedFees;

        return $balance;
    }


    
    public function calculateProfit()
    {
        // Sum all fees collected across transactions where the account's account_type is 'profit'
        $totalProfit = AccountTransaction::whereHas('from_account', function ($query) {
            $query->where('account_type', 'profit'); // Ensure the account type is 'profit'
        })
        ->with('fees') // Load all fees for the filtered transactions
        ->get()
        ->sum(function ($transaction) {
            return $transaction->fees->sum('fee_amount'); // Sum all fee amounts
        });
    
        return $totalProfit;
    }
    

    public function accountTransactions()
    {
        return $this->hasMany(AccountTransaction::class);
    }

    public function relatedTransactions()
    {
        return $this->hasMany(AccountTransaction::class, 'from_account_id')
        ->where('status', 'approved')
                    ->where(function ($query) {
                        $query->where('to_account_id', '!=', $this->id) // Avoid duplicate reflection
                              ->orWhere('transaction_type', '!=', 'fee'); // Exclude fee-type transactions, if needed
                    });
    }
    
    public function allTransactions()
    {
        return AccountTransaction::where(function ($query) {
            $query->where('from_account_id', $this->id)
                ->where('status', 'approved')
                ->orWhere('to_account_id', $this->id);
        });
    }

    public function pendingTransactions()
    {
        return $this->hasMany(AccountTransaction::class, 'from_account_id')
            ->where('status', '!=', 'rejected'); // Exclude rejected transactions
    }
    public function activeTransactions()
    {
        return $this->hasMany(AccountTransaction::class, 'from_account_id')
            ->whereIn('status', ['pending', 'approved']); // Only include pending and approved
    }
    public function transactions()
    {
        return $this->hasMany(AccountTransaction::class, 'from_account_id')
        ->where('status', 'approved');
    }

    public function pending_transactions()
    {
        return $this->hasMany(AccountTransaction::class, 'from_account_id')
        ->where('moneyTransferOnly',1)
        ->where('status', 'pending');
    }

    // 1. Get the count of all transactions for this account
    public function getCountTransaction()
    {
        return $this->transactions()->count();
    }

    // 2. Get the total sum of all transaction amounts
    public function getTotalAccountTransactionAmount()
    {
        return $this->transactions()->sum('transaction_amount');
    }

    // 3. Get the total deposits (assuming transaction_type = 'deposit' for deposits)
    public function getTotalDeposit()
    {
        return $this->transactions()->where('transaction_type', 'deposit')->sum('transaction_amount');
    }

    // 4. Get the total withdrawals (assuming transaction_type = 'withdrawal' for withdrawals)
    public function getTotalWithdrawal()
    {
        return $this->transactions()->where('transaction_type', 'withdrawal')->sum('transaction_amount');
    }


     // Sum the deposit transactions
     public function totalDeposits()
     {
         return $this->transactions()
             ->where('transaction_type', 'deposit')
             ->where('status', 'approved')
             ->sum('transaction_amount');
     }
 
     // Sum the withdrawal transactions
     public function totalWithdrawals()
     {
         return $this->transactions()
             ->where('transaction_type', 'withdrawal')
             ->where('status', 'approved')
             ->sum('transaction_amount');
     }
 
     // Sum the transfer transactions
     public function totalTransfers()
     {
         return $this->transactions()
             ->where('transaction_type', 'transfer')
             ->where('status', 'approved')
             ->sum('transaction_amount');
     }
 
     // Optionally, a method to get the net balance of deposits and withdrawals
     public function netBalance()
     {
         $deposits = $this->totalDeposits();
         $withdrawals = $this->totalWithdrawals();
         
         return $deposits - $withdrawals;
     }

    public function currency()
    {
        return $this->belongsTo(Currency::class);
    }

    public function getDescriptionForEvent(string $modelName): string
    {
        return " {$this->title} Account was  {$modelName}";
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logExcept(['created_at', 'updated_at'])
            ->logOnlyDirty()
            ->setDescriptionForEvent(
                fn(string $modelName) => "Account has been {$modelName}"
            );
    }
}
