<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use App\Models\Transaction; // Adjust model based on your structure
use App\Exports\TransactionsExport;
use Maatwebsite\Excel\Facades\Excel;

class TransactionsAccount extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $startDate;
    public $endDate;
    public $account;

    public function mount($account)
    {
        $this->account = $account;
        $this->startDate = Carbon::now()->subMonth()->format('Y-m-d'); // Default: last month
        $this->endDate = Carbon::now()->format('Y-m-d'); // Default: today
    }

    public function updated($propertyName)
    {
        // Reset pagination when date filters change
        if (in_array($propertyName, ['startDate', 'endDate'])) {
            $this->resetPage();
            
            // Log the change for debugging
            \Log::info("Date filter updated: {$propertyName} = " . $this->$propertyName);
            
            // Show success message
            session()->flash('success', "Date filter updated: {$this->startDate} to {$this->endDate}");
        }
    }
    
    /**
     * Reset date filters to default values
     */
    public function resetDateFilter()
    {
        $this->startDate = Carbon::now()->subMonth()->format('Y-m-d');
        $this->endDate = Carbon::now()->format('Y-m-d');
        $this->resetPage();
        
        // Show success message
        session()->flash('success', "Date filters reset to defaults");
        
        // Log the reset
        \Log::info("Date filter reset to defaults");
    }

    public function downloadExcel()
    {
        try {
            // Get all filtered transactions without pagination for Excel export
            $allTransactions = $this->account->relatedTransactions()
                ->whereNot('transaction_type', 'fee')
                ->whereNot('to_account_id', $this->account->id)
                ->whereNotIn('from_account_id', [1])
                ->whereBetween('created_at', [
                    Carbon::parse($this->startDate)->startOfDay(),
                    Carbon::parse($this->endDate)->endOfDay()
                ])
                ->orderBy('created_at', 'desc')
                ->get();

            if ($allTransactions->isEmpty()) {
                session()->flash('error', 'No transactions found for the selected date range.');
                return;
            }

            // Calculate totals for all transactions
            $totalIncome = 0;
            $totalExpense = 0;
            $totalDeposit = 0;
            $totalWithdrawal = 0;
            $runningTotal = 0;

            foreach ($allTransactions as $transaction) {
                if ($transaction->transaction_type == 'withdrawal') {
                    $totalExpense += $transaction->transaction_amount;
                    $totalWithdrawal += $transaction->transaction_amount;
                    $runningTotal -= $transaction->transaction_amount;
                } else {
                    $totalIncome += $transaction->transaction_amount;
                    $totalDeposit += $transaction->transaction_amount;
                    $runningTotal += $transaction->transaction_amount;
                }
            }

            // Create completely clean data from scratch to avoid UTF-8 issues
            $cleanTransactions = collect();
            foreach ($allTransactions as $transaction) {
                try {
                    // Extract only numeric and safe data
                    $cleanTransaction = [
                        'id' => (int) $transaction->id,
                        'transaction_type' => (string) $transaction->transaction_type,
                        'transaction_amount' => (float) $transaction->transaction_amount,
                        'created_at' => $transaction->created_at ? $transaction->created_at->format('Y-m-d H:i:s') : 'N/A',
                        'note' => $this->createCleanNote($transaction->note),
                    ];
                    $cleanTransactions->push($cleanTransaction);
                } catch (\Exception $e) {
                    // Skip corrupted transactions
                    \Log::warning('Skipping corrupted transaction ID: ' . ($transaction->id ?? 'unknown'));
                    continue;
                }
            }

            // Create clean account data
            $cleanAccount = [
                'customer_name' => $this->createCleanText($this->account->customer->customer_name ?? 'Customer'),
                'customer_id' => (int) ($this->account->customer->id ?? 0),
                'currency_name' => $this->createCleanText($this->account->currency->currency_name ?? 'Currency'),
                'balance' => (float) ($this->account->calculateTotalBalance() ?? 0),
            ];

            // Generate Excel export
            $fileName = 'transactions-' . $this->startDate . '-to-' . $this->endDate . '.xlsx';
            
            $totals = [
                'totalIncome' => $totalIncome,
                'totalExpense' => $totalExpense,
                'totalDeposit' => $totalDeposit,
                'totalWithdrawal' => $totalWithdrawal,
                'runningTotal' => $runningTotal
            ];
            
            return Excel::download(
                new TransactionsExport($cleanTransactions, $cleanAccount, $this->startDate, $this->endDate, $totals),
                $fileName
            );

        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Excel Generation Error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            session()->flash('error', 'Error generating Excel. Please check the logs for details.');
            return;
        }
    }

    /**
     * Sanitize string to prevent UTF-8 encoding issues
     */
    private function sanitizeString($string)
    {
        if (empty($string)) {
            return '';
        }

        // Convert to string if it's not already
        $string = (string) $string;

        // Remove any invalid UTF-8 sequences
        $string = mb_convert_encoding($string, 'UTF-8', 'UTF-8');
        
        // Remove any remaining invalid characters
        $string = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $string);
        
        // Convert to ASCII-safe characters
        $string = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $string);
        
        // If iconv fails, fall back to basic sanitization
        if ($string === false) {
            $string = preg_replace('/[^\x20-\x7E]/', '', $string);
        }

        return $string ?: 'N/A';
    }

    /**
     * Create completely clean text by removing all problematic characters
     */
    private function createCleanText($text)
    {
        if (empty($text)) {
            return 'N/A';
        }

        try {
            // Convert to string
            $text = (string) $text;
            
            // Remove all non-ASCII characters and replace with safe alternatives
            $cleanText = preg_replace('/[^\x20-\x7E]/', '', $text);
            
            // If text becomes empty, return default
            if (empty($cleanText)) {
                return 'N/A';
            }
            
            return $cleanText;
            
        } catch (\Exception $e) {
            return 'N/A';
        }
    }

    /**
     * Create clean note text
     */
    private function createCleanNote($note)
    {
        if (empty($note)) {
            return '-';
        }

        try {
            // Convert to string
            $note = (string) $note;
            
            // Remove all non-ASCII characters
            $cleanNote = preg_replace('/[^\x20-\x7E]/', '', $note);
            
            // If note becomes empty, return default
            if (empty($cleanNote)) {
                return '-';
            }
            
            return $cleanNote;
            
        } catch (\Exception $e) {
            return '-';
        }
    }

    public function render()
    {
        try {
            // Log the current date values for debugging
            \Log::info("Rendering with dates: startDate = {$this->startDate}, endDate = {$this->endDate}");
            
            // Get transactions for display (paginated)
            $transactions = $this->account->relatedTransactions()
                ->whereNot('transaction_type', 'fee')
                ->whereNot('to_account_id', $this->account->id)
                ->whereNotIn('from_account_id', [1])
                ->whereBetween('created_at', [
                    Carbon::parse($this->startDate)->startOfDay(),
                    Carbon::parse($this->endDate)->endOfDay()
                ])
                ->orderBy('created_at', 'desc')
                ->paginate(50);
            
            // Log the query results for debugging
            \Log::info("Query returned {$transactions->count()} transactions out of {$transactions->total()} total");
        
            return view('livewire.transactions-account', compact('transactions'));
            
        } catch (\Exception $e) {
            // If everything fails, return empty collection
            \Log::error('Livewire render failed: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $transactions = new \Illuminate\Pagination\LengthAwarePaginator(
                collect(),
                0,
                50,
                1,
                ['path' => request()->url()]
            );
            
            return view('livewire.transactions-account', compact('transactions'));
        }
    }
    
}
