<?php

namespace App\Livewire;

use App\Models\AccountTransaction;
use App\Models\Currency;
use App\Models\TransactionFee;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;

class TransactionReport extends Component
{
    use WithPagination;
    protected $paginationTheme = 'bootstrap';
    protected $queryString = ['page'];

    public $startDate;
    public $endDate;
    public $transactionType;
    public $currency;
    public $totalDeposits = 0;
    public $totalTransfers = 0;
    public $totalWithdrawals = 0;
    public $totalFees = 0;
    public $totalTransactions = 0;

    public function mount()
    {
        // Set default dates to today only for faster initial load
        $this->startDate = now()->toDateString();
        $this->endDate = now()->toDateString();
        $this->loadSummaryData();
    }

    public function updated($propertyName)
    {
        // Only reload summary data if filter properties changed
        if (in_array($propertyName, ['startDate', 'endDate', 'transactionType', 'currency'])) {
            $this->loadSummaryData();
        }
    }

    public function loadSummaryData()
    {
        // Create cache key based on current filters
        $cacheKey = 'transaction_summary_' . md5(serialize([
            'startDate' => $this->startDate,
            'endDate' => $this->endDate,
            'transactionType' => $this->transactionType,
            'currency' => $this->currency
        ]));

        // OPTIMIZATION: Cache summary data for 5 minutes to avoid repeated calculations
        $summaryData = cache()->remember($cacheKey, 300, function () {
            // Build base query with filters
            $query = AccountTransaction::query();
        
            // Apply date range filter
            if ($this->startDate && $this->endDate) {
                $query->whereBetween('transaction_date', [$this->startDate, $this->endDate]);
            }
        
            // Apply transaction type filter
            if ($this->transactionType) {
                $query->where('transaction_type', $this->transactionType);
            }
        
            // Apply currency filter by checking related account's currency
            if ($this->currency) {
                $query->whereHas('from_account', function ($accountQuery) {
                    $accountQuery->where('currency_id', $this->currency);
                });
            }
        
            // OPTIMIZATION: Use single query with conditional aggregation for all totals
            return $query->selectRaw('
                COUNT(*) as total_transactions,
                SUM(CASE WHEN transaction_type = "deposit" THEN 1 ELSE 0 END) as total_deposits,
                SUM(CASE WHEN transaction_type = "transfer" THEN 1 ELSE 0 END) as total_transfers,
                SUM(CASE WHEN transaction_type = "withdrawal" THEN 1 ELSE 0 END) as total_withdrawals,
                SUM(get_fee) as total_fees
            ')->first();
        });
    
        // Assign the calculated values
        $this->totalTransactions = $summaryData->total_transactions ?? 0;
        $this->totalDeposits = $summaryData->total_deposits ?? 0;
        $this->totalTransfers = $summaryData->total_transfers ?? 0;
        $this->totalWithdrawals = $summaryData->total_withdrawals ?? 0;
        $this->totalFees = $summaryData->total_fees ?? 0;
    }
    

    public function render()
    {
        // OPTIMIZATION: Use proper eager loading to prevent N+1 queries
        $query = AccountTransaction::with([
            'fees:id,fee_amount,feeable_id,feeable_type',
            'user:id,name',
            'from_account.customer:id,customer_name',
            'from_account.currency:id,currency_name'
        ])
        ->select([
            'id', 'transaction_type', 'transaction_amount', 'transaction_date', 
            'transaction_time', 'person_name', 'note', 'from_account_id', 
            'to_account_id', 'get_fee', 'user_id', 'created_at'
        ]);

        // Apply filters
        if ($this->startDate && $this->endDate) {
            $query->whereBetween('transaction_date', [$this->startDate, $this->endDate]);
        }

        if ($this->transactionType) {
            $query->where('transaction_type', $this->transactionType);
        }

        if ($this->currency) {
            $query->whereHas('from_account', function ($accountQuery) {
                $accountQuery->where('currency_id', $this->currency);
            });
        }

        $transactions = $query->latest('transaction_date')->paginate(25);

        // OPTIMIZATION: Cache currencies to avoid repeated queries
        $currencies = cache()->remember('currencies_for_report', 3600, function () {
            return Currency::select('id', 'currency_name')->get();
        });

        return view('livewire.transaction-report', [
            'transactions' => $transactions,
            'currencies' => $currencies,
        ]);
    }

    public function resetFilters()
    {
        // Reset to today's date for faster loading
        $this->startDate = now()->toDateString();
        $this->endDate = now()->toDateString();
        $this->transactionType = null;
        $this->currency = null;
        $this->loadSummaryData();
    }
}
