<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Transaction;
use App\Models\Cashier;
use App\Models\Account;
use App\Models\Currency;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Models\CashierShift;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\AccountTransaction as AccountTransactionModel;


class TransactionList extends Component
{
    use LivewireAlert, WithPagination;

    public $selectedCashier = '';
    public $startDate;
    public $endDate;
    public $transactionType;
    public $filters;

    public $from_currency_name, $to_currency_name, $transactionId;
    public $cashier_shift_id, $transaction_fee, $customer_name;
    public $transaction_date;
    public $transaction_time;
    public $transaction_amount;
    public $transaction_type;
    public $from_currency;
    public $to_currency;
    public $amount_given , $amount_received ;
    public $currencies;

    public $is_digital_currency = false;
    public $showModal = false;
    public $showProfitSummary = false; // Property to toggle profit summary display

    protected $paginationTheme = 'bootstrap';

    protected $queryString = ['selectedCashier', 'startDate', 'endDate', 'transactionType', 'showProfitSummary'];

    public function mount()
    {
        // Set default dates to today for faster initial loading
        $this->startDate = now()->toDateString();
        $this->endDate = now()->toDateString();
        
        $this->currencies = Currency::all();


        if (Auth::check()) {
            $user = Auth::user();
            $cashier = Cashier::where('user_id', $user->id)->first(); // Get the related Cashier
            if (!$cashier) {
              $this->cashier_shift_id = null;
              //   session()->flash('error', 'Cashier Shift ID is not available.');
               // $this->alert('error', 'Cashier Shift ID is not available!');
  
                return;
          } else {
            $cashier_shift = CashierShift::where('cashier_id', $cashier->id)->first(); // Get the related Cashier
            if ($cashier_shift) {
                $this->cashier_shift_id = $cashier_shift->id;
            } else {
              $this->cashier_shift_id = null;
  
              //   session()->flash('error', 'Cashier Shift ID is not available.');
                $this->alert('error', 'Cashier Shift ID is not available!');
  
                return;
            }
          }
        }

    }

    public function toggleProfitSummary()
    {
        $this->showProfitSummary = !$this->showProfitSummary;
    }

    public function deleteTransaction($transactionId)
    {
        DB::beginTransaction(); // Start a transaction to ensure data consistency
    
        try {
            // Fetch the main transaction
            $transaction = Transaction::findOrFail($transactionId);
    
            // Retrieve all related account transactions using the transaction_id
            $accountTransactions = AccountTransactionModel::with('fees')->where('transaction_id', $transactionId)->get();
    
            if ($accountTransactions->isEmpty()) {
                $this->alert('error', 'کێشەیەک هەیە لە سڕینەوە');
                return;
            }
    
            // Function to delete fees associated with a transaction
            $deleteFees = function ($transaction) {
                foreach ($transaction->fees as $fee) {
                    $fee->delete(); // Delete each fee associated with the transaction
                }
            };
    
            // Delete all related account transactions
            foreach ($accountTransactions as $accountTransaction) {
                // Delete associated fees
                $deleteFees($accountTransaction);
    
                // Delete the account transaction
                $accountTransaction->delete();
            }
    
            // Finally, delete the main transaction
            $transaction->delete();
    
            DB::commit(); // Commit the transaction
    
            // Flash success message
            $this->alert('success', 'سڕینەوەکە سەرکەوتوو بوو!.');

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            session()->flash('error', 'Transaction deletion failed: ' . $e->getMessage());
        }
    }

    public function calculateImprovedProfit()
    {
        // Get the authenticated user
        $user = Auth::user();
        
        // Build base query for transactions
        $query = Transaction::query()
            ->when($this->startDate, function ($query) {
                $query->whereDate('transaction_date', '>=', $this->startDate);
            })
            ->when($this->endDate, function ($query) {
                $query->whereDate('transaction_date', '<=', $this->endDate);
            });
    
        // If the user is a cashier, only show their transactions
        if ($user->hasRole('cashier')) {
            $cashier = Cashier::where('user_id', $user->id)->first();
            if ($cashier) {
                $query->whereHas('cashierShift', function($q) use ($cashier) {
                    $q->where('cashier_id', $cashier->id);
                });
            }
        } else if ($user->hasRole('super-admin') && $this->selectedCashier) {
            // If the user is a super-admin and a cashier is selected
            $query->whereHas('cashierShift', function($q) {
                $q->where('cashier_id', $this->selectedCashier);
            });
        }
    
        // Get all transactions with their relationships
        $transactions = $query->with([
            'cashierShift.cashier.user', 
            'from_currency_code',
            'to_currency_code',
            'fees'
        ])->get();
    
        // Initialize profit summary structure
        $profitSummary = [
            'sell' => [],
            'buy' => [],
            'profit_by_pair' => [],
            'matched_transactions' => [],
            'total_profit' => 0
        ];
        
        // First, organize transactions by currency pair and type
        $sellTransactions = [];
        $buyTransactions = [];
        
        foreach ($transactions as $transaction) {
            $fee = $transaction->fees->first() ? $transaction->fees->first()->fee_amount : 0;
            
            // Skip transactions with invalid/null fee
            if ($fee === null || $fee <= 0) {
                continue;
            }
            
            // Check if both currencies exist
            if (!$transaction->from_currency_code || !$transaction->to_currency_code) {
                continue;
            }
            
            // Skip if either currency has type 1
            if ($transaction->from_currency_code->currency_type == 1 || 
                $transaction->to_currency_code->currency_type == 1) {
                continue;
            }
            
            // Get currency names
            $fromCurrency = $transaction->from_currency_code->currency_name;
            $toCurrency = $transaction->to_currency_code->currency_name;
            
            // Only process USD and IQD currency pairs
            // Check if both currencies are either USD or IQD
            if (!(($fromCurrency == 'USD' && $toCurrency == 'IQD') || 
                  ($fromCurrency == 'IQD' && $toCurrency == 'USD'))) {
                continue;
            }
            
            // Create a standardized currency pair key
            $currencyPair = $fromCurrency . '/' . $toCurrency;
            
            // Populate the appropriate storage based on transaction type
            if ($transaction->transaction_type === 'sell') {
                if (!isset($sellTransactions[$currencyPair])) {
                    $sellTransactions[$currencyPair] = [];
                }
                
                $sellTransactions[$currencyPair][] = [
                    'id' => $transaction->id,
                    'fee' => $fee,
                    'amount_given' => $transaction->amount_given,
                    'amount_received' => $transaction->amount_received,
                    'from_currency' => $fromCurrency,
                    'to_currency' => $toCurrency,
                    'remaining_amount' => $transaction->amount_given // Initialize remaining amount
                ];
                
                // Also store in the summary structure
                if (!isset($profitSummary['sell'][$currencyPair])) {
                    $profitSummary['sell'][$currencyPair] = [];
                }
                
                if (!isset($profitSummary['sell'][$currencyPair][$fee])) {
                    $profitSummary['sell'][$currencyPair][$fee] = [
                        'total_amount_given' => 0,
                        'total_amount_received' => 0,
                        'count' => 0,
                        'fee' => $fee,
                        'given_currency' => $toCurrency,
                        'received_currency' => $fromCurrency
                    ];
                }
                
                $profitSummary['sell'][$currencyPair][$fee]['total_amount_given'] += $transaction->amount_given;
                $profitSummary['sell'][$currencyPair][$fee]['total_amount_received'] += $transaction->amount_received;
                $profitSummary['sell'][$currencyPair][$fee]['count']++;
                
            } elseif ($transaction->transaction_type === 'buy') {
                if (!isset($buyTransactions[$currencyPair])) {
                    $buyTransactions[$currencyPair] = [];
                }
                
                $buyTransactions[$currencyPair][] = [
                    'id' => $transaction->id,
                    'fee' => $fee,
                    'amount_given' => $transaction->amount_given,
                    'amount_received' => $transaction->amount_received,
                    'from_currency' => $fromCurrency,
                    'to_currency' => $toCurrency,
                    'remaining_amount' => $transaction->amount_received // Initialize remaining amount
                ];
                
                // Also store in the summary structure
                if (!isset($profitSummary['buy'][$currencyPair])) {
                    $profitSummary['buy'][$currencyPair] = [];
                }
                
                if (!isset($profitSummary['buy'][$currencyPair][$fee])) {
                    $profitSummary['buy'][$currencyPair][$fee] = [
                        'total_amount_given' => 0,
                        'total_amount_received' => 0,
                        'count' => 0,
                        'fee' => $fee,
                        'given_currency' => $toCurrency,
                        'received_currency' => $fromCurrency
                    ];
                }
                
                $profitSummary['buy'][$currencyPair][$fee]['total_amount_given'] += $transaction->amount_given;
                $profitSummary['buy'][$currencyPair][$fee]['total_amount_received'] += $transaction->amount_received;
                $profitSummary['buy'][$currencyPair][$fee]['count']++;
            }
        }
        
        // Process for both direct pairs and reverse pairs
        $allCurrencyPairs = array_unique(
            array_merge(array_keys($sellTransactions), array_keys($buyTransactions))
        );
        
        foreach ($allCurrencyPairs as $currencyPair) {
            // Parse the currency pair
            list($fromCurrency, $toCurrency) = explode('/', $currencyPair);
            
            // Define the reverse pair
            $reversePair = $toCurrency . '/' . $fromCurrency;
            
            // For direct pairs
            $this->processTransactionPairs($sellTransactions, $buyTransactions, $currencyPair, $currencyPair, $profitSummary);
            
            // For reverse pairs
            $this->processTransactionPairs($sellTransactions, $buyTransactions, $currencyPair, $reversePair, $profitSummary);
        }
        
        return $profitSummary;
    }
    
    // Helper method to process and match transaction pairs
    private function processTransactionPairs(&$sellTransactions, &$buyTransactions, $sellPair, $buyPair, &$profitSummary)
    {
        // Skip if either sell or buy transactions for the given pairs don't exist
        if (!isset($sellTransactions[$sellPair]) || !isset($buyTransactions[$buyPair])) {
            return;
        }
        
        // Sort sell transactions by fee in descending order (highest fee first)
        usort($sellTransactions[$sellPair], function($a, $b) {
            return $b['fee'] <=> $a['fee'];
        });
        
        // Sort buy transactions by fee in ascending order (lowest fee first) 
        usort($buyTransactions[$buyPair], function($a, $b) {
            return $a['fee'] <=> $b['fee'];
        });
        
        // Sort sell transactions by amount_given in descending order (highest amount first)
        usort($sellTransactions[$sellPair], function($a, $b) {
            // Only sort by amount if fees are equal
            if ($a['fee'] == $b['fee']) {
                return $b['remaining_amount'] <=> $a['remaining_amount'];
            }
            return 0; // Maintain the fee-based order
        });
        
        // Sort buy transactions by amount_received in descending order (highest amount first)
        usort($buyTransactions[$buyPair], function($a, $b) {
            // Only sort by amount if fees are equal
            if ($a['fee'] == $b['fee']) {
                return $b['remaining_amount'] <=> $a['remaining_amount'];
            }
            return 0; // Maintain the fee-based order
        });
        
        // Now match and calculate profits
        foreach ($sellTransactions[$sellPair] as $sellKey => $sellTransaction) {
            // Skip if this sell transaction has no remaining amount
            if ($sellTransaction['remaining_amount'] <= 0) {
                continue;
            }
            
            foreach ($buyTransactions[$buyPair] as $buyKey => $buyTransaction) {
                // Skip if this buy transaction has no remaining amount
                if ($buyTransaction['remaining_amount'] <= 0) {
                    continue;
                }
                
                // Calculate fee difference
                $feeDifference = $sellTransaction['fee'] - $buyTransaction['fee'];
                
                // Only consider positive profit
                if ($feeDifference <= 0) {
                    continue;
                }
                
                // Determine the amount to match (the smaller of the two remaining amounts)
                $matchAmount = min($sellTransaction['remaining_amount'], $buyTransaction['remaining_amount']);
                
                // Calculate profit for this match
                $profit = $feeDifference * $matchAmount;
                
                // Update remaining amounts
                $sellTransactions[$sellPair][$sellKey]['remaining_amount'] -= $matchAmount;
                $buyTransactions[$buyPair][$buyKey]['remaining_amount'] -= $matchAmount;
                
                // Determine the profit currency (usually the received currency of the buy transaction)
                $profitCurrency = $buyTransaction['from_currency'];
                
                // Prepare currency pair key for the profit summary
                $profitPairKey = $sellTransaction['from_currency'] . '/' . $sellTransaction['to_currency'];
                
                // Initialize the profit data structure if needed
                if (!isset($profitSummary['profit_by_pair'][$profitPairKey])) {
                    $profitSummary['profit_by_pair'][$profitPairKey] = [
                        'total_profit' => 0,
                        'currency_pairs' => [],
                        'details' => []
                    ];
                }
                
                // Add this profit to the total
                $profitSummary['profit_by_pair'][$profitPairKey]['total_profit'] += $profit;
                $profitSummary['total_profit'] += $profit;
                
                // Store the details for display
                $index = count($profitSummary['profit_by_pair'][$profitPairKey]['details']);
                $profitSummary['profit_by_pair'][$profitPairKey]['details'][] = [
                    'high_fee' => $sellTransaction['fee'],
                    'low_fee' => $buyTransaction['fee'],
                    'fee_difference' => $feeDifference,
                    'amount' => $matchAmount,
                    'profit' => $profit,
                    'currency' => $profitCurrency,
                    'sell_id' => $sellTransaction['id'],
                    'buy_id' => $buyTransaction['id'],
                    'sell_remaining' => $sellTransactions[$sellPair][$sellKey]['remaining_amount'],
                    'buy_remaining' => $buyTransactions[$buyPair][$buyKey]['remaining_amount']
                ];
                
                // Store the currency pair info
                $profitSummary['profit_by_pair'][$profitPairKey]['currency_pairs'][$index] = [
                    'given' => $sellTransaction['from_currency'],
                    'received' => $sellTransaction['to_currency']
                ];
                
                // Store the matched transaction info
                $profitSummary['matched_transactions'][] = [
                    'sell_id' => $sellTransaction['id'],
                    'buy_id' => $buyTransaction['id'],
                    'sell_amount' => $sellTransaction['amount_given'],
                    'buy_amount' => $buyTransaction['amount_received'],
                    'match_amount' => $matchAmount,
                    'sell_fee' => $sellTransaction['fee'],
                    'buy_fee' => $buyTransaction['fee'],
                    'fee_difference' => $feeDifference,
                    'profit' => $profit,
                    'profit_currency' => $profitCurrency,
                    'sell_remaining' => $sellTransactions[$sellPair][$sellKey]['remaining_amount'],
                    'buy_remaining' => $buyTransactions[$buyPair][$buyKey]['remaining_amount']
                ];
                
                // If either transaction has no remaining amount, break the inner loop
                if ($sellTransactions[$sellPair][$sellKey]['remaining_amount'] <= 0) {
                    break;
                }
            }
        }
    }

    public function render()
    {
        // Get the authenticated user
        $user = Auth::user();
        
        // OPTIMIZATION: Use proper eager loading to prevent N+1 queries
        $transactions = Transaction::with([
            'cashierShift.cashier.user:id,name',
            'from_currency_code:id,currency_name',
            'to_currency_code:id,currency_name',
            'fees:id,fee_amount,feeable_id,feeable_type'
        ])
        ->select([
            'id', 'transaction_type', 'amount_given', 'amount_received', 
            'transaction_date', 'transaction_time', 'from_currency', 
            'to_currency', 'cashier_shift_id', 'created_at'
        ])
        ->when($this->startDate, function ($query) {
            $query->whereDate('transaction_date', '>=', $this->startDate);
        })
        ->when($this->endDate, function ($query) {
            $query->whereDate('transaction_date', '<=', $this->endDate);
        })
        ->when($this->transactionType, function ($query) {
            $query->where('transaction_type', $this->transactionType);
        });

        // If the user is a cashier, only show their transactions
        if ($user->hasRole('cashier') || $user->hasRole('supervisor')) {
            $cashier = Cashier::where('user_id', $user->id)->first(); // Get the cashier associated with the user
            if ($cashier) {
                $transactions->whereHas('cashierShift', function($q) use ($cashier) {
                    $q->where('cashier_id', $cashier->id);
                });
            }
        } else if ($user->hasRole('super-admin')) {
            // If the user is a super-admin, allow filtering by selected cashier
            $transactions->when($this->selectedCashier, function ($query) {
                $query->whereHas('cashierShift', function($q) {
                    $q->where('cashier_id', $this->selectedCashier);
                });
            });
        }

        // OPTIMIZATION: Reduce pagination size for better performance
        $transactions = $transactions->orderBy('id', 'desc')->paginate(25); 

        // OPTIMIZATION: Cache cashiers to avoid repeated queries
        $cashiers = [];
        if ($user->hasRole('super-admin')) {
            $cashiers = cache()->remember('cashiers_for_transaction_list', 1800, function () {
                return Cashier::with('user:id,name')
                    ->whereIn('cashier_type', ['cashier', 'supervisor'])
                    ->select('id', 'user_id', 'cashier_name', 'cashier_type')
                    ->get();
            });
        }

        // Calculate profit if needed
        $profitSummary = null;
        if ($this->showProfitSummary) {
            $profitSummary = $this->calculateImprovedProfit();
        }

        return view('livewire.transaction-list', [
            'transactions' => $transactions,
            'cashiers' => $cashiers, // Only available for super-admins
            'profitSummary' => $profitSummary,
        ]);
    }

    public function resetFilters()
    {
        // Reset to today's date for faster loading
        $this->startDate = now()->toDateString();
        $this->endDate = now()->toDateString();
        $this->transactionType = null;
        $this->selectedCashier = null;
        $this->showProfitSummary = false;
    }
}