<?php

namespace App\Livewire;

use App\Models\Cashier;
use App\Models\TransactionFee;
use App\Models\Transaction;
use App\Models\Account;
use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\CashierShift;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Currency; // Import the Currency model
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class TransactionForm extends Component
{
    use LivewireAlert, WithPagination;

    public $from_currency_name, $to_currency_name,$transaction_ID;
    public $cashier_shift_id, $transaction_fee, $customer_name;
    public $transaction_date;
    public $transaction_time;
    public $transaction_amount;
    public $transaction_type;
    public $from_currency;
    public $to_currency;
    public $amount_given , $amount_received ;
    public $currencies = [];
    public $filters;
    public $is_digital_currency = false;
    public $fee_mode = 'auto';
    public $showModal = false;

    public $visibleFromCurrencies = 6; // Number of "From" currencies to show initially
    public $visibleToCurrencies = 6; // Number of "To" currencies to show initially

    private ?Currency $usdCurrency = null;

    // Currency conversion properties
    public $convertedAmount = '';
    public $conversionRate = '';
    public $showConversionInfo = false;


    public function mount()
    {
        $this->filters = array(
            'to_currency' => null,
            'from_currency' => null
        );
        // Set default values for date and time
        $this->transaction_date = now()->format('Y-m-d');
        $this->transaction_time = now()->format('H:i:s');
        $this->currencies = Currency::orderBy('sort_order')->get();
        // Set cashier_shift_id from related Cashier model
        if (Auth::check()) {
          $user = Auth::user();
          $cashier = Cashier::where('user_id',$user->id)->first(); // Get the related Cashier
          if (!$cashier) {
            $this->cashier_shift_id = null;

            //   session()->flash('error', 'Cashier Shift ID is not available.');
              $this->alert('error', 'تکایە سەرەتا کاشێر و دەوامی کاشێر تۆمار بکە');

              return;
        } else {
          $cashier_shift = CashierShift::where('cashier_id',$cashier->id)->first(); // Get the related Cashier
          if ($cashier_shift) {
              $this->cashier_shift_id = $cashier_shift->id;
          } else {
            $this->cashier_shift_id = null;

            //   session()->flash('error', 'Cashier Shift ID is not available.');
            $this->alert('error', 'تکایە سەرەتا کاشێر و دەوامی کاشێر تۆمار بکە');

              return;
          }
        }
      }

    }

    public function loadMoreFromCurrencies()
    {
        $this->visibleFromCurrencies += 5; // Increment by 5
    }

    public function loadMoreToCurrencies()
    {
        $this->visibleToCurrencies += 5; // Increment by 5
    }

    protected $listeners = ['valueSelected'];

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            // It's two words separated by a dot
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
    }

    public function updatedFeeMode($value)
    {
        if ($this->is_digital_currency) {
            return;
        }

        if ($value === 'manual' && !$this->isUsdConversion) {
            $this->fee_mode = 'auto';
            $this->alert('warning', 'Manual fee is only available when one of the currencies is USD.');
        }

        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }

        protected $rules = [
          'amount_received' => 'required|numeric|min:0',
          'transaction_type' => 'required',
          'filters.from_currency' => 'required',
          'filters.to_currency' => 'required|different:filters.from_currency', // Ensures that 'to_currency' is different from 'from_currency'
          'amount_given' => 'required|numeric|min:0',
      ];

      protected $messages = [
        'filters.to_currency.different' => 'The to_currency must be different from the from_currency.',
    ];

    private function getCurrencyName($currencyId)
    {
        // Assuming $currencies is passed to this component
        // global $currencies;
        $currency = Currency::find($currencyId);
        return $currency ? $currency->currency_name : __('word.UnknownCurrency');
    }

    public function closeModal()
    {
        $this->showModal = false; // Close the modal
    }
    public function updatedTransactionAmount()
    {
        $this->calculateAmountGiven();
    }
    public function updatedAmountGiven()
    {
        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }
    
    public function updatedAmountReceived()
    {
        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }
    
    public function updatedTransactionFee()
    {
        if ($this->is_digital_currency) {
            return;
        }

        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }

    public function updatedFiltersFromCurrency()
    {
        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }

    public function updatedFiltersToCurrency()
    {
        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }

    public function updatedIsDigitalCurrency($value): void
    {
        if ($value) {
            $this->clearConversionInfo();
            return;
        }

        $this->calculateAmountGiven();
        $this->calculateCurrencyConversion();
    }

    public function updatedTransactionType()
    {
        // Recalculate amounts when transaction type (buy/sell) changes
        $this->calculateAmountGiven();
    }

    private function clearConversionInfo(): void
    {
        $this->convertedAmount = '';
        $this->conversionRate = '';
        $this->showConversionInfo = false;
    }

    public function getIsUsdConversionProperty(): bool
    {
        $fromCurrency = !empty($this->filters['from_currency']) ? Currency::find($this->filters['from_currency']) : null;
        $toCurrency = !empty($this->filters['to_currency']) ? Currency::find($this->filters['to_currency']) : null;

        return ($fromCurrency && $fromCurrency->currency_name === 'USD')
            || ($toCurrency && $toCurrency->currency_name === 'USD');
    }

    public function calculateAmountGiven()
    {
        if ($this->is_digital_currency) {
            // For digital currency, allow manual entry and skip automatic calculations
            return;
        }

        $amountGiven = (float) str_replace(',', '', (string) $this->amount_given);
        $amountReceived = (float) str_replace(',', '', (string) $this->amount_received);
        $transactionFee = (float) str_replace(',', '', (string) $this->transaction_fee);

        if ($this->fee_mode === 'manual' && $this->isUsdConversion) {
            $this->applyManualFeeCalculations($amountGiven, $amountReceived, $transactionFee);
            return;
        }

        if ($this->fee_mode === 'manual' && !$this->isUsdConversion) {
            $this->fee_mode = 'auto';
        }

        if ($this->transaction_type === 'buy') {
            $calculation = $this->calculateBuyConversion();

            if ($calculation) {
                [$givenAmount, $steps, $targetCurrency] = $calculation;
                $formattedAmount = $this->formatConvertedAmount($givenAmount, $targetCurrency);

                $this->amount_given = $formattedAmount;
                $this->convertedAmount = $formattedAmount;
                $this->conversionRate = implode(' | ', $steps);
                $this->showConversionInfo = true;
                $this->setAutoTransactionFee($amountReceived, $givenAmount);
            } else {
                $this->amount_given = null;
                $this->transaction_fee = '';
                $this->clearConversionInfo();
            }
        } elseif ($this->transaction_type === 'sell') {
            $calculation = $this->calculateSellConversion();

            if ($calculation) {
                [$receivedAmount, $steps, $targetCurrency] = $calculation;
                $formattedAmount = $this->formatConvertedAmount($receivedAmount, $targetCurrency);

                $this->amount_received = $formattedAmount;
                $this->convertedAmount = $formattedAmount;
                $this->conversionRate = implode(' | ', $steps);
                $this->showConversionInfo = true;
                $this->setAutoTransactionFee($amountGiven, $receivedAmount);
            } else {
                $this->amount_received = null;
                $this->transaction_fee = '';
                $this->clearConversionInfo();
            }
        }
    }

    private function applyManualFeeCalculations(float $amountGiven, float $amountReceived, float $transactionFee): void
    {
        $this->clearConversionInfo();

        if (!$this->isUsdConversion) {
            return;
        }

        if ($transactionFee <= 0) {
            if ($this->transaction_type === 'buy') {
                $this->amount_given = null;
            } elseif ($this->transaction_type === 'sell') {
                $this->amount_received = null;
            }

            return;
        }

        $useDivision = $this->shouldUseDivisionForManual();

        if ($this->transaction_type === 'buy') {
            if ($amountReceived <= 0) {
                $this->amount_given = null;
                return;
            }

            $result = $useDivision ? ($amountReceived / $transactionFee) : ($amountReceived * $transactionFee);
            $this->amount_given = $this->formatManualResult($result);
        } elseif ($this->transaction_type === 'sell') {
            if ($amountGiven <= 0) {
                $this->amount_received = null;
                return;
            }

            $result = $useDivision ? ($amountGiven / $transactionFee) : ($amountGiven * $transactionFee);
            $this->amount_received = $this->formatManualResult($result);
        }
    }

    private function shouldUseDivisionForManual(): bool
    {
        $fromCurrency = !empty($this->filters['from_currency']) ? Currency::find($this->filters['from_currency']) : null;
        $toCurrency = !empty($this->filters['to_currency']) ? Currency::find($this->filters['to_currency']) : null;

        return ($fromCurrency && $fromCurrency->use_division_for_buy) || ($toCurrency && $toCurrency->use_division_for_buy);
    }

    private function formatManualResult(float $value): ?string
    {
        if ($value <= 0) {
            return null;
        }

        return fmod($value, 1) == 0.0
            ? number_format($value, 0, '.', ',')
            : number_format($value, 2, '.', ',');
    }

    private function setAutoTransactionFee(float $baseAmount, float $targetAmount): void
    {
        if ($this->fee_mode !== 'auto') {
            return;
        }

        if ($baseAmount <= 0 || $targetAmount <= 0) {
            $this->transaction_fee = '';
            return;
        }

        $rate = $targetAmount / $baseAmount;
        $this->transaction_fee = $this->formatRateValue($rate);
    }

    private function formatRateValue(float $rate): string
    {
        if ($rate <= 0) {
            return '';
        }

        $decimals = $rate < 1 ? 6 : 4;
        $formatted = number_format($rate, $decimals, '.', ',');
        $trimmed = rtrim(rtrim($formatted, '0'), '.');

        return $trimmed === '' ? '0' : $trimmed;
    }

    private function formatNumberWithProperDecimals($number)
    {
        // If the number has no decimal part or it's zero, format with 0 decimal places
        if (floor($number) == $number) {
            return number_format($number, 0, '.', ',');
        }
        
        // Otherwise format with 2 decimal places
        return number_format($number, 2, '.', ',');
    }

    private function validateCurrencyRate($currency, $currencyName, $isBuyTransaction = false)
    {
        if (!$currency) {
            return "دراوی {$currencyName} نەدۆزرایەوە";
        }
        
        $rateField = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
        $rateType = $isBuyTransaction ? 'کڕین' : 'فرۆشتن';
        
        if (empty($currency->$rateField)) {
            return "نرخی {$rateType} بۆ {$currencyName} دانەمەزراوە";
        }
        
        if ($currency->$rateField <= 0) {
            return "نرخی {$rateType} بۆ {$currencyName} دەبێت لە سفر زیاتر بێت";
        }
        
        return null; // No error
    }

    public function calculateCurrencyConversion()
    {
        if ($this->is_digital_currency) {
            $this->clearConversionInfo();
            return;
        }

        if ($this->fee_mode === 'manual' && $this->isUsdConversion) {
            $this->clearConversionInfo();
            return;
        }

        if ($this->fee_mode !== 'auto') {
            $this->clearConversionInfo();
            return;
        }

        // Reset conversion values
        $this->convertedAmount = '';
        $this->conversionRate = '';
        $this->showConversionInfo = false;

        // Check if we have the required values
        if (empty($this->filters['to_currency']) || empty($this->filters['from_currency'])) {
            return;
        }

        // Get currency information
        $toCurrency = Currency::find($this->filters['to_currency']);
        $fromCurrency = Currency::find($this->filters['from_currency']);

        if (!$toCurrency || !$fromCurrency) {
            return;
        }

        if ($this->transaction_type === 'sell') {
            $amount = str_replace(',', '', $this->amount_given);
            if (!is_numeric($amount) || $amount <= 0) {
                return;
            }

            $calculation = $this->calculateSellConversion((float) $amount, $toCurrency, $fromCurrency);

            if ($calculation) {
                [$receivedAmount, $steps] = [$calculation[0], $calculation[1]];
                $formattedAmount = $this->formatConvertedAmount($receivedAmount, $fromCurrency);

                $this->convertedAmount = $formattedAmount;
                $this->conversionRate = implode(' | ', $steps);
                $this->showConversionInfo = true;
                $this->amount_received = $formattedAmount;
            }
        } elseif ($this->transaction_type === 'buy') {
            $amount = str_replace(',', '', $this->amount_received);
            if (!is_numeric($amount) || $amount <= 0) {
                return;
            }

            $calculation = $this->calculateBuyConversion((float) $amount, $fromCurrency, $toCurrency);

            if ($calculation) {
                [$givenAmount, $steps] = [$calculation[0], $calculation[1]];
                $formattedAmount = $this->formatConvertedAmount($givenAmount, $toCurrency);

                $this->convertedAmount = $formattedAmount;
                $this->conversionRate = implode(' | ', $steps);
                $this->showConversionInfo = true;
                $this->amount_given = $formattedAmount;
            }
        } elseif ($fromCurrency->currency_name !== $toCurrency->currency_name) {
            $this->showConversionInfo = true;
            $this->performCurrencyConversion($fromCurrency, $toCurrency);
        }
    }

    private function performCurrencyConversion($fromCurrency, $toCurrency)
    {
        try {
            $amount = str_replace(',', '', $this->amount_given);
            
            if (!is_numeric($amount)) {
                return;
            }

            // Determine which rate to use based on transaction type
            $isBuyTransaction = $this->transaction_type === 'buy';
            
            // Validate that both currencies have valid rates (buy_rate for buy, sell_rate for sell)
            $fromCurrencyError = $this->validateCurrencyRate($fromCurrency, $fromCurrency->currency_name, $isBuyTransaction);
            $toCurrencyError = $this->validateCurrencyRate($toCurrency, $toCurrency->currency_name, $isBuyTransaction);
            
            if ($fromCurrencyError) {
                $this->conversionRate = $fromCurrencyError;
                return;
            }
            
            if ($toCurrencyError) {
                $this->conversionRate = $toCurrencyError;
                return;
            }

            $convertedAmount = 0;
            $conversionSteps = [];

            // Check if either currency uses division instead of multiplication
            $fromUsesDivision = $fromCurrency->use_division_for_buy ?? false;
            $toUsesDivision = $toCurrency->use_division_for_buy ?? false;

            if ($fromCurrency->currency_name === 'USD' && $toCurrency->currency_name === 'IQD') {
                // Direct conversion from USD to IQD
                $rate = $isBuyTransaction ? $toCurrency->buy_rate : $toCurrency->sell_rate;
                $rateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                
                // Check if division should be used based on currency settings
                $useDivision = $toCurrency->use_division_for_buy ?? false;
                
                if ($useDivision) {
                    $convertedAmount = $amount / $rate;
                    $conversionSteps[] = "USD → IQD: {$amount} ÷ {$rate} ({$rateType}) [Division]";
                } else {
                    $convertedAmount = $amount * $rate;
                    $conversionSteps[] = "USD → IQD: {$amount} × {$rate} ({$rateType}) [Multiplication]";
                }
            } elseif ($fromCurrency->currency_name === 'IQD' && $toCurrency->currency_name === 'USD') {
                // Direct conversion from IQD to USD
                $rate = $isBuyTransaction ? $fromCurrency->buy_rate : $fromCurrency->sell_rate;
                $rateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                
                // Check if division should be used based on currency settings
                $useDivision = $fromCurrency->use_division_for_buy ?? false;
                
                if ($useDivision) {
                    $convertedAmount = $amount / $rate;
                    $conversionSteps[] = "IQD → USD: {$amount} ÷ {$rate} ({$rateType}) [Division]";
                } else {
                    $convertedAmount = $amount * $rate;
                    $conversionSteps[] = "IQD → USD: {$amount} × {$rate} ({$rateType}) [Multiplication]";
                }
            } elseif ($fromCurrency->currency_name === 'IQD' || $toCurrency->currency_name === 'IQD') {
                // One currency is IQD, convert through USD
                $usdCurrency = Currency::where('currency_name', 'USD')->first();
                $usdCurrencyError = $this->validateCurrencyRate($usdCurrency, 'USD', $isBuyTransaction);
                
                if ($usdCurrencyError) {
                    $this->conversionRate = $usdCurrencyError;
                    return;
                }

                if ($fromCurrency->currency_name === 'IQD') {
                    // IQD to Foreign Currency: IQD → USD → Foreign
                    $iqdRate = $isBuyTransaction ? $fromCurrency->buy_rate : $fromCurrency->sell_rate;
                    $iqdRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                    
                    // Check if division should be used based on currency settings
                    $iqdUseDivision = $fromCurrency->use_division_for_buy ?? false;
                    
                    if ($iqdUseDivision) {
                        $usdAmount = $amount / $iqdRate;
                        $conversionSteps[] = "IQD → USD: {$amount} ÷ {$iqdRate} ({$iqdRateType}) [Division] = " . number_format($usdAmount, 4);
                    } else {
                        $usdAmount = $amount * $iqdRate;
                        $conversionSteps[] = "IQD → USD: {$amount} × {$iqdRate} ({$iqdRateType}) [Multiplication] = " . number_format($usdAmount, 4);
                    }

                    // USD to Foreign Currency
                    $foreignRate = $isBuyTransaction ? $toCurrency->buy_rate : $toCurrency->sell_rate;
                    $foreignRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                    
                    // Check if division should be used based on currency settings
                    $foreignUseDivision = $toCurrency->use_division_for_buy ?? false;
                    
                    if ($foreignUseDivision) {
                        $convertedAmount = $usdAmount / $foreignRate;
                        $conversionSteps[] = "USD → {$toCurrency->currency_name}: " . number_format($usdAmount, 4) . " ÷ {$foreignRate} ({$foreignRateType}) [Division]";
                    } else {
                        $convertedAmount = $usdAmount * $foreignRate;
                        $conversionSteps[] = "USD → {$toCurrency->currency_name}: " . number_format($usdAmount, 4) . " × {$foreignRate} ({$foreignRateType}) [Multiplication]";
                    }
                } else {
                    // Foreign Currency to IQD: Foreign → USD → IQD
                    $foreignRate = $isBuyTransaction ? $fromCurrency->buy_rate : $fromCurrency->sell_rate;
                    $foreignRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                    
                    // Check if division should be used based on currency settings
                    $foreignUseDivision = $fromCurrency->use_division_for_buy ?? false;
                    
                    if ($foreignUseDivision) {
                        $usdAmount = $amount / $foreignRate;
                        $conversionSteps[] = "{$fromCurrency->currency_name} → USD: {$amount} ÷ {$foreignRate} ({$foreignRateType}) [Division] = " . number_format($usdAmount, 4);
                    } else {
                        $usdAmount = $amount * $foreignRate;
                        $conversionSteps[] = "{$fromCurrency->currency_name} → USD: {$amount} × {$foreignRate} ({$foreignRateType}) [Multiplication] = " . number_format($usdAmount, 4);
                    }

                    // USD to IQD
                    $iqdRate = $isBuyTransaction ? $toCurrency->buy_rate : $toCurrency->sell_rate;
                    $iqdRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                    
                    // Check if division should be used based on currency settings
                    $iqdUseDivision = $toCurrency->use_division_for_buy ?? false;
                    
                    if ($iqdUseDivision) {
                        $convertedAmount = $usdAmount / $iqdRate;
                        $conversionSteps[] = "USD → IQD: " . number_format($usdAmount, 4) . " ÷ {$iqdRate} ({$iqdRateType}) [Division]";
                    } else {
                        $convertedAmount = $usdAmount * $iqdRate;
                        $conversionSteps[] = "USD → IQD: " . number_format($usdAmount, 4) . " × {$iqdRate} ({$foreignRateType}) [Multiplication]";
                    }
                }
            } else {
                // Both currencies are foreign (e.g., EURO to POUND): Foreign1 → USD → Foreign2
                $usdCurrency = Currency::where('currency_name', 'USD')->first();
                $usdCurrencyError = $this->validateCurrencyRate($usdCurrency, 'USD', $isBuyTransaction);
                
                if ($usdCurrencyError) {
                    $this->conversionRate = $usdCurrencyError;
                    return;
                }

                // First currency to USD
                $fromRate = $isBuyTransaction ? $fromCurrency->buy_rate : $fromCurrency->sell_rate;
                $fromRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                
                // Check if division should be used based on currency settings
                $fromUseDivision = $fromCurrency->use_division_for_buy ?? false;
                
                if ($fromUseDivision) {
                    $usdAmount = $amount / $fromRate;
                    $conversionSteps[] = "{$fromCurrency->currency_name} → USD: {$amount} ÷ {$fromRate} ({$fromRateType}) [Division] = " . number_format($usdAmount, 4);
                } else {
                    $usdAmount = $amount * $fromRate;
                    $conversionSteps[] = "{$fromCurrency->currency_name} → USD: {$amount} × {$fromRate} ({$fromRateType}) [Multiplication] = " . number_format($usdAmount, 4);
                }

                // USD to second currency
                $toRate = $isBuyTransaction ? $toCurrency->buy_rate : $toCurrency->sell_rate;
                $toRateType = $isBuyTransaction ? 'buy_rate' : 'sell_rate';
                
                // Check if division should be used based on currency settings
                $toUseDivision = $toCurrency->use_division_for_buy ?? false;
                
                if ($toUseDivision) {
                    $convertedAmount = $usdAmount / $toRate;
                    $conversionSteps[] = "USD → {$toCurrency->currency_name}: " . number_format($usdAmount, 4) . " ÷ {$toRate} ({$toRateType}) [Division]";
                } else {
                    $convertedAmount = $usdAmount * $toRate;
                    $conversionSteps[] = "USD → {$toCurrency->currency_name}: " . number_format($usdAmount, 4) . " × {$toRate} ({$toRateType}) [Multiplication]";
                }
            }

            // Format the converted amount based on the target currency
            if ($toCurrency->currency_name === 'IQD') {
                $this->convertedAmount = number_format($convertedAmount, 0, '.', ',');
            } else {
                $this->convertedAmount = number_format($convertedAmount, 4, '.', ',');
            }
            
            // Create conversion rate display
            $this->conversionRate = implode(' | ', $conversionSteps);

            // Auto-fill the appropriate field with the converted amount based on transaction type
            if ($this->transaction_type === 'sell') {
                // For selling: set converted amount to amount_received (بڕی گۆڕدراو)
                $this->amount_received = $this->convertedAmount;
            } elseif ($this->transaction_type === 'buy') {
                // For buying: set converted amount to amount_given (بڕی پێدراو)
                $this->amount_given = $this->convertedAmount;
            }

        } catch (\Exception $e) {
            $this->conversionRate = 'هەڵە لە گۆڕینی دراو: ' . $e->getMessage();
            \Log::error('Currency conversion error: ' . $e->getMessage());
        }
    }

    private function calculateSellConversion(?float $amount = null, ?Currency $sourceCurrency = null, ?Currency $targetCurrency = null): ?array
    {
        $amountToConvert = $amount ?? str_replace(',', '', $this->amount_given);

        if (!is_numeric($amountToConvert) || $amountToConvert <= 0) {
            return null;
        }

        $sourceCurrency = $sourceCurrency ?? Currency::find($this->filters['to_currency']);
        $targetCurrency = $targetCurrency ?? Currency::find($this->filters['from_currency']);

        if (!$sourceCurrency || !$targetCurrency) {
            return null;
        }

        $usdCurrency = $this->getUsdCurrency();
        if (!$usdCurrency) {
            $this->conversionRate = 'نرخەکانی USD دانەنراون.';
            return null;
        }

        $usdRateError = $this->validateCurrencyRate($usdCurrency, 'USD', false);
        if ($usdRateError) {
            $this->conversionRate = $usdRateError;
            return null;
        }

        if (!in_array($sourceCurrency->currency_name, ['USD', 'IQD'], true)) {
            $sourceError = $this->validateCurrencyRate($sourceCurrency, $sourceCurrency->currency_name, false);
            if ($sourceError) {
                $this->conversionRate = $sourceError;
                return null;
            }
        }

        if (!in_array($targetCurrency->currency_name, ['USD', 'IQD'], true)) {
            $targetError = $this->validateCurrencyRate($targetCurrency, $targetCurrency->currency_name, false);
            if ($targetError) {
                $this->conversionRate = $targetError;
                return null;
            }
        }

        $conversion = $this->convertViaUsd((float) $amountToConvert, $sourceCurrency, $targetCurrency, 'sell_rate');

        if (!$conversion) {
            return null;
        }

        [$convertedAmount, $steps] = $conversion;

        return [$convertedAmount, $steps, $targetCurrency];
    }

    private function calculateBuyConversion(?float $amount = null, ?Currency $sourceCurrency = null, ?Currency $targetCurrency = null): ?array
    {
        $amountToConvert = $amount ?? str_replace(',', '', $this->amount_received);

        if (!is_numeric($amountToConvert) || $amountToConvert <= 0) {
            return null;
        }

        $sourceCurrency = $sourceCurrency ?? Currency::find($this->filters['from_currency']);
        $targetCurrency = $targetCurrency ?? Currency::find($this->filters['to_currency']);

        if (!$sourceCurrency || !$targetCurrency) {
            return null;
        }

        $usdCurrency = $this->getUsdCurrency();
        if (!$usdCurrency) {
            $this->conversionRate = 'نرخەکانی USD دانەنراون.';
            return null;
        }

        $usdRateError = $this->validateCurrencyRate($usdCurrency, 'USD', true);
        if ($usdRateError) {
            $this->conversionRate = $usdRateError;
            return null;
        }

        if (!in_array($sourceCurrency->currency_name, ['USD', 'IQD'], true)) {
            $sourceError = $this->validateCurrencyRate($sourceCurrency, $sourceCurrency->currency_name, true);
            if ($sourceError) {
                $this->conversionRate = $sourceError;
                return null;
            }
        }

        if (!in_array($targetCurrency->currency_name, ['USD', 'IQD'], true)) {
            $targetError = $this->validateCurrencyRate($targetCurrency, $targetCurrency->currency_name, true);
            if ($targetError) {
                $this->conversionRate = $targetError;
                return null;
            }
        }

        $conversion = $this->convertViaUsd((float) $amountToConvert, $sourceCurrency, $targetCurrency, 'buy_rate');

        if (!$conversion) {
            return null;
        }

        [$convertedAmount, $steps] = $conversion;

        return [$convertedAmount, $steps, $targetCurrency];
    }

    private function convertViaUsd(float $amount, Currency $sourceCurrency, Currency $targetCurrency, string $rateField): ?array
    {
        $usd = $this->getUsdCurrency();

        if (!$usd || empty($usd->$rateField) || $usd->$rateField <= 0) {
            return null;
        }

        if ($sourceCurrency->id === $targetCurrency->id) {
            return [$amount, []];
        }

        $steps = [];

        // Step 1: source currency to USD
        if ($sourceCurrency->currency_name === 'USD') {
            $usdAmount = $amount;
        } elseif ($sourceCurrency->currency_name === 'IQD') {
            $usdRate = (float) $usd->$rateField;
            $usdRate = $this->normalizeRateValue($usdRate, $usd);
            if ($usdRate <= 0) {
                return null;
            }
            $usdAmount = $amount / $usdRate;
            $steps[] = sprintf(
                'IQD → USD: %s ÷ %s (%s)',
                $this->formatStepValue($amount, 0),
                $this->formatStepValue($usdRate, 0),
                $rateField
            );
        } else {
            $sourceRate = (float) $sourceCurrency->$rateField;
            $sourceRate = $this->normalizeRateValue($sourceRate, $sourceCurrency);
            if ($sourceRate <= 0) {
                return null;
            }
            $useDivision = (bool) ($sourceCurrency->use_division_for_buy ?? false);
            if ($useDivision) {
                $usdAmount = $amount / $sourceRate;
                $steps[] = sprintf(
                    '%s → USD: %s ÷ %s (%s)',
                    $sourceCurrency->currency_name,
                    $this->formatStepValue($amount),
                    $this->formatStepValue($sourceRate),
                    $rateField
                );
            } else {
                $usdAmount = $amount * $sourceRate;
                $steps[] = sprintf(
                    '%s → USD: %s × %s (%s)',
                    $sourceCurrency->currency_name,
                    $this->formatStepValue($amount),
                    $this->formatStepValue($sourceRate),
                    $rateField
                );
            }
        }

        // Step 2: USD to target currency
        if ($targetCurrency->currency_name === 'USD') {
            $finalAmount = $usdAmount;
        } elseif ($targetCurrency->currency_name === 'IQD') {
            $usdRate = (float) $usd->$rateField;
            $usdRate = $this->normalizeRateValue($usdRate, $usd);
            if ($usdRate <= 0) {
                return null;
            }
            $useDivision = (bool) ($targetCurrency->use_division_for_buy ?? false);
            if ($useDivision) {
                $finalAmount = $usdAmount / $usdRate;
                $steps[] = sprintf(
                    'USD → IQD: %s ÷ %s (%s)',
                    $this->formatStepValue($usdAmount),
                    $this->formatStepValue($usdRate, 0),
                    $rateField
                );
            } else {
                $finalAmount = $usdAmount * $usdRate;
                $steps[] = sprintf(
                    'USD → IQD: %s × %s (%s)',
                    $this->formatStepValue($usdAmount),
                    $this->formatStepValue($usdRate, 0),
                    $rateField
                );
            }
        } else {
            $targetRate = (float) $targetCurrency->$rateField;
            $targetRate = $this->normalizeRateValue($targetRate, $targetCurrency);
            if ($targetRate <= 0) {
                return null;
            }
            $useDivision = (bool) ($targetCurrency->use_division_for_buy ?? false);
            if ($useDivision) {
                $finalAmount = $usdAmount / $targetRate;
                $steps[] = sprintf(
                    'USD → %s: %s ÷ %s (%s)',
                    $targetCurrency->currency_name,
                    $this->formatStepValue($usdAmount),
                    $this->formatStepValue($targetRate),
                    $rateField
                );
            } else {
                $finalAmount = $usdAmount * $targetRate;
                $steps[] = sprintf(
                    'USD → %s: %s × %s (%s)',
                    $targetCurrency->currency_name,
                    $this->formatStepValue($usdAmount),
                    $this->formatStepValue($targetRate),
                    $rateField
                );
            }
        }

        return [$finalAmount, $steps];
    }

    private function normalizeRateValue(float $rate, Currency $currency): float
    {
        if ($rate <= 0) {
            return $rate;
        }

        $hasDecimal = str_contains((string) $rate, '.');

        if ($currency->currency_name === 'IQD') {
            if (!$hasDecimal && $rate >= 10000) {
                return round($rate / 100, 4);
            }

            return $rate;
        }

        if ($rate >= 10) {
            return round($rate / 100, 4);
        }

        if (!$hasDecimal && $rate > 100) {
            return round($rate / 100, 4);
        }

        return round($rate, 4);
    }

    private function getUsdCurrency(): ?Currency
    {
        if ($this->usdCurrency === null) {
            $this->usdCurrency = Currency::where('currency_name', 'USD')->first();
        }

        return $this->usdCurrency;
    }

    private function formatConvertedAmount(float $amount, Currency $currency): string
    {
        if ($currency->currency_name === 'IQD') {
            return number_format($amount, 0, '.', ',');
        }

        return number_format($amount, 4, '.', ',');
    }

    private function formatStepValue(float $value): string
    {
        return rtrim(rtrim(number_format($value, 4, '.', ','), '0'), '.');
    }

    public function submit()
    {
       \Log::info('Submit triggered', [
            'customer_name' => $this->customer_name,
            'transaction_type' => $this->transaction_type,
            'amount_received' => $this->amount_received,
            'amount_given' => $this->amount_given,
            'transaction_fee' => $this->transaction_fee,
            'filters' => $this->filters,
            'is_digital_currency' => $this->is_digital_currency,
            'fee_mode' => $this->fee_mode,
        ]);

         // Remove commas from amount_given and transaction_amount before validation and database operations
        $this->amount_given = str_replace(',', '', $this->amount_given);
        $this->amount_received = str_replace(',', '', $this->amount_received);
        $this->transaction_fee = str_replace(',', '', $this->transaction_fee);

         // to store unique GroupId 
         $transactionGroupId = Str::uuid();

        // Validate the form data
        try {
            $this->validate([
                'cashier_shift_id' => 'required|exists:cashier_shifts,id',
                'transaction_date' => 'required|date',
                'transaction_time' => 'required|date_format:H:i:s',
                'amount_received' => 'required|numeric|min:0',
                'amount_given' => 'required|numeric|min:0',
                'customer_name' => 'nullable|string',
                'transaction_type' => 'required|in:buy,sell',
                'filters.from_currency' => 'required|exists:currencies,id',
                'filters.to_currency' => 'required|different:filters.from_currency', // Ensures that 'to_currency' is different from 'from_currency'
                'transaction_fee' => 'required|min:0', // Added validation for the transaction fee
            ]);
        } catch (ValidationException $e) {
            $firstError = collect($e->errors())->flatten()->first();
            \Log::warning('Submit validation failed', [
                'errors' => $e->errors(),
                'first_error' => $firstError,
            ]);
            session()->flash('error', $firstError ?? 'Invalid form data.');
            throw $e;
        }

        // Check if the cashier shift ID is present
        if ($this->cashier_shift_id === null) {
            session()->flash('error', 'Cashier Shift ID is not available.');
            \Log::warning('Submit aborted: cashier_shift_id missing');
            return;
        }

        DB::beginTransaction(); // Start a transaction to ensure data consistency
        \Log::info('Submit transaction: DB transaction started');

        try {
            // Insert into the 'transactions' table
            $transactionId = DB::table('transactions')->insertGetId([
                'cashier_shift_id' => $this->cashier_shift_id,
                'transaction_date' => $this->transaction_date,
                'transaction_time' => $this->transaction_time,
                'amount_given' => $this->amount_given,
                'note' => $this->customer_name,
                'amount_received' => $this->amount_received,
                'transaction_amount' => $this->amount_received,
                'transaction_type' => $this->transaction_type, // "buy" or "sell"
                'from_currency' => $this->filters['from_currency'],
                'to_currency' => $this->filters['to_currency'],
            ]);
            \Log::info('Transaction inserted', ['transaction_id' => $transactionId]);
            $this->transaction_ID =  $transactionId;
            // Get the logged-in user
            $user = Auth::user();

            // Retrieve the customer related to the user
            $customer = $user->customer;

            if (!$customer) {
                $this->alert('error', 'No customer associated with this user.');
                \Log::warning('Submit aborted: no customer associated with user', ['user_id' => $user?->id]);
                return;
            }

            \Log::info('Customer resolved for transaction', ['customer_id' => $customer->id]);

            // Fetch the account for the customer based on the selected currency
            $fromAccount = Account::where('currency_id', $this->filters['from_currency'])
                ->where('customer_id', $customer->id) // Ensure the account belongs to the customer
                ->firstOrFail();
            $toAccount = Account::where('currency_id', $this->filters['to_currency'])
                ->where('customer_id', $customer->id) // Ensure account belongs to customer_id 1
                ->firstOrFail();

            \Log::info('Accounts resolved', [
                'from_account' => $fromAccount->id ?? null,
                'to_account' => $toAccount->id ?? null,
                'is_digital' => $this->is_digital_currency,
                'transaction_type' => $this->transaction_type,
            ]);

            if ($this->is_digital_currency && $this->transaction_type === 'sell') {
                // Fetch the account for the customer based on the selected currency
                $profitAccount = Account::where('currency_id', $toAccount->currency_id)
                    ->where('customer_id', 1)
                    ->where('account_type', 'profit')
                    ->first();
                $MainAccount = Account::where('currency_id', $toAccount->currency_id)
                    ->where('customer_id', 1)
                    ->first();

            }else{
                    // Fetch the account for the customer based on the selected currency
                $profitAccount = Account::where('currency_id', $fromAccount->currency_id)
                    ->where('customer_id', 1)
                    ->where('account_type', 'profit')
                    ->first();
                $MainAccount = Account::where('currency_id', $fromAccount->currency_id)
                    ->where('customer_id', 1)
                    ->first();

            }

            \Log::info('Profit/Main accounts resolved', [
                'profit_account' => $profitAccount->id ?? null,
                'main_account' => $MainAccount->id ?? null,
            ]);

            // Check if $ProfitAccount is null
            if ($profitAccount == null) {
                $this->alert('error', 'قاسەی قازانج بۆ دراوی دیاریکراو بەردەست نیە، تکایە تۆماری بکە.');
                \Log::warning('Submit aborted: profit account missing', [
                    'from_currency' => $this->filters['from_currency'],
                    'to_currency' => $this->filters['to_currency'],
                    'is_digital_currency' => $this->is_digital_currency,
                ]);
                return; // Stop further execution
            }

            if ($this->is_digital_currency) {
                \Log::info('Processing digital currency transaction path');
                // Insert into the 'account_transactions' table
                $accountTransaction = AccountTransactionModel::create([
                    'from_account_id' => $fromAccount->id,
                    'to_account_id' => $toAccount->id,
                    'transaction_type' => "deposit", // deposit or withdrawal
                    'transaction_amount' => $this->amount_received,
                    'get_fee'=> $this->transaction_fee,
                    'transaction_date' => $this->transaction_date,
                    'transaction_time' => $this->transaction_time,
                    'note' => 'جوڵە کردن لە  ' . $fromAccount->currency->currency_name . ' بۆ ' . $toAccount->currency->currency_name,
                    'user_id' => Auth::id(),
                    'transaction_id' => $transactionId,
                    'transaction_group_id' => $transactionGroupId,
                ]);
    
                $reciprocalTransaction = AccountTransactionModel::create([
                    'from_account_id' => $toAccount->id,
                    'to_account_id' => $fromAccount->id,
                    'transaction_type' => "withdrawal", // withdrawal or deposit
                    'transaction_amount' => $this->amount_given,
                    'get_fee'=> $this->transaction_fee,
                    'transaction_date' => $this->transaction_date,
                    'transaction_time' => $this->transaction_time,
                    'note' => 'جوڵە کردن لە ' . $toAccount->currency->currency_name . ' بۆ ' . $fromAccount->currency->currency_name,
                    'user_id' => Auth::id(),
                    'transaction_id' => $transactionId,
                    'transaction_group_id' => $transactionGroupId,
                ]);
    
                if ($this->transaction_type === 'sell') {
                    $type = 'withdrawal';
                    $amount = $this->amount_given;
                }else{
                    $type = 'deposit';
                    $amount = $this->amount_received;
                }
                 // insert fee to main account
                 $mainAccountTransaction = AccountTransactionModel::create([
                    'from_account_id' => $MainAccount->id,
                    'to_account_id' => $fromAccount->id,
                    'transaction_type' => $type, // withdrawal or deposit
                    'transaction_amount' => $amount,
                    'get_fee'=> 0,
                    'transaction_date' => $this->transaction_date,
                    'transaction_time' => $this->transaction_time,
                    'note' => 'جوڵە کردن لە ' . $toAccount->currency->currency_name . ' بۆ ' . $fromAccount->currency->currency_name,
                    'user_id' => Auth::id(),
                    'transaction_id' => $transactionId,
                    'transaction_group_id' => $transactionGroupId,
                ]);
    
    
                // insert fee to profit account
                $profitAccountTransaction = AccountTransactionModel::create([
                    'from_account_id' => $profitAccount->id,
                    'to_account_id' => $fromAccount->id,
                    'transaction_type' => "deposit", // withdrawal or deposit
                    'transaction_amount' => $this->transaction_fee,
                    'get_fee'=> 0,
                    'transaction_date' => $this->transaction_date,
                    'transaction_time' => $this->transaction_time,
                    'note' => 'جوڵە کردن لە ' . $toAccount->currency->currency_name . ' بۆ ' . $fromAccount->currency->currency_name,
                    'user_id' => Auth::id(),
                    'transaction_id' => $transactionId,
                    'transaction_group_id' => $transactionGroupId,
                ]);
    
            }else{
             // Insert into the 'account_transactions' table
             $accountTransaction = AccountTransactionModel::create([
                'from_account_id' => $fromAccount->id,
                'to_account_id' => $toAccount->id,
                'transaction_type' => "deposit", // deposit or withdrawal
                'transaction_amount' => $this->amount_received,
                'get_fee'=> $this->transaction_fee,
                'transaction_date' => $this->transaction_date,
                'transaction_time' => $this->transaction_time,
                'note' => 'جوڵە کردن لە  ' . $fromAccount->currency->currency_name . ' بۆ ' . $toAccount->currency->currency_name,
                'user_id' => Auth::id(),
                'transaction_id' => $transactionId,
                'transaction_group_id' => $transactionGroupId,
            ]);

            $reciprocalTransaction = AccountTransactionModel::create([
                'from_account_id' => $toAccount->id,
                'to_account_id' => $fromAccount->id,
                'transaction_type' => "withdrawal", // withdrawal or deposit
                'transaction_amount' => $this->amount_given,
                'get_fee'=> $this->transaction_fee,
                'transaction_date' => $this->transaction_date,
                'transaction_time' => $this->transaction_time,
                'note' => 'جوڵە کردن لە ' . $toAccount->currency->currency_name . ' بۆ ' . $fromAccount->currency->currency_name,
                'user_id' => Auth::id(),
                'transaction_id' => $transactionId,
                'transaction_group_id' => $transactionGroupId,
            ]);
        }

            // Fetch the inserted transaction by its ID using the Transaction model
            $transaction = Transaction::findOrFail($transactionId);

            // Create the polymorphic fee entry in the 'transaction_fees' table using the $transaction model
            if ($this->transaction_fee > 0) {
                $transaction->fees()->create([
                    'fee_amount' => $this->transaction_fee,
                    'fee_type' => 'deposit_fee', // 'deposit_fee' or 'withdrawal_fee'
                ]);
            }

            DB::commit(); // Commit the transaction
            \Log::info('Transaction committed successfully', ['transaction_id' => $transactionId]);
            // Flash success message and reset fields
            // session()->flash('message', 'Transaction and fee processed successfully.');
            $this->alert('success', 'Transaction and fee processed successfully!');

            // Reset form fields
            $this->reset('is_digital_currency' ,'customer_name', 'amount_received', 'amount_given', 'transaction_fee', 'transaction_type', 'convertedAmount', 'conversionRate', 'showConversionInfo', 'fee_mode');
            $this->filters['from_currency'] = null;
            $this->filters['to_currency'] = null;
            $this->dispatch('clearCustomSelect'); // Dispatch Livewire event to clear select fields
           // $this->showModal = false; // Close the modal

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            session()->flash('error', 'Transaction failed: ' . $e->getMessage());
            \Log::error('Transaction submission failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
        }
    }

    public function render()
    {
        // Fetch sorted currencies from the database directly
        $filteredFromCurrencies = Currency::orderBy('sort_order')->take($this->visibleFromCurrencies)->get();
        $filteredToCurrencies = Currency::orderBy('sort_order')->take($this->visibleToCurrencies)->get();
    
        return view('livewire.transaction-form', [
            'filteredFromCurrencies' => $filteredFromCurrencies,
            'filteredToCurrencies' => $filteredToCurrencies,
        ]);
    }
    
}

