<?php

namespace App\Livewire;

use App\Models\AccountTransaction;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;
use Carbon\Carbon;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Facades\DB;

class TransactionApproval extends Component
{
    use LivewireAlert, WithPagination;
    protected $paginationTheme = 'bootstrap';

    public $search = '';
    public $selectedUser = '';
    public $startDate = '';
    public $endDate = '';
    public $transactionType = '';
    public $transactionCategory = '';
    public $selectedTransaction = null;
    public $approvalNote = '';
    public $rejectionReason = '';
    public $showApprovalModal = false;
    public $showRejectionModal = false;
    public $approvedTransactionsPagination = 10; // For controlling approved transactions pagination
    
    // Added for phone editing functionality
    public $editingTransaction = null;
    public $editingReceiverPhone = '';
    public $showPhoneEditModal = false;
    
    // Added for note editing functionality
    public $editingNote = '';
    public $showNoteEditModal = false;

    protected $listeners = ['refreshTransactions' => '$refresh'];

    protected $queryString = [
        'search' => ['except' => ''],
        'selectedUser' => ['except' => ''],
        'startDate' => ['except' => ''],
        'endDate' => ['except' => ''],
        'transactionType' => ['except' => ''],
        'transactionCategory' => ['except' => ''],
    ];

    public function mount()
    {
        // Set default dates to current month if not provided
        if (empty($this->startDate)) {
            $this->startDate = Carbon::now()->startOfMonth()->format('Y-m-d');
        }
        
        if (empty($this->endDate)) {
            $this->endDate = Carbon::now()->format('Y-m-d');
        }
    }

    public function render()
    {
        $users = User::whereHas('roles', function($query) {
            $query->whereIn('name', ['super-admin', 'supervisor', 'cashier']);
        })->orderBy('name')->get();

        $transactions = $this->getFilteredTransactions();
        $approvedTransactions = $this->getApprovedTransactions();
        
        return view('livewire.transaction-approval', [
            'transactions' => $transactions,
            'approvedTransactions' => $approvedTransactions,
            'users' => $users,
        ]);
    }

    public function getFilteredTransactions()
    {
        return AccountTransaction::where('status', 'pending')
            // Use a subquery to get the first transaction of each group
            ->whereIn('id', function($query) {
                $query->select(DB::raw('MIN(id)'))
                    ->from('account_transactions')
                    ->where('status', 'pending')
                    ->groupBy('transaction_group_id');
            })
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->whereHas('from_account.customer', function ($subQuery) {
                        $subQuery->where('customer_name', 'like', '%' . $this->search . '%');
                    })
                    ->orWhereHas('to_account.customer', function ($subQuery) {
                        $subQuery->where('customer_name', 'like', '%' . $this->search . '%');
                    })
                    ->orWhere('note', 'like', '%' . $this->search . '%')
                    ->orWhere('person_name', 'like', '%' . $this->search . '%')
                    ->orWhere('transaction_id', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->selectedUser, function ($query) {
                $query->where('user_id', $this->selectedUser);
            })
            ->when($this->transactionType, function ($query) {
                $query->where('transaction_type', $this->transactionType);
            })
            ->when($this->transactionCategory, function ($query) {
                $query->where('transaction_category', $this->transactionCategory);
            })
            ->when($this->startDate && $this->endDate, function ($query) {
                $query->whereDate('created_at', '>=', $this->startDate)
                    ->whereDate('created_at', '<=', $this->endDate);
            })
            ->with(['from_account.customer', 'to_account.customer', 'user'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);
    }

    public function getApprovedTransactions()
    {
        return AccountTransaction::where('status', 'approved')
            // Use a subquery to get the first transaction of each group
            ->whereIn('id', function($query) {
                $query->select(DB::raw('MIN(id)'))
                    ->from('account_transactions')
                    ->where('status', 'approved')
                    ->groupBy('transaction_group_id');
            })
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->whereHas('from_account.customer', function ($subQuery) {
                        $subQuery->where('customer_name', 'like', '%' . $this->search . '%');
                    })
                    ->orWhereHas('to_account.customer', function ($subQuery) {
                        $subQuery->where('customer_name', 'like', '%' . $this->search . '%');
                    })
                    ->orWhere('note', 'like', '%' . $this->search . '%')
                    ->orWhere('person_name', 'like', '%' . $this->search . '%')
                    ->orWhere('transaction_id', 'like', '%' . $this->search . '%');
                });
            })
            ->when($this->selectedUser, function ($query) {
                $query->where('user_id', $this->selectedUser);
            })
            ->when($this->transactionType, function ($query) {
                $query->where('transaction_type', $this->transactionType);
            })
            ->when($this->transactionCategory, function ($query) {
                $query->where('transaction_category', $this->transactionCategory);
            })
            ->when($this->startDate && $this->endDate, function ($query) {
                $query->whereDate('created_at', '>=', $this->startDate)
                    ->whereDate('created_at', '<=', $this->endDate);
            })
            ->with(['from_account.customer', 'to_account.customer', 'user', 'approvedBy'])
            ->orderBy('approved_at', 'desc')
            ->paginate($this->approvedTransactionsPagination);
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->selectedUser = '';
        $this->startDate = Carbon::now()->startOfMonth()->format('Y-m-d');
        $this->endDate = Carbon::now()->format('Y-m-d');
        $this->transactionType = '';
        $this->transactionCategory = '';
        $this->resetPage();
    }

    public function openRejectionModal($transactionId)
    {
        $this->selectedTransaction = $transactionId;
        $this->rejectionReason = '';
        $this->showRejectionModal = true;
    }

    public function openApprovalModal($transactionId)
    {
        $this->selectedTransaction = $transactionId;
        $this->approvalNote = '';
        $this->showApprovalModal = true;
    }
    
    public function approveTransaction()
    {
        $transaction = AccountTransaction::findOrFail($this->selectedTransaction);
        
        // Find and update all transactions with the same transaction_group_id
        AccountTransaction::where('transaction_group_id', $transaction->transaction_group_id)
            ->update([
                'moneyTransferOnly' => 0,
                'status' => 'approved',
                'approved_by' => auth()->id(),
                'approved_at' => now()
            ]);
        
        $this->showApprovalModal = false;
        $this->alert('success', 'جوڵەکان بە سەرکەوتوویی پەسەندکران');
    }
    
    public function rejectTransaction()
    {
        $transaction = AccountTransaction::findOrFail($this->selectedTransaction);
        
        // Find and update all transactions with the same transaction_group_id
        AccountTransaction::where('transaction_group_id', $transaction->transaction_group_id)
            ->update([
                'status' => 'rejected',
                'rejected_by' => auth()->id(),
                'rejected_at' => now()
            ]);
        
        $this->showRejectionModal = false;
        $this->alert('success', 'جوڵەکان ڕەتکرانەوە!');
    }
    
    // Phone editing functionality
    public function openPhoneEditModal($transactionId)
    {
        $transaction = AccountTransaction::findOrFail($transactionId);
        $this->editingTransaction = $transactionId;
        $this->editingReceiverPhone = $transaction->receiver_phone ?? '';
        $this->showPhoneEditModal = true;
    }
    
    public function updateReceiverPhone()
    {
        // Validate phone number (you can customize validation rules)
        $this->validate([
            'editingReceiverPhone' => 'required|regex:/^[0-9\+\-\(\)\s]*$/|min:10',
        ], [
            'editingReceiverPhone.required' => 'ژمارەی تەلەفۆن پێویستە',
            'editingReceiverPhone.regex' => 'فۆرماتی ژمارەی تەلەفۆن دروست نییە',
            'editingReceiverPhone.min' => 'ژمارەی تەلەفۆن دەبێت لانیکەم ١٠ ژمارە بێت',
        ]);
        
        $transaction = AccountTransaction::findOrFail($this->editingTransaction);
        
        // Find and update all transactions with the same transaction_group_id
        AccountTransaction::where('transaction_group_id', $transaction->transaction_group_id)
            ->update([
                'receiver_phone' => $this->editingReceiverPhone
            ]);
        
        $this->showPhoneEditModal = false;
        $this->alert('success', 'ژمارەی تەلەفۆن بە سەرکەوتوویی نوێکرایەوە');
        $this->dispatch('refreshTransactions');
    }
    
    // Note editing functionality
    public function openNoteEditModal($transactionId)
    {
        $transaction = AccountTransaction::findOrFail($transactionId);
        $this->editingTransaction = $transactionId;
        $this->editingNote = $transaction->note ?? '';
        $this->showNoteEditModal = true;
    }
    
    public function updateNote()
    {
        $this->validate([
            'editingNote' => 'nullable|string|max:255',
        ], [
            'editingNote.max' => 'تێبینی نابێت لە ٢٥٥ پیت زیاتر بێت',
        ]);
        
        $transaction = AccountTransaction::findOrFail($this->editingTransaction);
        
        // Find and update all transactions with the same transaction_group_id
        AccountTransaction::where('transaction_group_id', $transaction->transaction_group_id)
            ->update([
                'note' => $this->editingNote
            ]);
        
        $this->showNoteEditModal = false;
        $this->alert('success', 'تێبینی بە سەرکەوتوویی نوێکرایەوە');
        $this->dispatch('refreshTransactions');
    }
}