<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Currency;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use App\Services\OneSignalService;
use App\Jobs\SendCurrencyRateNotificationJob;

class ShowCurrencyUser extends Component
{
    public $currencies;
    
    // Variables for editing
    public $editingCurrency = null;
    public $editingField = null;
    public $editValue = null;
    public $originalValue = null;

    // Define protected listener for updating rates
    protected $listeners = ['updateRate' => 'handleUpdateRateEvent'];

    public function mount()
    {
        $this->loadCurrencies();
    }
    
    // Add a method to reload currencies
    public function loadCurrencies()
    {
        // Define the custom order of currencies (same as main ShowCurrency)
                    $currencyOrder = [
                'USD', 'EURO', 'POUND', 'TURKIYE', 'TUMAN', 'EMIRATES', 'JORDAN', 'SWEED', 'NARWEJ', 'SWESRA', 'SAUDI', 'AUSTRALA', 'CANADA'
            ];
        
        // Convert the array to a key-value pair with the position as the value
        $orderMap = array_flip($currencyOrder);
        
        // Fetch the currencies from the database
        $currencies = Currency::whereIn('currency_name', $currencyOrder)->get();
        
        // Sort the currencies based on your custom order
        $this->currencies = $currencies->sortBy(function($currency) use ($orderMap) {
            return $orderMap[$currency->currency_name] ?? 999; // Default high value for any not in the map
        })->values();
    }
    
    // Start editing a rate
    public function startEdit($currencyId, $field, $value = null)
    {
        // Check permissions
        if (!Auth::check() || !Auth::user()->hasRole(['super-admin', 'supervisor','cashier'])) {
            return;
        }
        
        $this->editingCurrency = $currencyId;
        $this->editingField = $field;
        $this->editValue = $value ?? 0; // Use null coalescing operator to default to 0
        $this->originalValue = $value ?? 0;
    }
    
    // Save edited rate
    public function saveEdit()
    {
        // Check permissions again
        if (!Auth::check() || !Auth::user()->hasRole(['super-admin', 'supervisor','cashier'])) {
            return;
        }
        
        if ($this->editingCurrency && $this->editingField) {
            $this->updateRate($this->editingCurrency, $this->editingField, $this->editValue);
        }
        
        $this->resetEditState();
    }
    
    // Cancel editing
    public function cancelEdit()
    {
        $this->resetEditState();
    }
    
    // Reset edit state
    private function resetEditState()
    {
        $this->editingCurrency = null;
        $this->editingField = null;
        $this->editValue = null;
        $this->originalValue = null;
    }
    
    // Method to handle event with potentially fewer parameters
    public function handleUpdateRateEvent($currencyId, $field = null)
    {
        // If field is provided, use it with saved editValue
        if ($field !== null) {
            $this->updateRate($currencyId, $field, $this->editValue);
        } else {
            // If only currencyId is provided in the event
            if ($this->editingCurrency == $currencyId && $this->editingField && $this->editValue !== null) {
                $this->updateRate($currencyId, $this->editingField, $this->editValue);
            }
        }
    }
    
    // Method to update currency rates
    public function updateRate($currencyId, $field, $value = null)
    {
        // Handle case where value might be null
        if ($value === null && $this->editValue !== null) {
            $value = $this->editValue;
        }
        
        // Only allow super-admin and supervisor roles to update rates
        if (!Auth::check() || !Auth::user()->hasRole(['super-admin', 'supervisor','cashier'])) {
            return;
        }
        
        // Validate field name to prevent SQL injection
        if (!in_array($field, ['buy_rate', 'sell_rate'])) {
            return;
        }
        
        // Validate value is numeric
        if (!is_numeric($value)) {
            return;
        }
        
        // Find the currency
        $currency = Currency::find($currencyId);
        if (!$currency) {
            return;
        }
        
        // Get the old rate BEFORE updating (this is the key fix!)
        $oldRate = $currency->$field;
        
        // Debug logging
        \Log::info('Currency rate update', [
            'currency_id' => $currencyId,
            'currency_name' => $currency->currency_name,
            'field' => $field,
            'oldRate' => $oldRate,
            'newRate' => $value,
            'oldRateType' => gettype($oldRate),
            'newRateType' => gettype($value),
            'ratesEqual' => $oldRate == $value,
            'ratesStrictEqual' => $oldRate === $value,
        ]);
        
        // Update the field
        $currency->$field = (float) $value;
        $currency->save();
        
        // Send notification to all users about rate update
        $this->sendRateUpdateNotification($currency, $field, $value, $oldRate);
        
        // Optional: Log the change
        // \Log::info("Currency rate updated: {$currency->currency_name} {$field} changed to {$value} by " . Auth::user()->name);
        
        // Reload currencies to reflect changes
        $this->loadCurrencies();
    }
    
    /**
     * Send notification to users when currency rate is updated
     * Only sends to users who have enabled currency rate notifications
     * Optimized for better performance using queue
     */
    private function sendRateUpdateNotification($currency, $field, $value, $oldRate)
    {
        try {
            // Dispatch the notification job to queue (non-blocking)
            SendCurrencyRateNotificationJob::dispatch(
                $currency,
                $field,
                $value,
                $oldRate,
                Auth::user()->name ?? 'System'
            );
            
            // Log that the job was dispatched
            \Log::info("Currency rate notification job dispatched", [
                'currency' => $currency->currency_name,
                'field' => $field,
                'value' => $value,
                'old_rate' => $oldRate,
                'updated_by' => Auth::user()->name ?? 'System'
            ]);
            
        } catch (\Exception $e) {
            // Log error but don't break the rate update process
            \Log::error("Failed to dispatch currency rate notification job", [
                'currency' => $currency->currency_name,
                'field' => $field,
                'value' => $value,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
    
    public function render()
    {
        // Refresh data every poll for non-editing users
        if (!Auth::check() || !Auth::user()->hasRole(['super-admin', 'supervisor','cashier'])) {
            $this->loadCurrencies();
        }
        
        return view('livewire.show-currency-user', [
            'currencies' => $this->currencies,
        ]);
    }
}