<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Currency;
use Illuminate\Support\Facades\Cache;
use Livewire\Attributes\Lazy;

#[Lazy]
class ShowCurrency extends Component
{
    public $currencies = [];
    public $lastUpdated;

    public function mount()
    {
        $this->loadCurrencies();
    }
    
    public function loadCurrencies()
    {
        // Cache currencies for 30 seconds to reduce database load for high traffic
        $this->currencies = Cache::remember('public_currencies', 30, function () {
                            $currencies = Currency::whereIn('currency_name', [
                    'USD', 'EURO', 'POUND', 'TURKIYE', 'TUMAN', 'EMIRATES', 'JORDAN', 'SWEED', 'NARWEJ', 'SWESRA', 'SAUDI', 'AUSTRALA', 'CANADA'
                ])
            ->select('currency_name', 'buy_rate', 'sell_rate', 'sort_order') // Only select needed columns
            ->get();
            
            // Sort currencies in the exact order specified by client
                            $orderMap = [
                    'USD' => 1, 'EURO' => 2, 'POUND' => 3, 'TURKIYE' => 4, 'TUMAN' => 5,
                    'EMIRATES' => 6, 'JORDAN' => 7, 'SWEED' => 8, 'NARWEJ' => 9, 'SWESRA' => 10,
                    'SAUDI' => 11, 'AUSTRALA' => 12, 'CANADA' => 13
                ];
            
            $currencies = $currencies->sortBy(function ($currency) use ($orderMap) {
                return $orderMap[$currency->currency_name] ?? 999;
            });
            
            // Debug: Log what we found
            \Log::info('Currencies found and sorted:', $currencies->toArray());
            
            return $currencies;
        });
        
        $this->lastUpdated = now()->format('H:i:s');
    }
    
    public function refreshCurrencies()
    {
        Cache::forget('public_currencies');
        $this->loadCurrencies();
    }
    
    public function checkAvailableCurrencies()
    {
        // Get all currencies in the database to see what's available
        $allCurrencies = Currency::select('currency_name')->get();
        \Log::info('All available currencies in database:', $allCurrencies->toArray());
        
        // Get the currencies we're trying to fetch
        $requestedCurrencies = ['USD', 'EURO', 'POUND', 'TURKIYE', 'TUMAN', 'EMIRATES', 'JORDAN', 'SWEED', 'NORWAY', 'SWESRA', 'SAUDI', 'AUSTRALA', 'CANADA'];
        
        // Check which ones exist and which don't
        $existing = Currency::whereIn('currency_name', $requestedCurrencies)->pluck('currency_name')->toArray();
        $missing = array_diff($requestedCurrencies, $existing);
        
        \Log::info('Existing currencies:', $existing);
        \Log::info('Missing currencies:', $missing);
        
        return [
            'all' => $allCurrencies->pluck('currency_name')->toArray(),
            'existing' => $existing,
            'missing' => $missing
        ];
    }
    
    public function setupCurrencyOrder()
    {
        // This method helps set up the correct sort_order for currencies
        $currencyOrder = [
            'USD' => 1, 'EURO' => 2, 'POUND' => 3, 'TURKIYE' => 4, 'TUMAN' => 5,
            'EMIRATES' => 6, 'JORDAN' => 7, 'SWEED' => 8, 'DANEMARK' => 9, 'SWESRA' => 10,
            'SAUDI' => 11, 'AUSTRALA' => 12, 'CANADA' => 13
        ];
        
        foreach ($currencyOrder as $currencyName => $order) {
            Currency::where('currency_name', $currencyName)->update(['sort_order' => $order]);
        }
        
        \Log::info('Currency order updated in database');
        return 'Currency order updated successfully!';
    }
    
    public function render()
    {
        return view('livewire.show-currency', [
            'currencies' => $this->currencies,
        ]);
    }
    
    public function placeholder()
    {
        return view('livewire.currency-skeleton');
    }
}

