<?php

namespace App\Livewire;

use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\Account;
use App\Models\Customer;
use App\Models\TransactionFee;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Carbon\Carbon;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use Illuminate\Support\Str;

class ShowCashBox extends Component
{
    public $account, $showModal = false, $showModalTransfer = false;
    
    use LivewireAlert, WithPagination;
    protected $paginationTheme = 'bootstrap';

    // Create modal properties
    public $create_account_id, $create_person_name;
    public $create_transaction_amount;
    public $create_transaction_date;
    public $create_transaction_fee;
    public $create_transaction_time;
    public $create_note;
    public $filters, $accountCurrencyId;

    // For date range filtering
    public $startDate = null;
    public $endDate = null;
    public $showFilters = false;
    public $perPage = 10;
    public $sortDirection = 'desc';

    public $transactionTypes = ['transfer'];
    public $create_transaction_type = 'transfer'; // Set default to "transfer"
    public $availableAccounts = []; // Add this property to store available accounts

    protected $queryString = [
        'startDate' => ['except' => ''],
        'endDate' => ['except' => ''],
        'sortDirection' => ['except' => 'desc'],
    ];

    public function mount($account)
    {
        $this->account = $account;
        $this->accountCurrencyId = $account->currency_id; // Store the currency ID of the from account
        $this->filters = [
            'to_account' => null,
        ];
        
        // Pre-load available accounts to avoid query issues during rendering
        $this->loadAvailableAccounts();
    }
    
    // New method to load available accounts once
    public function loadAvailableAccounts()
    {
        $this->availableAccounts = Account::where('id', '!=', $this->account->id)
            ->where(function ($query) {
                $query->where('account_type', 'checking')
                    ->orWhere('account_type', 'supervisor')
                    ->orWhere('account_type', 'cashier')
                    ->orWhere('account_title', 'Business Owner Cashbox');
            })
            ->where('currency_id', $this->accountCurrencyId)
            ->get();
            
        // Log for debugging
        \Log::info('Available accounts:', ['count' => $this->availableAccounts->count(), 'accounts' => $this->availableAccounts->pluck('account_type', 'id')]);
    }
    
    public function toggleFilters()
    {
        $this->showFilters = !$this->showFilters;
    }

    public function resetFilters()
    {
        $this->startDate = null;
        $this->endDate = null;
        $this->resetPage();
    }

    public function toggleSortDirection()
    {
        $this->sortDirection = $this->sortDirection === 'desc' ? 'asc' : 'desc';
        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedStartDate()
    {
        $this->resetPage();
    }

    public function updatedEndDate()
    {
        $this->resetPage();
    }

    public function openModalTransfer()
    {
        // Reset the filter first to avoid old selections
        $this->filters['to_account'] = null;
        $this->showModalTransfer = true; // Open the modal
        
        // Reload available accounts to ensure we have fresh data
        $this->loadAvailableAccounts();
    }

    public function closeModalTransfer()
    {
        $this->showModalTransfer = false; // Close the modal
    }
    
    public function openModal()
    {
        // Reset the filter first to avoid old selections
        $this->filters['to_account'] = null;
        $this->showModal = true; // Open the modal
        
        // Reload available accounts to ensure we have fresh data
        $this->loadAvailableAccounts();
    }

    public function closeModal()
    {
        $this->create_account_id = '';
        $this->create_person_name = '';
        $this->create_transaction_type = 'transfer'; // Reset to default
        $this->create_transaction_amount = '';
        $this->create_transaction_date = '';
        $this->create_transaction_time = '';
        $this->create_transaction_fee = '';
        $this->create_note = '';
        $this->filters['to_account'] = null;
        $this->showModal = false; // Close the modal
    }

    protected $listeners = ['valueSelected'];

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            // It's two words separated by a dot
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
        
        // Log for debugging
        \Log::info('Value selected', [
            'identifier' => $identifier,
            'value' => $selectedValue,
            'to_account' => $this->filters['to_account'] ?? 'not set'
        ]);
    }

    public function saveTransfer()
    {
        $this->create_transaction_amount = str_replace(',', '', $this->create_transaction_amount);

        // to store unique GroupId 
        $transactionGroupId = Str::uuid();

        // Validate the input fields
        $this->validate([
            'filters.to_account' => 'required|exists:accounts,id',
            'create_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount' => 'required|min:0',
            'create_person_name' => 'nullable|string',
            'create_note' => 'nullable|string',
        ]);

        // Retrieve the fromAccount (origin account) and toAccount (destination)
        $fromAccount = Account::findOrFail($this->account->id);
        $toAccount = Account::findOrFail($this->filters['to_account']);

        // Get the current date and time
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        // Create the main transaction record
        $transactionFrom = AccountTransactionModel::create([
            'from_account_id' => $this->account->id,
            'to_account_id' => $this->filters['to_account'],
            'transaction_type' => 'withdrawal',
            'transaction_amount' => $this->create_transaction_amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

         // Create the main transaction record
         $transactionTo = AccountTransactionModel::create([
            'from_account_id' => $this->filters['to_account'],
            'to_account_id' => $this->account->id,
            'transaction_type' => 'deposit',
            'transaction_amount' => $this->create_transaction_amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Flash success message
        $this->alert('success', 'گواستنەوەکە سەرکەوتوو بوو!');

        // Reset input fields and close modal
        $this->resetExcept(['account', 'accountCurrencyId', 'availableAccounts']);
        $this->showModalTransfer = false;
    }

    public function saveCreate()
    {
        $this->create_transaction_amount = str_replace(',', '', $this->create_transaction_amount);

        // to store unique GroupId 
        $transactionGroupId = Str::uuid();

        // Validate the input fields
        $this->validate([
            'filters.to_account' => 'required|exists:accounts,id',
            'create_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount' => 'required|min:0',
            'create_person_name' => 'nullable|string',
            'create_note' => 'nullable|string',
        ]);

        // Retrieve the fromAccount (origin account) and toAccount (destination)
        $fromAccount = Account::findOrFail($this->account->id);
        $toAccount = Account::findOrFail($this->filters['to_account']);

        // Get the current date and time
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        // Create the main transaction record
        $transactionFrom = AccountTransactionModel::create([
            'from_account_id' => $this->account->id,
            'to_account_id' => $this->filters['to_account'],
            'transaction_type' => 'withdrawal',
            'transaction_amount' => $this->create_transaction_amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

         // Create the main transaction record
         $transactionTo = AccountTransactionModel::create([
            'from_account_id' => $this->filters['to_account'],
            'to_account_id' => $this->account->id,
            'transaction_type' => 'deposit',
            'transaction_amount' => $this->create_transaction_amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Flash success message
        $this->alert('success', 'Transaction processed successfully!');

        // Reset input fields and close modal
        $this->resetExcept(['account', 'accountCurrencyId', 'availableAccounts']);
        $this->showModal = false;
    }
 
   public function getTransactionsProperty()
{
    $account = $this->account;
    
    // Start building the query
    $query = AccountTransactionModel::where(function ($query) use ($account) {
        $query->where(function ($q) use ($account) {
            // First condition: from_account_id and status
            $q->where('from_account_id', $account->id)
              ->whereIn('status', ['approved', 'completed']) // Changed from != 'rejected'
              ->where('moneyTransferOnly', '=', 0); // Changed from != 0 to = 0
        })->orWhere(function ($q) use ($account) {
            // Second condition: to_account_id and moneyTransferOnly  
            $q->where('to_account_id', $account->id)
              ->where('moneyTransferOnly', '=', 1); // Changed from != 1 to = 1
        });
    });
    
    // Apply date filters if set
    if ($this->startDate) {
        $query->whereDate('created_at', '>=', $this->startDate);
    }
    
    if ($this->endDate) {
        $query->whereDate('created_at', '<=', $this->endDate);
    }
    
    // Apply sorting
    $query->orderBy('created_at', $this->sortDirection);
    
    // When date filters are applied, return all records without pagination
    if ($this->startDate || $this->endDate) {
        return $query->get();
    } else {
        // Otherwise, paginate according to perPage setting
        if ($this->perPage === 'all') {
            return $query->get();
        } else {
            return $query->paginate($this->perPage);
        }
    }
}

    public function render()
    {
        $account = $this->account;
        $currency = $account->currency->currency_name;
        
        // Debug information
        \Log::info('Rendering ShowCashBox', [
            'account_id' => $account->id,
            'account_type' => $account->account_type,
            'available_accounts' => $this->availableAccounts->count(),
            'selected_to_account' => $this->filters['to_account'] ?? 'none',
            'currency_id' => $this->accountCurrencyId
        ]);
        
        // Use the transactions property to get filtered results
        $transactions = $this->transactions;
        
        // If not paginated (when using 'all' or date filters), we need to perform calculations on the collection
        if ($this->perPage === 'all' || $this->startDate || $this->endDate) {
            // Calculate total deposit
            $totalDeposit = $transactions->where('transaction_type', 'deposit')
                                        ->sum('transaction_amount');
                                        
            // Calculate total withdrawal
            $totalWithdrawal = $transactions->where('transaction_type', 'withdrawal')
                                            ->sum('transaction_amount');
        } else {
            // For paginated results, we need to get all matching records to calculate totals
            $allTransactions = AccountTransactionModel::where('from_account_id', $account->id);
            
            if ($this->startDate) {
                $allTransactions->whereDate('created_at', '>=', $this->startDate);
            }
            
            if ($this->endDate) {
                $allTransactions->whereDate('created_at', '<=', $this->endDate);
            }
            
            $allTransactions = $allTransactions->get();
            
            $totalDeposit = $allTransactions->where('transaction_type', 'deposit')
                                           ->sum('transaction_amount');
                                           
            $totalWithdrawal = $allTransactions->where('transaction_type', 'withdrawal')
                                              ->sum('transaction_amount');
        }
    
        return view('livewire.show-cash-box', compact('transactions', 'account', 'totalDeposit', 'totalWithdrawal', 'currency'));
    }
}