<?php

namespace App\Livewire;

use App\Models\Product;
use App\Models\Supplier;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Facades\Auth;

class ProductList extends Component
{
    use LivewireAlert, WithPagination;

    protected $paginationTheme = 'bootstrap';
    
    public $search = '';
    public $supplierFilter = '';
    public $stockFilter = 'all'; // all, in_stock, low_stock, out_of_stock
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';

    protected $queryString = [
        'search' => ['except' => ''],
        'supplierFilter' => ['except' => ''],
        'stockFilter' => ['except' => 'all'],
    ];

    public function mount()
    {
        // Check for session errors and display them
        if (session()->has('errors') && session('errors')->has('error')) {
            $this->alert('error', session('errors')->first('error'));
        }
        
        if (session('success')) {
            $this->alert('success', session('success'));
        }
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedSupplierFilter()
    {
        $this->resetPage();
    }

    public function updatedStockFilter()
    {
        $this->resetPage();
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function deleteProduct($productId)
    {
        try {
            $product = Product::findOrFail($productId);
            
            // Check for related data
            if ($product->invoiceItems()->count() > 0) {
                $this->alert('error', 'Cannot delete product. This product has been used in invoices.');
                return;
            }

            if ($product->batches()->count() > 0) {
                $this->alert('error', 'Cannot delete product. This product has associated batches.');
                return;
            }

            $product->delete();
            $this->alert('success', 'Product deleted successfully!');
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                $this->alert('error', 'Cannot delete product. This product has related data that prevents deletion.');
            } else {
                $this->alert('error', 'An error occurred while deleting the product. Please try again.');
            }
        } catch (\Exception $e) {
            $this->alert('error', 'An unexpected error occurred. Please try again.');
        }
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->supplierFilter = '';
        $this->stockFilter = 'all';
        $this->sortBy = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function render()
    {
        $query = Product::with('supplier');

        // Search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('name', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%');
            });
        }

        // Supplier filter
        if ($this->supplierFilter) {
            $query->where('supplier_id', $this->supplierFilter);
        }

        // Stock filter
        if ($this->stockFilter === 'in_stock') {
            $query->where('stock_quantity', '>', 10);
        } elseif ($this->stockFilter === 'low_stock') {
            $query->where('stock_quantity', '>', 0)
                  ->where('stock_quantity', '<=', 10);
        } elseif ($this->stockFilter === 'out_of_stock') {
            $query->where('stock_quantity', '<=', 0);
        }

        // Sorting
        $query->orderBy($this->sortBy, $this->sortDirection);

        $suppliers = Supplier::orderBy('name')->get();

        return view('livewire.product-list', [
            'products' => $query->paginate(50),
            'suppliers' => $suppliers,
        ]);
    }
}

