<?php

namespace App\Livewire;

use App\Models\Account;
use App\Models\Currency;
use App\Models\DailyBalanceReport;
use App\Models\Customer;
use Livewire\Component;

class NetBalanceComponent extends Component
{
    public $businessOwnerCashboxIqdBalance = 0;
    public $accountsBalances = [];
    public $digitalCurrencyAccounts = [];
    public $totalBalance = 0;
    public $totalCustomerIQD = 0;
    public $subtotalBeforeCustomerIQD = 0;
    public $currencyBalances = [];

   // Remove these methods
    protected $listeners = [
        'show-loading' => 'showLoading',
        'hide-loading' => 'hideLoading'
    ];
    // Conversion properties
    public $balanceAfterUsdToIqdConversion = 0;
    public $balanceAfterIqdToUsdConversion = 0;

    public $businessOwnerCashboxUsdBalance = 0;
    public $accountsBalancesUSD = [];
    public $totalBalanceUSD = 0;
    public $totalCustomerUSD = 0;
    
    public $showCurrencyModal = false;
    public $availableCurrencies = [];
    public $selectedCurrencies = [];
    public $additionalCurrencyBalances = [];
    public $totalAdditionalCurrencyUsdValue = 0;

    // Currency conversion modals and values
    public $showUsdToIqdModal = false;
    public $showIqdToUsdModal = false;
    public $usdToIqdRate = 0;
    public $iqdToUsdRate = 0;
    public $convertedUsdToIqd = 0;
    public $convertedIqdToUsd = 0;
    
    // Add new properties to track conversions that need to be subtracted
    public $usdConvertedToIqd = 0; // USD amount converted to IQD (needs to be subtracted from USD)
    public $iqdConvertedToUsd = 0; // IQD amount converted to USD (needs to be subtracted from IQD)
    
    // Add new properties to track profit/loss from conversions
    public $profitFromConversion = 0;
    public $profitCurrency = ''; // Will store 'USD' or 'IQD' to indicate which currency the profit is in

    public function mount()
    {
        $this->loadBalances();
        $this->loadAvailableCurrencies();
        $this->loadCurrencyBalances();

        // Load conversion rates and values from session
        $this->usdToIqdRate = session('usdToIqdRate', 0);
        $this->iqdToUsdRate = session('iqdToUsdRate', 0);
        $this->convertedUsdToIqd = session('convertedUsdToIqd', 0);
        $this->convertedIqdToUsd = session('convertedIqdToUsd', 0);
        $this->usdConvertedToIqd = session('usdConvertedToIqd', 0);
        $this->iqdConvertedToUsd = session('iqdConvertedToUsd', 0);
        $this->balanceAfterUsdToIqdConversion = session('balanceAfterUsdToIqdConversion', 0);
        $this->balanceAfterIqdToUsdConversion = session('balanceAfterIqdToUsdConversion', 0);
        $this->profitFromConversion = session('profitFromConversion', 0);
        $this->profitCurrency = session('profitCurrency', '');
    }


    public function showLoading() {}
    public function hideLoading() {}

    
    public function loadBalances()
    {
        $this->resetBalances();
        $this->loadIQDBalances();
        $this->loadUSDBalances();
        $this->loadAdditionalCurrencies();
        
        // Reload conversion values from session after resetting
        $this->usdToIqdRate = session('usdToIqdRate', 0);
        $this->iqdToUsdRate = session('iqdToUsdRate', 0);
        $this->convertedUsdToIqd = session('convertedUsdToIqd', 0);
        $this->convertedIqdToUsd = session('convertedIqdToUsd', 0);
        $this->usdConvertedToIqd = session('usdConvertedToIqd', 0);
        $this->iqdConvertedToUsd = session('iqdConvertedToUsd', 0);
        $this->profitFromConversion = session('profitFromConversion', 0);
        $this->profitCurrency = session('profitCurrency', '');
        
        // Calculate adjusted balances after conversions
        $this->calculateBalancesAfterConversion();
    }
    public function saveDailyReport()
    {
        $existingReport = DailyBalanceReport::whereDate('report_date', today())->first();
        
        if ($existingReport) {
            $this->dispatch('alert', type: 'warning', message: 'Report already exists for today!');
            return;
        }

        // First dispatch an event to capture screenshot
        $this->dispatch('captureScreenshot');
        
        // This will be handled by JavaScript which will then call the saveReportWithImage method
    }

    // Add this new method to handle the screenshot data
    public function saveReportWithImage($imageData)
    {
        // Calculate grand total in USD
        $grandTotalUsd = ($this->totalBalanceUSD + $this->totalAdditionalCurrencyUsdValue + $this->convertedIqdToUsd) - $this->usdConvertedToIqd;

        // Create the report with the image data
        DailyBalanceReport::create([
            'report_date' => now(),
            'total_iqd' => $this->totalBalance,
            'total_usd' => $this->totalBalanceUSD,
            'total_euro' => 0, // You might want to add Euro support later
            'euro_to_usd_rate' => 0, // Add if you implement Euro
            'converted_euro_to_usd' => 0, // Add if you implement Euro
            'additional_currencies' => json_encode($this->additionalCurrencyBalances),
            'total_additional_usd' => $this->totalAdditionalCurrencyUsdValue,
            'grand_total_usd' => $grandTotalUsd,
            'profit_amount' => $this->profitFromConversion,
            'profit_currency' => $this->profitCurrency,
            'usd_to_iqd_rate' => $this->usdToIqdRate,
            'iqd_to_usd_rate' => $this->iqdToUsdRate,
            'converted_usd_to_iqd' => $this->convertedUsdToIqd,
            'converted_iqd_to_usd' => $this->convertedIqdToUsd,
            'image' => $imageData, // Store the image data in the new image column
        ]);

        $this->dispatch('alert', type: 'success', message: 'Daily report saved successfully!');
    }

    public function loadCurrencyBalances()
    {
        $this->currencyBalances = [];
        
        foreach ($this->availableCurrencies as $currency) {
            $cashboxAccount = Account::where('account_title', 'Business Owner Cashbox')
                ->where('currency_id', $currency->id)
                ->where('account_type', 'savings')
                ->first();
                
            $balance = $cashboxAccount ? $cashboxAccount->calculateBalance() : 0;
            
            if ($balance != 0) {
                $this->currencyBalances[$currency->id] = $balance;
            }
        }
    }

    private function resetBalances()
    {
        $this->businessOwnerCashboxIqdBalance = 0;
        $this->accountsBalances = [];
        $this->digitalCurrencyAccounts = [];
        $this->totalBalance = 0;
        $this->totalCustomerIQD = 0;
        $this->subtotalBeforeCustomerIQD = 0;
        $this->businessOwnerCashboxUsdBalance = 0;
        $this->accountsBalancesUSD = [];
        $this->totalBalanceUSD = 0;
        $this->totalCustomerUSD = 0;
        $this->totalAdditionalCurrencyUsdValue = 0;
        $this->additionalCurrencyBalances = [];
        $this->balanceAfterUsdToIqdConversion = 0;
        $this->balanceAfterIqdToUsdConversion = 0;
        $this->profitFromConversion = 0;
        $this->profitCurrency = '';
    }

    // Add a method to clear all session conversion data
    public function clearConversionData()
    {
        session()->forget(['usdToIqdRate', 'iqdToUsdRate', 'convertedUsdToIqd', 'convertedIqdToUsd', 
                          'usdConvertedToIqd', 'iqdConvertedToUsd', 'balanceAfterUsdToIqdConversion', 
                          'balanceAfterIqdToUsdConversion', 'profitFromConversion', 'profitCurrency']);
        
        $this->usdToIqdRate = 0;
        $this->iqdToUsdRate = 0;
        $this->convertedUsdToIqd = 0;
        $this->convertedIqdToUsd = 0;
        $this->usdConvertedToIqd = 0;
        $this->iqdConvertedToUsd = 0;
        $this->balanceAfterUsdToIqdConversion = 0;
        $this->balanceAfterIqdToUsdConversion = 0;
        $this->profitFromConversion = 0;
        $this->profitCurrency = '';
        
        $this->loadBalances();
        $this->dispatch('alert', type: 'success', message: 'Conversion data cleared successfully!');
    }

    private function loadIQDBalances()
    {
        $iqdCurrency = Currency::where('currency_name', 'IQD')->first();
        if ($iqdCurrency) {
            $cashboxIqdAccount = Account::where('account_title', 'Business Owner Cashbox')
                ->where('currency_id', $iqdCurrency->id)
                ->where('account_type', 'savings')
                ->first();
                
            if ($cashboxIqdAccount) {
                $this->businessOwnerCashboxIqdBalance = $cashboxIqdAccount->calculateBalance();
            }
        }
        
        $this->calculateTotalCustomerIQD();
        
        $this->digitalCurrencyAccounts = Account::whereHas('currency', function ($query) {
            $query->where('type', 1);
        })
        ->where('customer_id', 1)
        ->where('account_type', 'savings')
        ->with(['currency', 'customer'])
        ->get();
        
        $this->subtotalBeforeCustomerIQD = $this->businessOwnerCashboxIqdBalance;
        
        if ($this->businessOwnerCashboxIqdBalance != 0) {
            $this->accountsBalances[] = [
                'title' => 'قاسەی خاوەن کار (IQD)',
                'currency_name' => 'IQD',
                'balance' => $this->businessOwnerCashboxIqdBalance,
            ];
        }
        
        foreach ($this->digitalCurrencyAccounts as $account) {
            $balance = $account->calculateBalance();
            $currencyName = $account->currency->currency_name ?? 'Unknown';
            
            $title = "({$account->id}) - ({$currencyName})";
            
            if ($balance != 0) {
                $this->accountsBalances[] = [
                    'title' => $title,
                    'currency_name' => $currencyName,
                    'balance' => $balance,
                ];
                
                $this->subtotalBeforeCustomerIQD += $balance;
            }
        }
        
        if ($this->totalCustomerIQD >= 0) {
            $this->totalBalance = $this->subtotalBeforeCustomerIQD - abs($this->totalCustomerIQD);
        } else {
            $this->totalBalance = $this->subtotalBeforeCustomerIQD + abs($this->totalCustomerIQD);
        }
        
        if ($this->totalCustomerIQD != 0) {
            $this->accountsBalances[] = [
                'title' => 'هاتوو و چوو (IQD)',
                'currency_name' => 'IQD',
                'balance' => $this->totalCustomerIQD,
                'is_customer_total' => true,
            ];
        }
    }
    
    private function loadUSDBalances()
    {
        $usdCurrency = Currency::where('currency_name', 'USD')->first();
        if ($usdCurrency) {
            $cashboxUsdAccount = Account::where('account_title', 'Business Owner Cashbox')
                ->where('currency_id', $usdCurrency->id)
                ->where('account_type', 'savings')
                ->first();
            if ($cashboxUsdAccount) {
                $this->businessOwnerCashboxUsdBalance = $cashboxUsdAccount->calculateBalance();
            }
        }
        
        $this->calculateTotalCustomerUSD();
        
        $totalBalanceUSD = $this->businessOwnerCashboxUsdBalance - $this->totalCustomerUSD;
        
        if ($this->businessOwnerCashboxUsdBalance != 0) {
            $this->accountsBalancesUSD[] = [
                'title' => 'قاسەی خاوەن کار (USD)',
                'currency_name' => 'USD',
                'balance' => $this->businessOwnerCashboxUsdBalance,
            ];
        }
        
        if ($this->totalCustomerUSD != 0) {
            $this->accountsBalancesUSD[] = [
                'title' => 'هاتوو و چوو (USD)',
                'currency_name' => 'USD',
                'balance' => $this->totalCustomerUSD,
                'is_customer_total' => true,
            ];
        }
        
        $this->totalBalanceUSD = $totalBalanceUSD;
    }
    
    private function calculateTotalCustomerIQD()
    {
        $customersIQD = Customer::whereNotNull('customer_name')
            ->whereNotIn('customer_name', ['Cashiers', 'Supervisor', 'Bank (Cashboxes)'])
            ->whereHas('accounts.currency', function ($query) {
                $query->where('currency_name', 'IQD');
            })->with(['accounts' => function ($query) {
                $query->whereHas('currency', function ($q) {
                    $q->where('currency_name', 'IQD');
                })->with(['currency', 'transactions']);
            }])->get();
        
        $totalLamanaIQD = 0;
        $totalLatanaIQD = 0;
        
        foreach ($customersIQD as $customer) {
            $balanceIQD = 0;
            
            foreach ($customer->accounts as $account) {
                if ($account->currency->currency_name === 'IQD') {
                    foreach ($account->transactions as $transaction) {
                        if ($transaction->transaction_type === 'deposit') {
                            $balanceIQD += $transaction->transaction_amount;
                        } elseif ($transaction->transaction_type === 'withdrawal') {
                            $balanceIQD -= $transaction->transaction_amount;
                        }
                    }
                }
            }
            
            if ($balanceIQD > 0) {
                $totalLamanaIQD += $balanceIQD;
            } elseif ($balanceIQD < 0) {
                $totalLatanaIQD += abs($balanceIQD);
            }
        }
        
        $this->totalCustomerIQD = $totalLamanaIQD - $totalLatanaIQD;
    }
    
    private function calculateTotalCustomerUSD()
    {
        $customersUSD = Customer::whereNotNull('customer_name')
            ->whereNotIn('customer_name', ['Cashiers', 'Supervisor', 'Bank (Cashboxes)'])
            ->whereHas('accounts.currency', function ($query) {
                $query->where('currency_name', 'USD');
            })->with(['accounts' => function ($query) {
                $query->whereHas('currency', function ($q) {
                    $q->where('currency_name', 'USD');
                })->with(['currency', 'transactions']);
            }])->get();
        
        $totalLamanaUSD = 0;
        $totalLatanaUSD = 0;
        
        foreach ($customersUSD as $customer) {
            $balanceUSD = 0;
            
            foreach ($customer->accounts as $account) {
                if ($account->currency->currency_name === 'USD') {
                    foreach ($account->transactions as $transaction) {
                        if ($transaction->transaction_type === 'deposit') {
                            $balanceUSD += $transaction->transaction_amount;
                        } elseif ($transaction->transaction_type === 'withdrawal') {
                            $balanceUSD -= $transaction->transaction_amount;
                        }
                    }
                }
            }
            
            if ($balanceUSD > 0) {
                $totalLamanaUSD += $balanceUSD;
            } elseif ($balanceUSD < 0) {
                $totalLatanaUSD += abs($balanceUSD);
            }
        }
        
        $this->totalCustomerUSD = $totalLamanaUSD - $totalLatanaUSD;
    }
    
    private function loadAvailableCurrencies()
    {
        $this->availableCurrencies = Currency::where('type', 0)
            ->whereNotIn('currency_name', ['IQD', 'USD'])
            ->get();
    }
    
    public function openCurrencyModal()
    {
        $this->showCurrencyModal = true;
    }
    
    public function closeCurrencyModal()
    {
        $this->showCurrencyModal = false;
    }
    
    public function toggleCurrency($currencyId)
    {
        if (isset($this->selectedCurrencies[$currencyId])) {
            unset($this->selectedCurrencies[$currencyId]);
        } else {
            $this->selectedCurrencies[$currencyId] = [
                'rate' => 0
            ];
        }
    }
    
    public function updateCurrencyRate($currencyId, $rate)
    {
        if (isset($this->selectedCurrencies[$currencyId])) {
            $this->selectedCurrencies[$currencyId]['rate'] = $rate;
            $this->loadAdditionalCurrencies();
        }
    }
    
    public function loadAdditionalCurrencies()
    {
        $this->additionalCurrencyBalances = [];
        $this->totalAdditionalCurrencyUsdValue = 0;
        
        if (empty($this->selectedCurrencies)) {
            return;
        }
        
        foreach ($this->selectedCurrencies as $currencyId => $data) {
            $currency = Currency::find($currencyId);
            if (!$currency) continue;
            
            $cashboxAccount = Account::where('account_title', 'Business Owner Cashbox')
                ->where('currency_id', $currency->id)
                ->where('account_type', 'savings')
                ->first();
            
            $balance = 0;
            if ($cashboxAccount) {
                $balance = $cashboxAccount->calculateBalance();
            }
            
            $convertedValue = 0;
            if (isset($data['rate']) && is_numeric($data['rate']) && $data['rate'] > 0) {
                $convertedValue = $balance * $data['rate'];
                $this->totalAdditionalCurrencyUsdValue += $convertedValue;
            }
            
            $this->additionalCurrencyBalances[] = [
                'id' => $currency->id,
                'currency_name' => $currency->currency_name,
                'title' => 'قاسەی خاوەن کار (' . $currency->currency_name . ')',
                'balance' => $balance,
                'rate' => $data['rate'] ?? 0,
                'converted_value' => $convertedValue
            ];
        }
        
       // $this->showCurrencyModal = false;
    }

    // New methods for currency conversion
    public function openUsdToIqdModal()
    {
        $this->showUsdToIqdModal = true;
        $this->usdToIqdRate = session('usdToIqdRate', 0);
    }

    public function closeUsdToIqdModal()
    {
        $this->showUsdToIqdModal = false;
    }

    public function openIqdToUsdModal()
    {
        $this->showIqdToUsdModal = true;
        $this->iqdToUsdRate = session('iqdToUsdRate', 0);
    }

    public function closeIqdToUsdModal()
    {
        $this->showIqdToUsdModal = false;
    }

    // Improved conversion methods with profit calculation
    public function convertUsdToIqd()
    {
        if (is_numeric($this->usdToIqdRate) && $this->usdToIqdRate > 0) {
            $totalUsd = $this->totalBalanceUSD + $this->totalAdditionalCurrencyUsdValue;
            $this->convertedUsdToIqd = $totalUsd * $this->usdToIqdRate;
            $this->usdConvertedToIqd = $totalUsd; // Track how much USD was converted to IQD
            
            // Calculate profit - If total IQD is negative, we're converting USD to IQD to cover the deficit
            if ($this->totalBalance < 0) {
                // Calculate how much of the converted IQD is used to cover the deficit
                $iqdDeficit = abs($this->totalBalance);
                
                // If the converted amount exceeds the deficit, we have a profit in IQD
                if ($this->convertedUsdToIqd > $iqdDeficit) {
                    $this->profitFromConversion = $this->convertedUsdToIqd - $iqdDeficit;
                    $this->profitCurrency = 'IQD';
                } else {
                    // If the converted amount doesn't cover the deficit, there's no profit
                    $this->profitFromConversion = 0;
                    $this->profitCurrency = '';
                }
            } else {
                // If IQD balance is positive, there's no immediate profit from this conversion operation
                $this->profitFromConversion = 0;
                $this->profitCurrency = '';
            }
            
            // Store in session
            session([
                'convertedUsdToIqd' => $this->convertedUsdToIqd,
                'usdToIqdRate' => $this->usdToIqdRate,
                'usdConvertedToIqd' => $this->usdConvertedToIqd,
                'profitFromConversion' => $this->profitFromConversion,
                'profitCurrency' => $this->profitCurrency,
            ]);
            
            $this->calculateBalancesAfterConversion();
            $this->closeUsdToIqdModal();
            $this->dispatch('alert', type: 'success', message: 'Conversion saved successfully!');
        } else {
            $this->dispatch('alert', type: 'warning', message: 'Please enter a valid conversion rate!');
        }
    }

    public function convertIqdToUsd()
{
    if (is_numeric($this->iqdToUsdRate) && $this->iqdToUsdRate > 0) {
        $this->iqdConvertedToUsd = $this->totalBalance; // Track how much IQD was converted to USD
        $this->convertedIqdToUsd = $this->totalBalance / $this->iqdToUsdRate;
        
        // Calculate the total USD balance before conversion
        $totalUsdBeforeConversion = $this->totalBalanceUSD + $this->totalAdditionalCurrencyUsdValue;
        
        // Calculate the new total after conversion
        $totalUsdAfterConversion = $totalUsdBeforeConversion + $this->convertedIqdToUsd;
        
        // The profit is the difference between the converted amount and the deficit
        if ($totalUsdBeforeConversion < 0) {
            // If there was a deficit, profit is how much we covered beyond the deficit
            $this->profitFromConversion = $this->convertedIqdToUsd - abs($totalUsdBeforeConversion);
        } else {
            // If no deficit, the entire converted amount is profit
            $this->profitFromConversion = $this->convertedIqdToUsd;
        }
        
        // Only show profit if it's positive
        if ($this->profitFromConversion > 0) {
            $this->profitCurrency = 'USD';
        } else {
            $this->profitFromConversion = 0;
            $this->profitCurrency = '';
        }
        
        // Store in session
        session([
            'convertedIqdToUsd' => $this->convertedIqdToUsd,
            'iqdToUsdRate' => $this->iqdToUsdRate,
            'iqdConvertedToUsd' => $this->iqdConvertedToUsd,
            'profitFromConversion' => $this->profitFromConversion,
            'profitCurrency' => $this->profitCurrency,
        ]);
        
        $this->calculateBalancesAfterConversion();
        $this->closeIqdToUsdModal();
        $this->dispatch('alert', type: 'success', message: 'Conversion saved successfully!');
    } else {
        $this->dispatch('alert', type: 'warning', message: 'Please enter a valid conversion rate!');
    }
}
    
    // Calculate balances after applying conversions
    private function calculateBalancesAfterConversion()
{
    // Calculate adjustments for IQD balance
    $adjustedIqdBalance = $this->totalBalance;
    if ($this->convertedUsdToIqd > 0) {
        $adjustedIqdBalance += $this->convertedUsdToIqd;
    }
    if ($this->iqdConvertedToUsd > 0) {
        $adjustedIqdBalance -= $this->iqdConvertedToUsd;
    }
    $this->balanceAfterUsdToIqdConversion = $adjustedIqdBalance;
    
    // Calculate adjustments for USD balance
    $adjustedUsdBalance = $this->totalBalanceUSD + $this->totalAdditionalCurrencyUsdValue;
    if ($this->usdConvertedToIqd > 0) {
        $adjustedUsdBalance -= $this->usdConvertedToIqd;
    }
    if ($this->convertedIqdToUsd > 0) {
        $adjustedUsdBalance += $this->convertedIqdToUsd;
    }
    $this->balanceAfterIqdToUsdConversion = $adjustedUsdBalance;
    
    // Store adjusted balances in session
    session([
        'balanceAfterUsdToIqdConversion' => $this->balanceAfterUsdToIqdConversion,
        'balanceAfterIqdToUsdConversion' => $this->balanceAfterIqdToUsdConversion,
    ]);
}
    
    public function render()
    {
        $todayReportExists = DailyBalanceReport::whereDate('report_date', today())->exists();
    
        return view('livewire.net-balance-component', [
            'accountsBalances' => $this->accountsBalances,
            'totalBalance' => $this->totalBalance,
            'subtotalBeforeCustomerIQD' => $this->subtotalBeforeCustomerIQD,
            'accountsBalancesUSD' => $this->accountsBalancesUSD,
            'totalBalanceUSD' => $this->totalBalanceUSD,
            'totalCustomerIQD' => $this->totalCustomerIQD,
            'subtotalBeforeCustomerUSD' => $this->businessOwnerCashboxUsdBalance,
            'availableCurrencies' => $this->availableCurrencies,
            'additionalCurrencyBalances' => $this->additionalCurrencyBalances,
            'totalAdditionalCurrencyUsdValue' => $this->totalAdditionalCurrencyUsdValue,
            'todayReportExists' => $todayReportExists,
            'convertedUsdToIqd' => $this->convertedUsdToIqd,
            'convertedIqdToUsd' => $this->convertedIqdToUsd,
            'balanceAfterUsdToIqdConversion' => $this->balanceAfterUsdToIqdConversion,
            'balanceAfterIqdToUsdConversion' => $this->balanceAfterIqdToUsdConversion,
            'usdConvertedToIqd' => $this->usdConvertedToIqd,
            'iqdConvertedToUsd' => $this->iqdConvertedToUsd,
            'profitFromConversion' => $this->profitFromConversion,
            'profitCurrency' => $this->profitCurrency,
        ]);
    }
}