<?php

namespace App\Livewire;

use App\Models\Invoice;
use App\Models\Patient;
use App\Models\User;
use Livewire\Component;
use Livewire\WithPagination;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use App\Traits\HasUtf8Encoding;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class InvoiceReport extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';

    // Filters
    public $search = '';
    public $startDate = '';
    public $endDate = '';
    public $status = '';
    public $patient_id = '';
    public $user_id = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';

    public function mount()
    {
        // Set default date range to last 30 days
        $this->endDate = now()->format('Y-m-d');
        $this->startDate = now()->subDays(30)->format('Y-m-d');
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedStartDate()
    {
        $this->resetPage();
    }

    public function updatedEndDate()
    {
        $this->resetPage();
    }

    public function updatedStatus()
    {
        $this->resetPage();
    }

    public function updatedPatientId()
    {
        $this->resetPage();
    }

    public function updatedUserId()
    {
        $this->resetPage();
    }

    public function sortByField($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function resetFilters()
    {
        $this->search = '';
        $this->startDate = now()->subDays(30)->format('Y-m-d');
        $this->endDate = now()->format('Y-m-d');
        $this->status = '';
        $this->patient_id = '';
        $this->user_id = '';
        $this->sortBy = 'created_at';
        $this->sortDirection = 'desc';
        $this->resetPage();
    }

    public function getAnalysisData()
    {
        $query = $this->getBaseQuery();

        $allInvoices = Invoice::with('items')->get();
        $filteredInvoices = $query->with('items')->get();

        // Calculate totals
        $totalRevenue = $allInvoices->where('status', 'completed')->sum('total_amount');
        $filteredRevenue = $filteredInvoices->where('status', 'completed')->sum('total_amount');

        // Status breakdown
        $statusBreakdown = Invoice::select('status', DB::raw('count(*) as count'), DB::raw('sum(total_amount) as total'))
            ->groupBy('status')
            ->get()
            ->pluck('total', 'status')
            ->toArray();

        $filteredStatusBreakdown = $query->select('status', DB::raw('count(*) as count'), DB::raw('sum(total_amount) as total'))
            ->groupBy('status')
            ->get()
            ->pluck('total', 'status')
            ->toArray();

        // Average invoice value
        $avgInvoiceValue = $allInvoices->where('status', 'completed')->avg('total_amount');
        $filteredAvgInvoiceValue = $filteredInvoices->where('status', 'completed')->avg('total_amount');

        // Today's stats
        $todayInvoices = Invoice::whereDate('invoice_date', today())->get();
        $todayRevenue = $todayInvoices->where('status', 'completed')->sum('total_amount');

        // This month's stats
        $monthInvoices = Invoice::whereMonth('invoice_date', now()->month)
            ->whereYear('invoice_date', now()->year)
            ->get();
        $monthRevenue = $monthInvoices->where('status', 'completed')->sum('total_amount');

        return [
            'total_invoices' => Invoice::count(),
            'filtered_invoices' => $query->count(),
            'total_revenue' => $totalRevenue,
            'filtered_revenue' => $filteredRevenue,
            'total_items_sold' => $allInvoices->sum(function($invoice) {
                return $invoice->items->sum('quantity');
            }),
            'filtered_items_sold' => $filteredInvoices->sum(function($invoice) {
                return $invoice->items->sum('quantity');
            }),
            'avg_invoice_value' => $avgInvoiceValue ?? 0,
            'filtered_avg_invoice_value' => $filteredAvgInvoiceValue ?? 0,
            'today_invoices' => $todayInvoices->count(),
            'today_revenue' => $todayRevenue,
            'month_invoices' => $monthInvoices->count(),
            'month_revenue' => $monthRevenue,
            'status_breakdown' => $statusBreakdown,
            'filtered_status_breakdown' => $filteredStatusBreakdown,
        ];
    }

    private function getBaseQuery()
    {
        $query = Invoice::query();

        // Search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('invoice_number', 'like', '%' . $this->search . '%')
                  ->orWhere('notes', 'like', '%' . $this->search . '%')
                  ->orWhereHas('patient', function($patientQuery) {
                      $patientQuery->where('name', 'like', '%' . $this->search . '%')
                                   ->orWhere('code', 'like', '%' . $this->search . '%');
                  });
            });
        }

        // Date filter
        if ($this->startDate) {
            $query->whereDate('invoice_date', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('invoice_date', '<=', $this->endDate);
        }

        // Status filter
        if ($this->status) {
            $query->where('status', $this->status);
        }

        // Patient filter
        if ($this->patient_id) {
            $query->where('patient_id', $this->patient_id);
        }

        // User filter
        if ($this->user_id) {
            $query->where('user_id', $this->user_id);
        }

        return $query;
    }

    public function render()
    {
        $query = $this->getBaseQuery();

        // Add relationships
        $query->with(['patient', 'user'])->withCount('items');

        // Sorting
        $query->orderBy($this->sortBy, $this->sortDirection);

        $invoices = $query->paginate(25);

        $analysis = $this->getAnalysisData();
        $patients = Patient::orderBy('name')->get();
        $users = User::orderBy('name')->get();

        return view('livewire.invoice-report', [
            'invoices' => $invoices,
            'analysis' => $analysis,
            'patients' => $patients,
            'users' => $users,
        ]);
    }
}

