<?php

namespace App\Livewire;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Patient;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Product;
use App\Models\ProductBatch;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use App\Traits\HasUtf8Encoding;

class InvoiceManagement extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';
    
    public $searchQuery = '';
    public $showCreateModal = false;
    public $showEditModal = false;
    public $showAddItemModal = false;
    
    // Invoice form properties
    public $invoice_id, $patient_id, $invoice_date, $notes;
    
    // Invoice item form properties
    public $item_category_id, $item_subcategory_id, $item_product_id, $item_product_batch_id;
    public $item_quantity = 1, $item_unit_price = 0, $item_notes;
    public $item_type = 'service'; // 'service' or 'drug'
    
    public $patients = [];
    public $categories = [];
    public $subcategories = [];
    public $products = [];
    public $productBatches = [];
    
    public $currentInvoice = null;
    public $invoiceItems = [];

    public function mount()
    {
        $this->patients = Patient::all();
        $this->categories = Category::with('subcategories')->get();
        $this->invoice_date = now()->format('Y-m-d');
    }

    public function openCreateModal()
    {
        $this->resetInvoiceForm();
        $this->showCreateModal = true;
    }

    public function closeCreateModal()
    {
        $this->resetInvoiceForm();
        $this->showCreateModal = false;
    }

    public function openEditModal($invoiceId)
    {
        $this->currentInvoice = Invoice::with(['items.subcategory', 'items.product', 'items.productBatch'])->findOrFail($invoiceId);
        $this->invoice_id = $this->currentInvoice->id;
        $this->patient_id = $this->currentInvoice->patient_id;
        $this->invoice_date = $this->currentInvoice->invoice_date->format('Y-m-d');
        $this->notes = $this->currentInvoice->notes;
        $this->invoiceItems = $this->currentInvoice->items->toArray();
        $this->showEditModal = true;
    }

    public function closeEditModal()
    {
        $this->resetInvoiceForm();
        $this->currentInvoice = null;
        $this->invoiceItems = [];
        $this->showEditModal = false;
    }

    public function openAddItemModal()
    {
        $this->resetItemForm();
        $this->showAddItemModal = true;
    }

    public function closeAddItemModal()
    {
        $this->resetItemForm();
        $this->showAddItemModal = false;
    }

    public function updatedItemCategoryId($value)
    {
        if ($value) {
            $category = Category::find($value);
            if ($category && $category->type === 'drug') {
                $this->item_type = 'drug';
                $this->products = Product::whereHas('supplier')->get();
            } else {
                $this->item_type = 'service';
                $this->subcategories = Subcategory::where('category_id', $value)->get();
            }
        }
    }

    public function updatedItemProductId($value)
    {
        if ($value) {
            $product = Product::find($value);
            if ($product) {
                $this->item_unit_price = $product->sale_price;
                $this->productBatches = ProductBatch::where('product_id', $value)
                    ->whereRaw('(quantity - sold_quantity + returned_quantity) > 0')
                    ->get();
            }
        }
    }

    public function updatedItemSubcategoryId($value)
    {
        if ($value) {
            $subcategory = Subcategory::find($value);
            if ($subcategory) {
                $this->item_unit_price = $subcategory->cost;
            }
        }
    }

    public function saveInvoice()
    {
        $this->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_date' => 'required|date',
        ]);

        try {
            $invoice = Invoice::create([
                'invoice_number' => Invoice::generateInvoiceNumber(),
                'patient_id' => $this->patient_id,
                'clinic_id' => null,
                'invoice_date' => $this->invoice_date,
                'notes' => $this->notes,
                'user_id' => Auth::id(),
                'status' => 'completed', // Always set to completed
                'total_amount' => 0,
            ]);

            $this->alert('success', 'Invoice created successfully!');
            $this->closeCreateModal();
            $this->openEditModal($invoice->id);
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to create invoice: ' . $e->getMessage());
        }
    }

    public function addInvoiceItem()
    {
        if (!$this->currentInvoice) {
            $this->alert('error', 'Please create or select an invoice first');
            return;
        }

        if ($this->item_type === 'service') {
            $this->validate([
                'item_subcategory_id' => 'required|exists:subcategories,id',
                'item_quantity' => 'required|numeric|min:1',
                'item_unit_price' => 'required|numeric|min:0',
            ]);
        } else {
            $this->validate([
                'item_product_id' => 'required|exists:products,id',
                'item_product_batch_id' => 'required|exists:product_batches,id',
                'item_quantity' => 'required|numeric|min:1',
                'item_unit_price' => 'required|numeric|min:0',
            ]);
        }

        try {
            $item = InvoiceItem::create([
                'invoice_id' => $this->currentInvoice->id,
                'subcategory_id' => $this->item_type === 'service' ? $this->item_subcategory_id : null,
                'product_id' => $this->item_type === 'drug' ? $this->item_product_id : null,
                'product_batch_id' => $this->item_type === 'drug' ? $this->item_product_batch_id : null,
                'quantity' => $this->item_quantity,
                'unit_price' => $this->item_unit_price,
                'total_price' => $this->item_quantity * $this->item_unit_price,
                'notes' => $this->item_notes,
            ]);

            // Update product batch if it's a drug
            if ($this->item_type === 'drug' && $this->item_product_batch_id) {
                $batch = ProductBatch::find($this->item_product_batch_id);
                if ($batch) {
                    $batch->sold_quantity += $this->item_quantity;
                    $batch->save();
                    
                    // Update product stock
                    $product = Product::find($this->item_product_id);
                    if ($product) {
                        $product->stock_quantity = max(0, $product->stock_quantity - $this->item_quantity);
                        $product->save();
                    }
                }
            }

            $this->currentInvoice->calculateTotal();
            $this->openEditModal($this->currentInvoice->id);
            $this->closeAddItemModal();
            $this->alert('success', 'Item added successfully!');
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to add item: ' . $e->getMessage());
        }
    }

    public function removeInvoiceItem($itemId)
    {
        try {
            $item = InvoiceItem::findOrFail($itemId);
            
            // Restore product batch quantity if it's a drug
            if ($item->product_batch_id) {
                $batch = ProductBatch::find($item->product_batch_id);
                if ($batch) {
                    $batch->sold_quantity = max(0, $batch->sold_quantity - $item->quantity);
                    $batch->save();
                }
                
                if ($item->product_id) {
                    $product = Product::find($item->product_id);
                    if ($product) {
                        $product->stock_quantity += $item->quantity;
                        $product->save();
                    }
                }
            }
            
            $invoice = $item->invoice;
            $item->delete();
            $invoice->calculateTotal();
            
            $this->openEditModal($invoice->id);
            $this->alert('success', 'Item removed successfully!');
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to remove item: ' . $e->getMessage());
        }
    }

    public function updateInvoice()
    {
        $this->validate([
            'patient_id' => 'required|exists:patients,id',
            'invoice_date' => 'required|date',
        ]);

        try {
            $invoice = Invoice::findOrFail($this->invoice_id);
            $invoice->update([
                'patient_id' => $this->patient_id,
                'clinic_id' => null,
                'invoice_date' => $this->invoice_date,
                'notes' => $this->notes,
                'status' => 'completed', // Always set to completed
            ]);
            $invoice->calculateTotal();

            $this->alert('success', 'Invoice updated successfully!');
            $this->closeEditModal();
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to update invoice: ' . $e->getMessage());
        }
    }

    private function resetInvoiceForm()
    {
        $this->invoice_id = null;
        $this->patient_id = '';
        $this->invoice_date = now()->format('Y-m-d');
        $this->notes = '';
    }

    private function resetItemForm()
    {
        $this->item_category_id = '';
        $this->item_subcategory_id = '';
        $this->item_product_id = '';
        $this->item_product_batch_id = '';
        $this->item_quantity = 1;
        $this->item_unit_price = 0;
        $this->item_notes = '';
        $this->item_type = 'service';
        $this->subcategories = [];
        $this->products = [];
        $this->productBatches = [];
    }

    public function render()
    {
        $query = Invoice::with(['patient', 'user']);

        if ($this->searchQuery) {
            $query->where(function($q) {
                $q->where('invoice_number', 'like', '%' . $this->searchQuery . '%')
                  ->orWhereHas('patient', function($q2) {
                      $q2->where('name', 'like', '%' . $this->searchQuery . '%')
                         ->orWhere('code', 'like', '%' . $this->searchQuery . '%');
                  });
            });
        }

        if ($this->currentInvoice) {
            $this->currentInvoice->refresh();
            $this->currentInvoice->load(['items.subcategory', 'items.product', 'items.productBatch']);
            $this->invoiceItems = $this->currentInvoice->items->toArray();
        }

        return view('livewire.invoice-management', [
            'invoices' => $query->paginate(50),
        ]);
    }
}
