<?php
namespace App\Livewire;

use App\Models\Customer;
use App\Models\User;
use App\Models\Account;
use App\Models\Currency;
use App\Models\AccountTransaction;
use Livewire\Component;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use App\Traits\HasUtf8Encoding;
use App\Helpers\KurdishTextHelper;
use Carbon\Carbon;
use Illuminate\Support\Str;

class CustomerManagement extends Component
{
    use LivewireAlert, WithPagination, HasUtf8Encoding;

    protected $paginationTheme = 'bootstrap';
    public $searchQuery = ''; // Property for search input

    // New properties for multiple currency selection
    public $transaction_amounts = []; // To store transaction amounts for each currency
    public $transaction_types = []; // To store transaction types for each currency
    public $transactionTypes = ['deposit', 'withdrawal']; // Available transaction types

    public $customer_name, $phone_number, $second_phone, $address, $note;
    public $account_balance, $showCreateModal, $showEditModal, $showCreateAccountModal;
    public $currency_id; // Use this for single currency (for create account modal)
    public $currency_ids = []; // Use this for multiple currency selection
    public $currencies = []; // Initialize as an array
    // public $customers = []; // Initialize as an array

    // Modal state
    public $showCustomerModal = false;
    public $showAccountModal = false; 
    public $selected_customer_id; // Store selected customer ID for new account

    public function mount()
    {
        $this->currencies = Currency::all();
        // $this->customers = Customer::with('accounts')->paginate(10); 
    }

    /**
     * Ensure proper UTF-8 encoding when customer_name is updated
     */
    public function updatedCustomerName($value)
    {
        if (is_string($value)) {
            $this->customer_name = $this->ensureUtf8Encoding($value);
        }
    }

    /**
     * Ensure proper UTF-8 encoding when address is updated
     */
    public function updatedAddress($value)
    {
        if (is_string($value)) {
            $this->address = $this->ensureUtf8Encoding($value);
        }
    }

    /**
     * Ensure proper UTF-8 encoding when note is updated
     */
    public function updatedNote($value)
    {
        if (is_string($value)) {
            $this->note = $this->ensureUtf8Encoding($value);
        }
    }


    public function openCreateModal()
    {
        $this->showCreateModal = true; // Open the modal
    }

    public function closeCreateModal()
    {
        $this->customer_name = '';
        $this->phone_number = '';
        $this->second_phone = '';
        $this->address = '';
        $this->note = '';
        $this->currency_ids = [];
        $this->transaction_amounts = [];
        $this->transaction_types = [];
        $this->account_balance = '';
        $this->showCreateModal = false; // Close the modal
    }

    public function openCreateAccountModal()
    {
        $this->showCreateAccountModal = true; // Open the modal
    }

    public function closeCreateAccountModal()
    {
        $this->currency_id = '';
        $this->account_balance = '';
        $this->showCreateAccountModal = false; // Close the modal
    }


    public function openEditModal()
    {
        $this->showEditModal = true; // Open the modal
    }

    public function closeEditModal()
    {
        $this->edit_customer_name = '';
        $this->edit_phone_number = '';
        $this->edit_second_phone = '';
        $this->edit_address = '';
        $this->edit_note = '';
        $this->editCustomerId = null;
        
        $this->showEditModal = false; // Close the modal
    }


    public function saveCustomer()
    {
        $this->validate([
            'customer_name' => 'required|string|max:255',
            'phone_number' => 'required|string|max:15',
            'second_phone' => 'nullable|string|max:15',
            'address' => 'required|string|max:255',
            'note' => 'nullable|string|max:255',
            'currency_ids' => 'required|array|min:1',
            'currency_ids.*' => 'exists:currencies,id',
            'transaction_amounts' => 'required|array',
            'transaction_types' => 'required|array',
        ], [
            'currency_ids.required' => 'Please select at least one currency.',
            'currency_ids.*.exists' => 'Selected currency is invalid.',
            'transaction_amounts.required' => 'Transaction amount is required for selected currencies.',
            'transaction_types.required' => 'Transaction type is required for selected currencies.',
        ]);
        
        // Additional validation for nested arrays
        foreach ($this->currency_ids as $currencyId) {
            $this->validate([
                "transaction_amounts.{$currencyId}" => 'required|min:0',
                "transaction_types.{$currencyId}" => 'required|in:deposit,withdrawal',
            ], [
                "transaction_amounts.{$currencyId}.required" => "Transaction amount is required for the selected currency.",
                "transaction_amounts.{$currencyId}.min" => "Transaction amount must be at least 0.",
                "transaction_types.{$currencyId}.required" => "Transaction type is required for the selected currency.",
                "transaction_types.{$currencyId}.in" => "Invalid transaction type selected.",
            ]);
        }

        try {
            // Transliterate customer name to English for user name and email
            $transliteratedName = KurdishTextHelper::transliterateToEnglish($this->customer_name);
            // Convert spaces to dots for email format
            $generatedEmail = str_replace(' ', '.', $transliteratedName) . '@LA DERMA.com';
            // Clean up email: remove multiple dots and trim
            $generatedEmail = preg_replace('/\.+/', '.', $generatedEmail);
            $generatedEmail = trim($generatedEmail, '.');

            $customer = Customer::where('customer_name', $this->customer_name)
                                ->where('phone_number', $this->phone_number)
                                ->first();

            if (!$customer) {
                $user = User::where('email', $generatedEmail)->first();
                if (!$user) {
                    $user = User::create([
                        'name' => $transliteratedName, // Use transliterated name for user
                        'email' => $generatedEmail,
                        'password' => Hash::make('password123'),
                    ]);
                }
                $user->syncRoles('customer');

                $customer = Customer::create([
                    'customer_name' => $this->customer_name,
                    'phone_number' => $this->phone_number,
                    'second_phone' => $this->second_phone,
                    'address' => $this->address,
                    'note' => $this->note,
                    'user_id' => $user->id,
                ]);
            }

            // Process accounts and transactions for each selected currency
            $skippedCurrencies = [];
            foreach ($this->currency_ids as $currency_id) {
                // Check if account already exists for this customer and currency
                $existingAccount = Account::where('customer_id', $customer->id)
                                          ->where('currency_id', $currency_id)
                                          ->first();

                if ($existingAccount) {
                    $currency = Currency::find($currency_id);
                    $currencyName = $currency->currency_name ?? 'Currency';
                    $skippedCurrencies[] = $currencyName;
                    continue; // Skip creating account if it already exists
                }

                // Remove comma and convert to float
                $transaction_amount = isset($this->transaction_amounts[$currency_id]) 
                    ? floatval(str_replace(',', '', $this->transaction_amounts[$currency_id])) 
                    : 0;
                $transaction_type = $this->transaction_types[$currency_id] ?? 'deposit';

                // Create account
                $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                                      ->orderBy('number', 'desc')
                                      ->first();

                $newAccountTitle = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';

                $account = Account::create([
                    'account_title' => $newAccountTitle,
                    'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                    'balance' => $transaction_amount,
                    'customer_id' => $customer->id,
                    'currency_id' => $currency_id,
                    'account_type' => 'savings',
                ]);

                // If there's a transaction amount, create a transaction
                if ($transaction_amount > 0) {
                    // For deposits, use Business Owner Cashbox account (owner's cashbox)
                    // For withdrawals, use profit account (customer_id = 1, account_type = 'profit')
                    if ($transaction_type == 'deposit') {
                        // Deposit: money goes to Business Owner Cashbox
                        $cashboxAccount = Account::where('account_title', 'Business Owner Cashbox')
                            ->where('currency_id', $currency_id)
                            ->where('account_type', 'savings')
                            ->first();
                        
                        // If Business Owner Cashbox doesn't exist, create it
                        if (!$cashboxAccount) {
                            $currency = Currency::find($currency_id);
                            $cashboxAccount = Account::create([
                                'account_title' => 'Business Owner Cashbox',
                                'account_number' => 'BO-' . strtoupper($currency->currency_name ?? 'CUR') . '-' . time(),
                                'balance' => 0,
                                'customer_id' => null, // No customer for owner cashbox
                                'currency_id' => $currency_id,
                                'account_type' => 'savings',
                            ]);
                        }
                    } else {
                        // Withdrawal: use profit account (customer_id = 1)
                        $cashboxAccount = Account::where('customer_id', 1)
                            ->where('currency_id', $currency_id)
                            ->where('account_type', 'profit')
                            ->first();
                        
                        // If profit account doesn't exist, try savings account as fallback
                        if (!$cashboxAccount) {
                            $cashboxAccount = Account::where('customer_id', 1)
                                ->where('currency_id', $currency_id)
                                ->where('account_type', 'savings')
                                ->first();
                        }
                        
                        // If neither exists, create the profit account
                        if (!$cashboxAccount) {
                            $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                                                  ->orderBy('number', 'desc')
                                                  ->first();
                            
                            $newAccountTitle = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';
                            
                            $cashboxAccount = Account::create([
                                'account_title' => $newAccountTitle,
                                'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                                'balance' => 0,
                                'customer_id' => 1,
                                'currency_id' => $currency_id,
                                'account_type' => 'profit',
                            ]);
                        }
                    }

                    // Get current date and time
                    $currentDate = Carbon::now()->toDateString();
                    $currentTime = Carbon::now()->toTimeString();
                
                    // to store unique GroupId 
                    $transactionGroupId = Str::uuid();

                    // Determine transaction fee
                    $transactionFee = 0;
                    $getFee = $transactionFee;

                    // Create the main transaction for the from_account
                    $transactionFrom = AccountTransaction::create([
                        'from_account_id' => $cashboxAccount->id,
                        'to_account_id' => $account->id,
                        'person_name' => $this->customer_name,
                        'get_fee' => $getFee,
                        'transaction_type' => $transaction_type,
                        'transaction_amount' => $transaction_amount,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => "Initial balance for new customer account",
                        'user_id' => Auth::id(),
                        'transaction_group_id' => $transactionGroupId,
                    ]);

                    // Create a corresponding transaction for the to_account
                    $transactionTo = AccountTransaction::create([
                        'from_account_id' => $account->id,
                        'to_account_id' => $cashboxAccount->id,
                        'person_name' => $this->customer_name,
                        'get_fee' => $getFee,
                        'transaction_type' => $transaction_type,
                        'transaction_amount' => $transaction_amount,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => "Initial balance for new customer account",
                        'user_id' => Auth::id(),
                        'transaction_group_id' => $transactionGroupId,
                    ]);

                    // Update account balances
                    if ($transaction_type == 'deposit') {
                        // Deposit: money goes FROM Business Owner Cashbox TO customer
                        // Customer receives the deposit amount
                        $account->balance += $transaction_amount;
                        // Business Owner Cashbox loses the deposit amount (money going out)
                        $cashboxAccount->balance -= $transaction_amount;
                        // If there's a fee, cashbox gains the fee
                        if ($getFee > 0) {
                            $cashboxAccount->balance += $getFee;
                        }
                    } elseif ($transaction_type == 'withdrawal') {
                        // Withdrawal: money goes FROM customer TO profit account
                        // Customer pays the withdrawal amount + fee
                        $account->balance -= ($transaction_amount + $getFee);
                        // Profit account receives the withdrawal amount
                        $cashboxAccount->balance += $transaction_amount;
                        // Profit account gains the fee amount
                        $cashboxAccount->balance += $getFee;
                    }

                    // Save the updated balances
                    $account->save();
                    $cashboxAccount->save();
                }
            }

            // Prepare success message
            $message = 'Customer and accounts created successfully!';
            if (!empty($skippedCurrencies)) {
                $skipped = implode(', ', $skippedCurrencies);
                $message .= ' Note: Account(s) for ' . $skipped . ' already exist and were skipped.';
            }
            
            $this->alert('success', $message);

            $this->resetForm();
            $this->showCreateModal = false;

        } catch (\Exception $e) {
            \Log::error('Customer creation failed', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $this->alert('error', 'Failed to create customer: ' . $e->getMessage());
        }
    }

    public $edit_customer_name, $edit_phone_number, $edit_second_phone, $edit_address, $edit_note;
    public $editCustomerId;

    public function showEditCustomerModal($customerId)
    {
        $customer = Customer::findOrFail($customerId);

        $this->editCustomerId = $customer->id;
        $this->edit_customer_name = $customer->customer_name;
        $this->edit_phone_number = $customer->phone_number;
        $this->edit_second_phone = $customer->second_phone;
        $this->edit_address = $customer->address;
        $this->edit_note = $customer->note;
        $this->showEditModal = true; // Close the modal
    }

    public function updateCustomer()
    {
        $this->validate([
            'edit_customer_name' => 'required|string',
            'edit_phone_number' => 'required|string',
            'edit_second_phone' => 'nullable|string',
            'edit_address' => 'nullable|string',
            'edit_note' => 'nullable|string',
        ]);

        $customer = Customer::findOrFail($this->editCustomerId);
        $customer->update([
            'customer_name' => $this->edit_customer_name,
            'phone_number' => $this->edit_phone_number,
            'second_phone' => $this->edit_second_phone,
            'address' => $this->edit_address,
            'note' => $this->edit_note,
        ]);

        $this->customers = Customer::with('accounts')->paginate(50);

        $this->resetEditForm();
        $this->showEditModal = false; // Close the modal

        $this->alert('success', 'Customer updated successfully!');

    }

    private function resetEditForm()
    {
        $this->edit_customer_name = '';
        $this->edit_phone_number = '';
        $this->edit_second_phone = '';
        $this->edit_address = '';
        $this->edit_note = '';
        $this->editCustomerId = null;
    }

    private function resetForm()
    {
        $this->customer_name = '';
        $this->phone_number = '';
        $this->second_phone = '';
        $this->address = '';
        $this->note = '';
        $this->currency_ids = [];
        $this->transaction_amounts = [];
        $this->transaction_types = [];
        $this->account_balance = '';
    }

    public function selectCustomerForNewAccount($customerId)
    {
        $this->selected_customer_id = $customerId; 
        $this->showCreateAccountModal=true; 
    }

    public function createAccountForSelectedCustomer()
    {
        $this->validate([
            'currency_id' => 'required|exists:currencies,id',
            'account_balance' => 'required|numeric|min:0',
        ]);

        $existingAccount = Account::where('customer_id', $this->selected_customer_id)
                                  ->where('currency_id', $this->currency_id)
                                  ->first();

        if ($existingAccount) {
            return $this->alert('error', 'Customer already has an account with this currency!');
        }

        $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                              ->orderBy('number', 'desc')
                              ->first();

        $newAccountTitle = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';

        Account::create([
            'account_title' => $newAccountTitle,
            'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
            'balance' => $this->account_balance,
            'customer_id' => $this->selected_customer_id,
            'currency_id' => $this->currency_id,
            'account_type' => 'savings',
        ]);

        $this->customers = Customer::with('accounts')->paginate(50);
        $this->showCreateAccountModal = false; // Close the modal

        $this->alert('success', 'Account created successfully!');
    }

    public function toggleCurrencyRateAccess($customerId)
    {
        try {
            $customer = Customer::with('user')->findOrFail($customerId);
            
            if (!$customer->user) {
                $this->alert('error', 'No user associated with this customer');
                return;
            }
            
            $user = User::find($customer->user_id);
            
            if (!$user) {
                $this->alert('error', 'User not found');
                return;
            }
            
            $currentValue = $user->currency_rate_access ?? 0;
            $newValue = $currentValue ? 0 : 1;
            
            \Log::info("Toggling currency_rate_access for user {$user->id}: {$currentValue} -> {$newValue}");
            
            // Update using the User model directly
            $user->currency_rate_access = $newValue;
            $saved = $user->save();
            
            if (!$saved) {
                \Log::error("Failed to save user {$user->id}");
                $this->alert('error', 'Failed to save changes');
                return;
            }
            
            // Verify the change
            $verifyUser = User::find($user->id);
            \Log::info("Verified currency_rate_access for user {$user->id}: {$verifyUser->currency_rate_access}");
            
            $status = $newValue ? 'enabled' : 'disabled';
            $this->alert('success', "Currency rate access {$status} for {$customer->customer_name}");
            
            // Force refresh of the component
            $this->dispatch('$refresh');
            
        } catch (\Exception $e) {
            \Log::error('Error toggling currency rate access: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            $this->alert('error', 'Failed to update: ' . $e->getMessage());
        }
    }

    public function render()
    {


        $query = Customer::with('accounts', 'user')
            ->where('customer_name', '!=', 'Bank (Cashboxes)')
            ->where('customer_name', '!=', 'Cashiers')
            ->where('id', '!=', 1); // Exclude customer with ID 1



        if ($this->searchQuery) {
            $query->where('customer_name', 'like', '%' . $this->searchQuery . '%')
                  ->orWhere('phone_number', 'like', '%' . $this->searchQuery . '%')
                  ->orWhere('id',  $this->searchQuery)
                  ->orWhere('second_phone', 'like', '%' . $this->searchQuery . '%');
        }

        return view('livewire.customer-management', [
            'customers' => $query->paginate(50),
        ]);
    }
}
