<?php

namespace App\Livewire;

use App\Models\Cashier;
use App\Models\Account;
use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\CashierShift;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Currency; // Import the Currency model

class CurrencyTransactionForm extends Component
{

    public $cashier_shift_id;
    public $transaction_date;
    public $transaction_time;
    public $transaction_amount;
    public $transaction_type;
    public $from_currency;
    public $to_currency;
    public $amount_given , $amount_received ;
    public $currencies = [];
    public $filters;

    public function mount()
    {

        $this->filters = array(
            'to_currency' => null,
            'from_currency' => null
        );
        // Set default values for date and time
        $this->transaction_date = now()->format('Y-m-d');
        $this->transaction_time = now()->format('H:i:s');
        $this->currencies = Currency::where('rate', 1)->get();
        // Set cashier_shift_id from related Cashier model
        if (Auth::check()) {
          $user = Auth::user();
          $cashier = Cashier::where('user_id',$user->id)->first(); // Get the related Cashier
          if (!$cashier) {
            $this->cashier_shift_id = null;

              session()->flash('error', 'Cashier Shift ID is not available.');
              return;
        } else {
          $cashier_shift = CashierShift::where('cashier_id',$cashier->id)->first(); // Get the related Cashier
          if ($cashier_shift) {
              $this->cashier_shift_id = $cashier_shift->id;
          } else {
            $this->cashier_shift_id = null;

              session()->flash('error', 'Cashier Shift ID is not available.');
              return;
          }
        }
      }

    }
    protected $listeners = ['valueSelected'];

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            // It's two words separated by a dot
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
    }

        protected $rules = [
          'transaction_amount' => 'required|numeric|min:0',
          'transaction_type' => 'required',
          'filters.from_currency' => 'required',
        //   'filters.to_currency' => 'required|different:filters.from_currency', // Ensures that 'to_currency' is different from 'from_currency'
        //   'amount_given' => 'required|numeric|min:0',
      ];

      protected $messages = [
        'filters.to_currency.different' => 'The to_currency must be different from the from_currency.',
    ];

    public function submit()
    {
        // Validate the form data
        $this->validate();

        // Check if the cashier shift ID is present
        if ($this->cashier_shift_id == null) {
            session()->flash('error', 'Cashier Shift ID is not available.');
            return;
        }

        DB::beginTransaction(); // Start a transaction to ensure data consistency

        try {
            // Insert into the 'transactions' table
            DB::table('transactions')->insert([
                'cashier_shift_id' => $this->cashier_shift_id,
                'transaction_date' => $this->transaction_date,
                'transaction_time' => $this->transaction_time,
                'amount_given' => 0,
                'amount_received' => 0,
                'transaction_amount' => $this->transaction_amount,
                'transaction_type' => $this->transaction_type, // "buy" or "sell"
                'from_currency' => $this->filters['from_currency'],
                'to_currency' => null,
            ]);

            // Fetch accounts based on the selected currencies and customer_id = 1
            $fromAccount = Account::where('currency_id', $this->filters['from_currency'])
                ->where('customer_id', 1) // Ensure account belongs to customer_id 1
                ->first();

            // $toAccount = Account::where('currency_id', $this->filters['to_currency'])
            //     ->where('customer_id', 1) // Ensure account belongs to customer_id 1
            //     ->first();

            // Ensure both accounts exist
            if (!$fromAccount ) {
                throw new \Exception('Matching accounts for the selected currencies not found.');
            }

            // Map transaction types for AccountTransactionModel
            $accountTransactionType = null;
            if ($this->transaction_type == 'buy') {
                $accountTransactionType = 'deposit';  // Buying currency is treated as a deposit
            } elseif ($this->transaction_type == 'sell') {
                $accountTransactionType = 'withdrawal';  // Selling currency is treated as a withdrawal
            }

            // Insert into the 'account_transactions' table
            AccountTransactionModel::create([
                'from_account_id' => $fromAccount->id,
                'to_account_id' => null,
                'transaction_type' => $accountTransactionType, // Map to deposit or withdrawal
                'transaction_amount' => $this->transaction_amount,
                'transaction_date' => $this->transaction_date,
                'transaction_time' => $this->transaction_time,
                'note' => 'Transaction from ' . $fromAccount->currency->name . ' | ' . $this->transaction_type . ' | '. $this->transaction_amount,
                'user_id' => Auth::id(),
            ]);

            // Update account balances based on transaction type
            if ($this->transaction_type == 'buy') {
                // For buy, the fromAccount will be deducted (i.e., selling the currency)
                // and the toAccount will be credited (i.e., buying the currency)
                $fromAccount->balance -= $this->transaction_amount; // Decrease fromAccount
                // $toAccount->balance += $this->transaction_amount;   // Increase toAccount
            } elseif ($this->transaction_type == 'sell') {
                // For sell, the fromAccount will be credited (i.e., buying the currency)
                // and the toAccount will be deducted (i.e., selling the currency)
                $fromAccount->balance += $this->transaction_amount;  // Increase fromAccount
                // $toAccount->balance -= $this->transaction_amount;    // Decrease toAccount
            }

            // Save updated balances
            $fromAccount->save();
            // $toAccount->save();

            DB::commit(); // Commit the transaction

            // Flash success message and reset fields
            session()->flash('message', 'Transaction successfully added.');
            $this->transaction_amount = '';
            $this->amount_given = '';
            $this->dispatch('clearCustomSelect');

        } catch (\Exception $e) {
            DB::rollBack(); // Rollback on failure
            session()->flash('error', 'Transaction failed: ' . $e->getMessage());
        }
    }





    public function render()
    {
        return view('livewire.currency-transaction-form');
    }
}

