<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Cashier;
use App\Models\CashierShift;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;

class CashierShifts extends Component
{
    use LivewireAlert, WithPagination;

    public $cashier_id;
    public $supervisor_id;
    public $start_time;
    public $end_time;
    public $starting_amount;
    public $ending_amount;

    public $cashiers = [];
    public $supervisors = [];
    public $shifts = [];

    public function mount()
    {
        // Fetch cashiers
        $this->cashiers = Cashier::with('user') // Eager load the user relationship
            ->get();

        // Fetch supervisors
        $this->supervisors = Cashier::where('cashier_type', 'supervisor')
            ->with('user') // Eager load the user relationship
            ->get();
    }

    public function render()
    {
        // Fetch all shifts with related cashiers and supervisors
        $this->shifts = CashierShift::with('cashier.user', 'supervisor') // Eager load relationships
            ->get();

        return view('livewire.cashier-shifts');
    }

    public function store()
    {
        // Validation
        $this->validate([
            'cashier_id' => 'required|exists:cashiers,id',
            'supervisor_id' => 'required|exists:users,id',
        ]);
    
        // Check if a CashierShift record already exists for the given cashier and supervisor
        $existingShift = CashierShift::where('cashier_id', $this->cashier_id)
            ->where('supervisor_id', $this->supervisor_id)
            ->first();
    
        if ($existingShift) {
            $this->alert('warning', 'A shift already exists for this cashier and supervisor!');
            return;
        }
    
        // Create a new CashierShift record
        CashierShift::create([
            'cashier_id' => $this->cashier_id,
            'supervisor_id' => $this->supervisor_id,
            'start_time' => now(),
            'end_time' => now(),
            'starting_amount' => 0,
            'ending_amount' => 0,
        ]);
    
        $this->alert('success', 'Cashier Shift created successfully!');
    
        // Reset fields
        $this->reset();
    
    
    }
    
}
