<?php
namespace App\Livewire;

use Livewire\Component;
use App\Models\Account;
use App\Models\AccountTransaction;
use App\Models\Currency;
use App\Models\Customer;
use Illuminate\Support\Facades\DB;

class CashboxTransactions extends Component
{
    public $customers;
    public $filterIQD = 'all';
    public $filterUSD = 'all';
    
    // Cache the calculated balances
    private $customerBalances = null;

    public function mount()
    {
        // Optimize the query with proper eager loading and aggregation
        $this->customers = Customer::whereNotNull('customer_name')
            ->whereNotIn('customer_name', ['Cashiers', 'Supervisor', 'Bank (Cashboxes)'])
            ->whereHas('accounts.currency', function ($query) {
                $query->whereIn('currency_name', ['IQD', 'USD']);
            })
            ->with(['accounts' => function ($query) {
                $query->whereHas('currency', function ($q) {
                    $q->whereIn('currency_name', ['IQD', 'USD']);
                })->with('currency');
            }])
            ->get();
            
        // Calculate balances once during mount
        $this->calculateCustomerBalances();
    }

    private function calculateCustomerBalances()
    {
        if ($this->customerBalances !== null) {
            return $this->customerBalances;
        }

        // Get all customer IDs
        $customerIds = $this->customers->pluck('id')->toArray();
        
        // Single optimized query to get all balances at once
        $balances = DB::table('account_transactions as at')
            ->join('accounts as a', 'at.from_account_id', '=', 'a.id')
            ->join('currencies as c', 'a.currency_id', '=', 'c.id')
            ->join('customers as cu', 'a.customer_id', '=', 'cu.id')
            ->whereIn('cu.id', $customerIds)
            ->whereIn('c.currency_name', ['IQD', 'USD'])
            ->where('at.status', '!=', 'rejected')
            ->select([
                'cu.id as customer_id',
                'c.currency_name',
                'at.status',
                DB::raw('SUM(CASE 
                    WHEN at.transaction_type = "deposit" THEN ROUND(at.transaction_amount)
                    WHEN at.transaction_type = "withdrawal" THEN -ROUND(at.transaction_amount)
                    ELSE 0 
                END) as balance')
            ])
            ->groupBy(['cu.id', 'c.currency_name', 'at.status'])
            ->get();

        // Process the results into a structured array
        $this->customerBalances = [];
        
        foreach ($balances as $balance) {
            $customerId = $balance->customer_id;
            $currency = $balance->currency_name;
            $status = $balance->status;
            
            if (!isset($this->customerBalances[$customerId])) {
                $this->customerBalances[$customerId] = [
                    'IQD' => ['approved' => 0, 'pending' => 0, 'total' => 0],
                    'USD' => ['approved' => 0, 'pending' => 0, 'total' => 0]
                ];
            }
            
            $this->customerBalances[$customerId][$currency][$status] = $balance->balance;
            $this->customerBalances[$customerId][$currency]['total'] = 
                $this->customerBalances[$customerId][$currency]['approved'] + 
                $this->customerBalances[$customerId][$currency]['pending'];
        }
        
        return $this->customerBalances;
    }

    private function getCustomerBalanceIQD($customer)
    {
        return $this->customerBalances[$customer->id]['IQD']['total'] ?? 0;
    }

    private function getCustomerBalanceUSD($customer)
    {
        return $this->customerBalances[$customer->id]['USD']['total'] ?? 0;
    }

    public function updatedFilterIQD($value)
    {
        // Reset customer balances cache when filter changes to ensure fresh data
        $this->customerBalances = null;
    }

    public function updatedFilterUSD($value)
    {
        // Reset customer balances cache when filter changes to ensure fresh data
        $this->customerBalances = null;
    }

    public function render()
    {
        // Ensure balances are calculated
        $this->calculateCustomerBalances();
        
        // Get the base collection of customers
        $filteredCustomers = $this->customers;
        
        // Create separate collections for IQD and USD views
        $customersForIQD = collect($filteredCustomers);
        $customersForUSD = collect($filteredCustomers);

        // Apply IQD filter if needed
        if ($this->filterIQD === 'positive') {
            $customersForIQD = $customersForIQD->filter(function ($customer) {
                $balance = $this->getCustomerBalanceIQD($customer);
                return $balance > 0;
            });
        } elseif ($this->filterIQD === 'negative') {
            $customersForIQD = $customersForIQD->filter(function ($customer) {
                $balance = $this->getCustomerBalanceIQD($customer);
                return $balance < 0;
            });
        }

        // Apply USD filter if needed
        if ($this->filterUSD === 'positive') {
            $customersForUSD = $customersForUSD->filter(function ($customer) {
                $balance = $this->getCustomerBalanceUSD($customer);
                return $balance > 0;
            });
        } elseif ($this->filterUSD === 'negative') {
            $customersForUSD = $customersForUSD->filter(function ($customer) {
                $balance = $this->getCustomerBalanceUSD($customer);
                return $balance < 0;
            });
        }

        return view('livewire.cashbox-transactions', [
            'customersIQD' => $customersForIQD,
            'customersUSD' => $customersForUSD,
            'customerBalances' => $this->customerBalances // Pass balances to view
        ]);
    }
}