<?php
namespace App\Livewire;

use Livewire\Component;
use App\Models\Account;  // Assuming there's an Account model for cashbox accounts
use App\Models\Cashier; 
use App\Models\Currency; 
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use App\Models\AccountTransaction as AccountTransactionModel;
use Illuminate\Support\Str;
use Carbon\Carbon;

class BankCashbox extends Component
{
    use LivewireAlert, WithPagination;

    public $accounts,$currencies,$currency_id, $showCreateModal=false, $showOwnerModal=false;
    public $cashiers, $cashier_accounts=[], $selectedCashier, $ownerNote = '',$amount, $selectedAccount, $transaction_type = ['','deposit','withdrawal'];
    public $filterType = 'savings'; // To store the selected filter (all, savings, or profit)
    public $showResetModal = false;
    public $accountId;


    public function mount()
    {
        $user = Auth::user();

        // Check user roles
        $isSuperAdmin = $user->hasRole('super-admin');
        $supervisor = $user->hasRole('supervisor');

         // Load stored preferences from session if they exist
         $this->filterType = session('cashbox_filter_type', $supervisor ? 'cashier' : 'savings');
         $this->selectedCashier = session('selected_cashier_id');
         
         // If a cashier was previously selected and we're on the right filter type, load their accounts
         if ($this->selectedCashier && ($this->filterType == 'box_cashier' || $this->filterType == 'cashier')) {
             $this->loadCashierAccounts();
         }
         
         // Set filterType to 'cashier' if the user is a supervisor
         if ($supervisor) {
             $this->filterType = 'box_cashier';
         }
        
        // Load currencies and cashiers with caching
        $this->currencies = cache()->remember('currencies_list', 3600, function () {
            return Currency::all();
        });
        
        $this->cashiers = cache()->remember('cashiers_list', 1800, function () {
            return Cashier::all();
        });

        $this->loadAccounts();
    }

    public function updatedFilterType()
    {
        session(['cashbox_filter_type' => $this->filterType]);
        
        // If changing away from cashier views, clear selected cashier
        if ($this->filterType != 'box_cashier' && $this->filterType != 'cashier') {
            $this->selectedCashier = null;
            session(['selected_cashier_id' => null]);
        }
        
        $this->loadAccounts();
    }

    public function ownerBalance()
{ 
    $this->amount = str_replace(',', '', $this->amount);

    $this->validate([
        'ownerNote' => 'required|string',
        'amount' => 'required|numeric|min:0',
        'transaction_type' => 'required|in:deposit,withdrawal',
    ]);

    $selectedAccount = Account::find($this->selectedAccount);
    if (!$selectedAccount) {
        $this->alert('error', 'Selected account not found');
        return;
    }

    // Find the owner's cashbox for this currency
    $ownerCashbox = Account::where('account_title', 'Business Owner Cashbox')
        ->where('currency_id', $selectedAccount->currency->id)
        ->first();

    if (!$ownerCashbox) {
        $this->alert('error', 'حسابەکە نەدۆزراوە تکایە سەرەتا حسابی دروست بکە');
        return;
    }

    // Prevent transactions between Business Owner Cashbox and itself
    // if ($selectedAccount->id == $ownerCashbox->id) {
    //     $this->alert('error', 'Cannot perform transactions between the same Business Owner Cashbox account');
    //     return;
    // }

    // Get current date and time
    $currentDate = Carbon::now()->toDateString();
    $currentTime = Carbon::now()->toTimeString();
    
    // Create a transaction group ID
    $transactionGroupId = Str::uuid();
    
    // Log transaction details for debugging
    \Log::info('Transaction details', [
        'selected_account' => $selectedAccount->id . ' (' . $selectedAccount->account_title . ')',
        'owner_cashbox' => $ownerCashbox->id . ' (' . $ownerCashbox->account_title . ')',
        'transaction_type' => $this->transaction_type,
        'amount' => $this->amount
    ]);
    
    // Create the transaction based on transaction type
    if ($this->transaction_type == 'deposit') {
        // Create a deposit transaction for the selected account
        // This increases the selected account's balance
        AccountTransactionModel::create([
            'from_account_id' => $selectedAccount->id,
            'to_account_id' => null, // to_account_id can be null as per your balance calculation
            'person_name' => '',
            'get_fee' => 0,
            'transaction_type' => 'deposit',
            'transaction_amount' => $this->amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->ownerNote,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);
        
        // Log for audit purposes
        \Log::info('Created deposit transaction', [
            'account_id' => $selectedAccount->id,
            'amount' => $this->amount,
            'transaction_group_id' => $transactionGroupId
        ]);
        
    } elseif ($this->transaction_type == 'withdrawal') {
        // Create a withdrawal transaction for the selected account
        // This decreases the selected account's balance
        AccountTransactionModel::create([
            'from_account_id' => $selectedAccount->id,
            'to_account_id' => null, // to_account_id can be null as per your balance calculation
            'person_name' => '',
            'get_fee' => 0,
            'transaction_type' => 'withdrawal',
            'transaction_amount' => $this->amount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => $this->ownerNote,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);
        
        // Log for audit purposes
        \Log::info('Created withdrawal transaction', [
            'account_id' => $selectedAccount->id,
            'amount' => $this->amount,
            'transaction_group_id' => $transactionGroupId
        ]);
    }

    $this->closeOwnerModal();
    $this->alert('success', 'جوڵەکە سەرکەوتوو بو');
    $this->showOwnerModal = false;
}

    public function openOwnerModal(Account $account)
    {
        $this->selectedAccount = $account->id;
        $this->showOwnerModal = true; // Open the modal
    }

    public function closeOwnerModal()
    {
    
        $this->ownerNote = '';
        $this->amount = '';
        $this->currency_id = null;
        $this->transaction_type = [''];
        
        $this->showOwnerModal = false; // Close the modal
    }
    
    public function openCreateModal()
    {
        $this->showCreateModal = true; // Open the modal
    }

    public function closeCreateModal()
    {
        $this->currency_id = null;
        
        $this->showCreateModal = false; // Close the modal
    }

    public function confirmReset($accountId)
    {
        $this->accountId = $accountId;
        $this->showResetModal = true;
    }

    public function closeResetModal()
    {
        $this->showResetModal = false;
    }

    public function reset_balance()
    {
        $account = Account::find($this->accountId);
        
        if ($account) {
            // Reset balance logic
            $this->resetAccountBalance($account);

            // Close modal
            $this->showResetModal = false;

            // Notify user (optional)
            $this->alert('success', 'بەسەرکەوتووی ئەنجام درا');
            // session()->flash('message', 'بەسەرکەوتووی ئەنجام درا');
        }
    }

    private function resetAccountBalance($account)
    {
        $currentBalance = $account->calculateTotalBalance();
        $transactionGroupId = Str::uuid();

        if ($currentBalance != 0) {
            \App\Models\AccountTransaction::create([
                'from_account_id' => $account->id,
                'to_account_id' => null,
                'transaction_type' => ($currentBalance > 0) ? 'withdrawal' : 'deposit',
                'transaction_amount' => abs($currentBalance),
                'transaction_date' => now()->toDateString(),
                'transaction_time' => now()->toTimeString(),
                'note' => 'سفر کردنەوەی بەڵانس',
                'get_fee' => 0,
                'person_name' => 'System',
                'user_id' => auth()->id(),
                'transaction_id' => null,
                'transaction_group_id' => $transactionGroupId, // Generate if necessary
            ]);
        }
    }

    public function loadCashierAccounts()
    {
        // Clear cashier_accounts if filterType is not box_cashier or cashier
        if ($this->filterType !== 'box_cashier' && $this->filterType !== 'cashier') {
            $this->cashier_accounts = [];
            return;
        }

        // Check if a cashier is selected
        if ($this->selectedCashier) {
            // Retrieve the selected cashier based on the cashier ID with eager loading
            $cashier = Cashier::with('user.customer')->find($this->selectedCashier);

            if ($cashier && $cashier->user && $cashier->user->customer) {
                // Fetch accounts where the customer_id matches the cashier's customer_id with eager loading
                $this->cashier_accounts = Account::with('currency')
                    ->where('customer_id', $cashier->user->customer->id)
                    ->get();

                // Pre-calculate balances for cashier accounts efficiently
                if ($this->cashier_accounts->isNotEmpty()) {
                    $accountIds = $this->cashier_accounts->pluck('id')->toArray();
                    $balances = Account::calculateBalancesForAccounts($accountIds);
                    
                    // Add balance as a computed property to each account
                    $this->cashier_accounts->each(function ($account) use ($balances) {
                        $account->computed_balance = $balances->get($account->id, 0);
                    });
                }
            } else {
                // If no cashier is found, set an empty array
                $this->cashier_accounts = [];
            }
        } else {
            // If no cashier is selected, clear the accounts
            $this->cashier_accounts = [];
        }
    }

    // Update the loadAccounts() method with eager loading and optimized balance calculation
    public function loadAccounts()
    {
        if ($this->filterType == 'savings') {
            // For Business Owner Cashbox, return all accounts with this title
            // Show accounts with non-digital currencies (type = 0 or NULL) or all if no type filter needed
            $this->accounts = Account::with('currency')
                ->where('account_title', 'Business Owner Cashbox')
                ->where(function($query) {
                    $query->whereHas('currency', function($q) {
                        $q->where(function($subQ) {
                            $subQ->where('type', 0)
                                 ->orWhereNull('type'); // Include currencies with NULL type (defaults to non-digital)
                        });
                    })
                    ->orDoesntHave('currency'); // Include accounts without currency relationship (shouldn't happen, but safe)
                })
                ->get();
        } elseif ($this->filterType == 'profit_only') {
            // Show only profit accounts with digital currencies (type = 1)
            $this->accounts = Account::with('currency')
                ->where('customer_id', 1)
                ->whereHas('currency', function ($query) {
                    // Filter by 'digital' currency type
                    $query->where('type', 1);
                })
                ->where('account_type','savings')
                ->get();
        } elseif ($this->filterType == 'cashier') {
            // For cashier filter, load cashier transfer accounts
            $this->accounts = Account::with('currency')
                ->where('account_type', 'cashier')
                ->where('customer_id', 1)
                ->get();
            
            // Also load cashier accounts if a cashier is selected
            if ($this->selectedCashier) {
                $this->loadCashierAccounts();
            }
        } else {
            // Original behavior for other filter types
            $this->accounts = Account::with('currency')
                ->where('account_type', $this->filterType)
                ->where('customer_id', 1)
                ->get();
        }

        // Pre-calculate balances for all accounts efficiently
        if ($this->accounts->isNotEmpty()) {
            $accountIds = $this->accounts->pluck('id')->toArray();
            $balances = Account::calculateBalancesForAccounts($accountIds);
            
            // Add balance as a computed property to each account
            $this->accounts->each(function ($account) use ($balances) {
                $account->computed_balance = $balances->get($account->id, 0);
            });
        }
    }

    /**
     * Generate optimized image URL with WebP support and fallback
     */
    public function getOptimizedImageUrl($currency)
    {
        if (!$currency || !$currency->code) {
            return null;
        }

        $originalUrl = \Storage::url($currency->code);
        
        // Check if WebP version exists
        $webpPath = str_replace(['.jpg', '.jpeg', '.png'], '.webp', $currency->code);
        $webpUrl = \Storage::url($webpPath);
        
        // For now, return original URL. In production, you might want to:
        // 1. Check if WebP exists
        // 2. Use WebP if available
        // 3. Fall back to original format
        
        return $originalUrl;
    }
    
     // Triggered when the cashier dropdown value changes
     public function updatedSelectedCashier()
     {
         // Store the selected cashier ID in session
         session(['selected_cashier_id' => $this->selectedCashier]);
         
         $this->loadCashierAccounts();
     }    

    public function CreateCashierAccount()
    {
        // Retrieve all currencies where type = 0
        $currencies = Currency::where('type', 0)->get();

        foreach ($currencies as $currency) {
            // Check if a cashier account already exists for the currency
            $existingCashierAccount = Account::where('customer_id', 1)
                                            ->where('currency_id', $currency->id)
                                            ->where('account_type', 'cashier')
                                            ->first();

            if (!$existingCashierAccount) {
                // Generate unique account title
                $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                                    ->orderBy('number', 'desc')
                                    ->first();

                $newAccountTitleCashier = $lastAccount 
                    ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) 
                    : 'CN00001';

                // Create the cashier account
                Account::create([
                    'account_title' => $newAccountTitleCashier,
                    'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                    'balance' => 0, // Default balance for cashier accounts
                    'customer_id' => 1,
                    'currency_id' => $currency->id,
                    'account_type' => 'cashier',
                ]);
            }
        }

        $this->alert('success', 'Cashier accounts created successfully for all applicable currencies!');

        // Clear cache to ensure fresh data
        cache()->forget('currencies_list');
        cache()->forget('cashiers_list');

        // Reload accounts after creation
        $this->loadAccounts();

        // Emit event to close modal (if needed)
        $this->dispatch('close-modal');
    }


    public function createAccount()
    {
        $this->validate([
            'currency_id' => 'required|exists:currencies,id',
        ]);
    
        // Check if the primary (cashbox/savings) account already exists
        $existingCashboxAccount = Account::where('customer_id', 1)
                                         ->where('currency_id', $this->currency_id)
                                         ->where('account_type', 'savings')
                                         ->first();
    
        // Check if the profit account already exists for the currency
        $existingProfitAccount = Account::where('customer_id', 1)
                                        ->where('currency_id', $this->currency_id)
                                        ->where('account_type', 'profit')
                                        ->first();
        
        // If both accounts already exist, show error and return
        if ($existingCashboxAccount && $existingProfitAccount) {
            $this->alert('error', 'ئەم دراوە پێشتر دروستکراوە!');
            $this->closeCreateModal();
            return;
        }
    
        if (!$existingCashboxAccount) {
            // Create the savings account (cashbox)
            $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                                  ->orderBy('number', 'desc')
                                  ->first();
    
            $newAccountTitleCashbox = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';
    
            $cashboxAccount = Account::create([
                'account_title' => $newAccountTitleCashbox,
                'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                'balance' => 0,
                'customer_id' => 1,
                'currency_id' => $this->currency_id,
                'account_type' => 'savings',
            ]);
        }
     
         if (!$existingProfitAccount) {
             // Create the profit account
             $lastAccount = Account::selectRaw('CAST(SUBSTRING(account_title, 3) AS UNSIGNED) AS number')
                                   ->orderBy('number', 'desc')
                                   ->first();
     
             $newAccountTitleProfit = $lastAccount ? 'CN' . str_pad($lastAccount->number + 1, 5, '0', STR_PAD_LEFT) : 'CN00001';
     
             $profitAccount = Account::create([
                 'account_title' => $newAccountTitleProfit,
                 'account_number' => 'ACC' . str_pad(rand(1, 999999), 6, '0', STR_PAD_LEFT),
                 'balance' => 0, // Profit accounts typically start with a zero balance
                 'customer_id' => 1,
                 'currency_id' => $this->currency_id,
                 'account_type' => 'profit',
             ]);
         }
     
        $this->alert('success', 'حسابەکان بە سەرکەوتوویی دروستکران!');
    
        // Clear cache to ensure fresh data
        cache()->forget('currencies_list');
        cache()->forget('cashiers_list');
    
        // Reload accounts after creation
        $this->loadAccounts();
    
        // Close modal
        $this->closeCreateModal();
    
        // Emit event to close modal (Bootstrap requires this)
        $this->dispatch('close-modal');
     }
     

    public function render()
    {
        // Use cached currencies and filter for type = 0
        $currencies = $this->currencies->where('type', 0);
        
        // Use the already loaded accounts from loadAccounts() method
        $accounts = $this->accounts ?? collect();
    
        return view('livewire.bank-cashbox', [
            'currencies' => $currencies,
            'accounts' => $accounts,
        ]);
    }
}
