<?php

namespace App\Livewire;

use App\Models\AppVersion;
use Livewire\Component;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Auth;

class AppVersionManagement extends Component
{
    use WithPagination;

    public $version = '';
    public $version_code = '';
    public $platform = 'android';
    public $release_notes = '';
    public $download_url = '';
    public $min_supported_version = '';
    public $is_active = true;
    public $is_force_update = false;
    public $editing_id = null;
    public $show_form = false;

    protected $rules = [
        'version' => 'required|string|max:20',
        'version_code' => 'required|integer|min:1',
        'platform' => 'required|in:android,ios',
        'release_notes' => 'nullable|string',
        'download_url' => 'nullable|url',
        'min_supported_version' => 'nullable|string|max:20',
        'is_active' => 'boolean',
        'is_force_update' => 'boolean',
    ];

    public function mount()
    {
        // Check if user has super-admin role
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
    }

    public function render()
    {
        // Double check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }

        $app_versions = AppVersion::orderBy('platform')
            ->orderBy('version_code', 'desc')
            ->paginate(10);

        return view('livewire.app-version-management', [
            'app_versions' => $app_versions
        ]);
    }

    public function create()
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        $this->resetForm();
        $this->show_form = true;
    }

    public function edit($id)
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        $app_version = AppVersion::findOrFail($id);
        
        $this->editing_id = $id;
        $this->version = $app_version->version;
        $this->version_code = $app_version->version_code;
        $this->platform = $app_version->platform;
        $this->release_notes = $app_version->release_notes;
        $this->download_url = $app_version->download_url;
        $this->min_supported_version = $app_version->min_supported_version;
        $this->is_active = $app_version->is_active;
        $this->is_force_update = $app_version->is_force_update;
        $this->show_form = true;
    }

    public function save()
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        $this->validate();

        $data = [
            'version' => $this->version,
            'version_code' => $this->version_code,
            'platform' => $this->platform,
            'release_notes' => $this->release_notes,
            'download_url' => $this->download_url,
            'min_supported_version' => $this->min_supported_version ?: $this->version,
            'is_active' => $this->is_active,
            'is_force_update' => $this->is_force_update,
        ];

        if ($this->editing_id) {
            AppVersion::findOrFail($this->editing_id)->update($data);
            session()->flash('message', 'App version updated successfully!');
        } else {
            AppVersion::create($data);
            session()->flash('message', 'App version created successfully!');
        }

        $this->resetForm();
        $this->show_form = false;
    }

    public function delete($id)
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        try {
            AppVersion::findOrFail($id)->delete();
            session()->flash('message', 'App version deleted successfully!');
        } catch (\Illuminate\Database\QueryException $e) {
            // Handle foreign key constraint violations
            if ($e->getCode() == 23000) {
                session()->flash('error', 'Cannot delete app version. This version has related data that prevents deletion.');
            } else {
                session()->flash('error', 'An error occurred while deleting the app version. Please try again.');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'An unexpected error occurred. Please try again.');
        }
    }

    public function toggleActive($id)
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        $app_version = AppVersion::findOrFail($id);
        $app_version->update(['is_active' => !$app_version->is_active]);
        session()->flash('message', 'App version status updated!');
    }

    public function toggleForceUpdate($id)
    {
        // Check authorization
        if (!Auth::check() || !Auth::user()->hasRole('super-admin')) {
            abort(403, 'Access denied. Super-admin role required.');
        }
        
        $app_version = AppVersion::findOrFail($id);
        $app_version->update(['is_force_update' => !$app_version->is_force_update]);
        session()->flash('message', 'Force update status updated!');
    }

    public function cancel()
    {
        $this->resetForm();
        $this->show_form = false;
    }

    private function resetForm()
    {
        $this->editing_id = null;
        $this->version = '';
        $this->version_code = '';
        $this->platform = 'android';
        $this->release_notes = '';
        $this->download_url = '';
        $this->min_supported_version = '';
        $this->is_active = true;
        $this->is_force_update = false;
    }
}
