<?php

namespace App\Livewire;

use App\Models\Account;
use App\Models\Customer;
use App\Models\AccountTransaction;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;

class AccountTransfer extends Component
{
    use LivewireAlert, WithPagination;

    public $from_account_id;
    public $to_account_id;
    public $transaction_amount;
    public $note;
    public $from_accounts;
    public $to_accounts;
    public $showModal = false;

    public function mount()
    {
        // Fetch all accounts for the logged-in user to select from
        $this->from_accounts = Account::where('customer_id',1)->get(); // Adjust as needed for permissions (e.g. user-specific)
        $this->to_accounts = Account::where('account_type','checking')->get();  // Adjust as needed for permissions (e.g. user-specific)
    }

    public function openModal()
    {
        $this->showModal = true; // Open the modal
    }

    public function closeModal()
    {
        $this->reset(['from_account_id', 'to_account_id', 'transaction_amount', 'note']);
        $this->showModal = false; // Close the modal
    }

    public function submit()
    {
        // Validate the form data
        $this->validate([
            'from_account_id' => 'required|exists:accounts,id|different:to_account_id',
            'to_account_id' => 'required|exists:accounts,id',
            'transaction_amount' => 'required|numeric|min:1',
            'note' => 'nullable|string',
        ]);

        // Get the authenticated user ID
        $userId = Auth::id();

        // Insert into the account_transaction table
        AccountTransaction::create([
            'from_account_id' => $this->from_account_id,
            'to_account_id' => $this->to_account_id,
            'transaction_type' => 'transfer',
            'transaction_amount' => $this->transaction_amount,
            'transaction_date' => now()->format('Y-m-d'),
            'transaction_time' => now()->format('H:i:s'),
            'note' => $this->note,
            'user_id' => $userId,
        ]);

        // Adjust balances in both accounts
        $fromAccount = Account::find($this->from_account_id);
        $toAccount = Account::find($this->to_account_id);

        $fromAccount->balance -= $this->transaction_amount;
        $toAccount->balance += $this->transaction_amount;

        $fromAccount->save();
        $toAccount->save();

        // Flash success message and close the modal
        $this->alert('success', 'Money transferred successfully!');

        $this->closeModal();
    }

    public function render()
    {
        return view('livewire.account-transfer', [
            'account_transactions' => AccountTransaction::where('transaction_type','transfer')->paginate(50),
        ]);
    }
}
