<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Account;
use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\WithFileUploads;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Str;
use App\Services\TransactionNotificationService;
use Illuminate\Support\Facades\Cache;

class AccountTransactionFormTransfer extends Component
{

    use LivewireAlert, WithPagination;
    use WithFileUploads;
    public $startDate;
    public $endDate;
    protected $paginationTheme = 'bootstrap';
    public $showConvertSellModal = false;
    public $showDigitalCurrency = false;
    public $selectedDigitalCurrency = null;
    public $digitalCurrencies = [];
    public $withoutCustomer = false, $withoutCustomerIQD = false;
    public $file, $IQDaccount, $USDaccount, $operationType, $destinationUSDAccount, $destinationIQDAccount;
    public $moneyTransferOnly = false; // Added property for Money Transfer checkbox
    public $moneyTransferOnlyIQD = false; // Added property for Money Transfer checkbox
    // Create modal properties for USD
    public $create_account_id, $create_person_name, $create_get_fee=0;
    public $create_transaction_type;
    public $create_transaction_amount;
    public $create_transaction_date;
    public $create_transaction_fee=0;
    public $create_transaction_time;
    public $create_note;

   // Add these properties to your Livewire component class
    public $sender_name;
    public $sender_phone;
    public $receiver_name;
    public $receiver_phone;
    public $searchCustomer = '';
    public $customers = [];
    public $selectedCustomerId = null; // This will store the selected customer ID


    // Properties for IQD customer search
    public $searchCustomerIQD = '';
    public $customersIQD = [];
    public $selectedCustomerIdIQD = null;
    public $iqdAccountId = null;
    public $sender_name_iqd = '';
    public $sender_phone_iqd = '';
    public $receiver_name_iqd;
    public $receiver_phone_iqd;

    // Create modal properties for IQD
    public $create_person_name_IQD, $create_get_fee_IQD=0;
    public $create_transaction_type_IQD;
    public $create_transaction_amount_IQD;
    public $create_transaction_date_IQD;
    public $create_transaction_fee_IQD=0;
    public $create_transaction_time_IQD;
    public $create_note_IQD;
    
    public $file_IQD;
    public $amountToConvert, $convertedAmount;

    public $customerOptions;
    public $selectedCustomerName;

    public $filters;
    public $showModal = false, $showEditModal = false;

    public $convert_note, $rate;

    // Define available transaction types (enum)
    public $transactionTypes = ['Select Transaction Type','deposit', 'withdrawal'];
    public $accounts, $account;
    public $transaction_type;

    public $accountsUSD = [];
    public $accountsIQD = [];
    public $selectedCustomer;
    public $createTransaction = [
        'from_account_id' => null,
        'to_account_id' => null,
        'transaction_type' => '',
        'transaction_amount' => '',
        'transaction_date' => '',
        'transaction_time' => '',
        'note' => '',
        'get_fee' => '',
        'person_name' => '',
        'user_id' => null,
        'transaction_id' => '',
    ];

    public function mount()
    {
        $this->filters = [
            'customer' => null,
        ];
        
        // Set default date range to last 3 days
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->toDateString();
        
        $this->digitalCurrencies = \App\Models\Currency::where('type', 1)->get();
        $this->createTransaction['transaction_date'] = Carbon::now()->toDateString();
        $this->createTransaction['transaction_time'] = Carbon::now()->toTimeString();

        $this->customerOptions = Cache::remember(
            'account_transaction_transfer_customer_options',
            now()->addMinutes(5),
            function () {
                return Customer::query()
                    ->select('id', 'customer_name')
                    ->whereNotIn('customer_name', ['Bank (Cashboxes)', 'Cashiers'])
                    ->orderBy('customer_name')
                    ->get()
                    ->mapWithKeys(function ($customer) {
                        return [$customer->id => "{$customer->id} - {$customer->customer_name}"];
                    });
            }
        );
    }

    public function resetDateFilter()
    {
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->toDateString();
    }

    // Updated property to handle money transfer checkbox changes
    public function updatedMoneyTransferOnly()
    {
        if ($this->moneyTransferOnly) {
            // If money transfer is checked, set customer info if available
            if (isset($this->filters['customer'])) {
                $customer = Customer::find($this->filters['customer']);
                if ($customer) {
                    $this->sender_name = $customer->customer_name;
                    $this->sender_phone = $customer->phone_number;
                    
                    // Find USD account if it exists
                    $usdAccount = Account::where('customer_id', $customer->id)
                        ->whereHas('currency', function ($query) {
                            $query->where('currency_name', 'USD');
                        })
                        ->first();
                        
                    if ($usdAccount) {
                        $this->usdAccountId = $usdAccount->id;
                    } else {
                        $this->usdAccountId = null;
                        $this->alert('warning', 'موشتەری حسابی بە دۆلار نیە');
                    }
                }
            }
        }
    }

    public function updatedMoneyTransferOnlyIQD()
    {
        if ($this->moneyTransferOnlyIQD) {
            // If money transfer is checked, set customer info if available
            if (isset($this->filters['customer'])) {
                $customer = Customer::find($this->filters['customer']);
                if ($customer) {
                    $this->sender_name_iqd = $customer->customer_name;
                    $this->sender_phone_iqd = $customer->phone_number;
                    
                    // Find IQD account if it exists
                    $iqdAccount = Account::where('customer_id', $customer->id)
                        ->whereHas('currency', function ($query) {
                            $query->where('currency_name', 'IQD');
                        })
                        ->first();
                        
                    if ($iqdAccount) {
                        $this->iqdAccountId = $iqdAccount->id;
                    } else {
                        $this->iqdAccountId = null;
                        $this->alert('warning', 'موشتەری حسابی بە دینار نیە');
                    }
                }
            }
        }
    }

    // Updated property to handle customer filter changes
    public function updatedFilters($value, $key)
    {
        // If customer filter changes and money transfer is checked, update sender info
        if ($key === 'customer' && $this->moneyTransferOnly && $value) {
            $customer = Customer::find($value);
            if ($customer) {
                $this->sender_name = $customer->customer_name;
                $this->sender_phone = $customer->phone_number;
                
                // Find USD account if it exists
                $usdAccount = Account::where('customer_id', $customer->id)
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'USD');
                    })
                    ->first();
                    
                if ($usdAccount) {
                    $this->usdAccountId = $usdAccount->id;
                } else {
                    $this->usdAccountId = null;
                    $this->alert('warning', 'موشتەری حسابی بە دۆلار نیە');
                }
            }
        }
         // If customer filter changes and IQD money transfer is checked, update sender info
        if ($key === 'customer' && $this->moneyTransferOnlyIQD && $value) {
            $customer = Customer::find($value);
            if ($customer) {
                $this->sender_name_iqd = $customer->customer_name;
                $this->sender_phone_iqd = $customer->phone_number;
                
                // Find IQD account if it exists
                $iqdAccount = Account::where('customer_id', $customer->id)
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();
                    
                if ($iqdAccount) {
                    $this->iqdAccountId = $iqdAccount->id;
                } else {
                    $this->iqdAccountId = null;
                    $this->alert('warning', 'موشتەری حسابی بە دینار نیە');
                }
            }
        }
    }

    // Load all customers initially when the input is focused for IQD
    public function loadCustomersIQD()
    {
        // Load first 10 customers by default when input is focused
        $this->customersIQD = Customer::select('id', 'customer_name', 'phone_number')
            ->limit(10)
            ->get()
            ->toArray();
    }

    // Filter customers as user types for IQD search
    public function updatedSearchCustomerIQD($value)
    {
        if (strlen($value) >= 1) {
            $this->customersIQD = Customer::select('id', 'customer_name', 'phone_number')
                ->where('customer_name', 'like', '%' . $value . '%')
                ->orWhere('phone_number', 'like', '%' . $value . '%')
                ->orWhere('id', $value )
                ->limit(10)
                ->get()
                ->toArray();
        } else {
            // Show initial list if input is cleared
            $this->loadCustomersIQD();
        }
    }

    // Select a customer from the dropdown for IQD
    public function selectCustomerIQD($id)
    {
        $customer = Customer::find($id);
        if ($customer) {
            $this->selectedCustomerIdIQD = $id; // Store the customer ID
            $this->searchCustomerIQD = $customer->customer_name; // Update search field with customer name
            $this->sender_name_iqd = $customer->customer_name;
            $this->sender_phone_iqd = $customer->phone_number;
            
            // Find IQD account if it exists
            $iqdAccount = Account::where('customer_id', $customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();
                
            if ($iqdAccount) {
                // Store the IQD account ID if needed
                $this->iqdAccountId = $iqdAccount->id;
            } else {
                $this->iqdAccountId = null;
                $this->alert('warning', 'موشتەری حسابی بە دینار نیە');
            }
        }
    }

   // Load all customers initially when the input is focused
    public function loadCustomers()
    {
        // Load first 10 customers by default when input is focused
        $this->customers = Customer::select('id', 'customer_name', 'phone_number')
            ->limit(10)
            ->get()
            ->toArray();
    }

    // Filter customers as user types
    public function updatedSearchCustomer($value)
    {
        if (strlen($value) >= 1) {
            $this->customers = Customer::select('id', 'customer_name', 'phone_number')
                ->where('customer_name', 'like', '%' . $value . '%')
                ->orWhere('phone_number', 'like', '%' . $value . '%')
                ->orWhere('id', $value )
                ->limit(10)
                ->get()
                ->toArray();
        } else {
            // Show initial list if input is cleared
            $this->loadCustomers();
        }
    }

    // Select a customer from the dropdown
    public function selectCustomer($id)
    {
        $customer = Customer::find($id);
        if ($customer) {
            $this->selectedCustomerId = $id; // Store the customer ID in a dedicated property
            $this->searchCustomer = $customer->customer_name;
            $this->sender_name = $customer->customer_name;
            $this->sender_phone = $customer->phone_number;
            
            // Find USD account if it exists
            $usdAccount = Account::where('customer_id', $customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
                
            if ($usdAccount) {
                // Store the USD account ID if needed
                $this->usdAccountId = $usdAccount->id;
            } else {
                $this->usdAccountId = null;
                $this->alert('warning', 'موشتەری حسابی بە دۆلار نیە');
            }
        }
    }

    public function updatedShowDigitalCurrency()
    {
        // Reset selected currency when unchecked
        if (!$this->showDigitalCurrency) {
            $this->selectedDigitalCurrency = null;
        }
    }

    public function updatedSelectedDigitalCurrency()
    {
        // Validate only if showDigitalCurrency is checked
        if ($this->showDigitalCurrency) {
            $this->validate([
                'selectedDigitalCurrency' => 'required',
            ]);
        }
    }


    protected $listeners = ['valueSelected'];

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            // It's two words separated by a dot
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
        
        // Update sender information if money transfer is checked and customer is selected
        if ($identifier === 'filters.customer' && $this->moneyTransferOnly && $selectedValue) {
            $customer = Customer::find($selectedValue);
            if ($customer) {
                $this->sender_name = $customer->customer_name;
                $this->sender_phone = $customer->phone_number;
                
                // Find USD account if it exists
                $usdAccount = Account::where('customer_id', $customer->id)
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'USD');
                    })
                    ->first();
                    
                if ($usdAccount) {
                    $this->usdAccountId = $usdAccount->id;
                } else {
                    $this->usdAccountId = null;
                    $this->alert('warning', 'موشتەری حسابی بە دۆلار نیە');
                }
            }
        }

        if ($identifier === 'filters.customer' && $this->moneyTransferOnlyIQD && $selectedValue) {
            $customer = Customer::find($selectedValue);
            if ($customer) {
                $this->sender_name_iqd = $customer->customer_name;
                $this->sender_phone_iqd = $customer->phone_number;
                
                // Find IQD account if it exists
                $iqdAccount = Account::where('customer_id', $customer->id)
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();
                    
                if ($iqdAccount) {
                    $this->iqdAccountId = $iqdAccount->id;
                } else {
                    $this->iqdAccountId = null;
                    $this->alert('warning', 'موشتەری حسابی بە دینار نیە');
                }
            }
        }
    }
    public function saveCreateUSD()
    {
        // Clean numeric values
        $this->create_transaction_amount = str_replace(',', '', $this->create_transaction_amount);
        $this->create_transaction_fee = str_replace(',', '', $this->create_transaction_fee);
        $this->create_get_fee = str_replace(',', '', $this->create_get_fee);

        // Set up validation rules
        $validationRules = [
            'create_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount' => 'required|numeric',
            'create_transaction_fee' => 'required|numeric',
            'create_get_fee' => 'required|numeric',
            'create_person_name' => 'nullable|string',
            'create_note' => 'nullable|string',
            'file' => 'nullable|image|max:1048',
            'receiver_name' => 'required|string',
            'receiver_phone' => 'required|string'
        ];

        // Always require customer selection like in regular transaction form
        $validationRules['filters.customer'] = 'required';
        
        // Add conditional validation based on withoutCustomer or moneyTransferOnly
        if ($this->withoutCustomer || $this->moneyTransferOnly) {
            $validationRules['sender_name'] = 'required|string';
            $validationRules['sender_phone'] = 'required|string';
        } else {
            $validationRules['selectedCustomerId'] = 'required'; // Validate the selectedCustomerId
        }

        // Validate the input fields
        $this->validate($validationRules);

        // Generate unique transaction group ID
        $transactionGroupId = Str::uuid();

        // Get current date and time
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        // Get current user and role
        $user = Auth::user();
        $userRole = $user->getRoleNames()->first(); // Assuming you're using spatie/laravel-permission

        // Determine the system USD account based on user role
        if ($userRole === 'cashier') {
            // Get cashier's USD account
            $systemUSDAccount = Account::where('customer_id', $user->customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
            
            if (!$systemUSDAccount) {
                $this->alert('error', 'کاشیر حسابی بە دۆلار نیە');
                return;
            }
        } 
        elseif ($userRole === 'supervisor') {
            // Get supervisor's USD account
            $systemUSDAccount = Account::where('customer_id', $user->customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
            
            if (!$systemUSDAccount) {
                $this->alert('error', 'سوپەرڤایزەر حسابی بە دۆلار نیە');
                return;
            }
        }
        else {
            // Default behavior for super-admin or other roles
            $systemUSDAccount = Account::where('customer_id', 1)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->where('account_type', 'savings')
                ->first();
                
            if (!$systemUSDAccount) {
                $this->alert('error', 'حسابی سەیڤینگ نەدۆزرایەوە');
                return;
            }
        }
        
        // Get profit account (always needed)
        $profitAccount = Account::where('customer_id', 1)
            ->whereHas('currency', function ($query) {
                $query->where('currency_name', 'USD');
            })
            ->where('account_type', 'profit')
            ->first();

        if (!$profitAccount) {
            $this->alert('error', 'حسابی قازانج بە دۆلار نیە');
            return;
        }

        if ($this->withoutCustomer) {
            // For withoutCustomer mode:
            // fromUSDAccount is the system account
            $fromUSDAccount = $systemUSDAccount;
            // toUSDAccount is from filters['customer']
            $toUSDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
            
            if (!$toUSDAccount) {
                $this->alert('error', 'موشتەری حسابی بە دۆلار نیە');
                return;
            }
            
            // Create transaction from system account (fromUSDAccount)
            // System account receives: create_transaction_fee + create_transaction_amount
            $transactionFrom = AccountTransactionModel::create([
                'from_account_id' => $fromUSDAccount->id,
                'to_account_id' => $toUSDAccount->id,
                'person_name' => $this->create_person_name,
                'get_fee' => $this->create_transaction_fee,
                'transaction_type' => $this->create_transaction_type,
                'transaction_amount' => $this->create_transaction_amount + $this->create_transaction_fee,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name,
                'sender_phone' => $this->sender_phone,
                'receiver_name' => $this->receiver_name,
                'receiver_phone' => $this->receiver_phone
            ]);

            // Handle file upload
            if ($this->file) {
                try {
                    $filePath = $this->file->store('files', 'public');
                    $transactionFrom->files()->create([
                        'file_path' => $filePath,
                    ]);
                } catch (\Exception $e) {
                    dd($e->getMessage());
                }
            }

            // Create corresponding transaction for customer account (toUSDAccount)
            // Customer account receives: create_get_fee + create_transaction_amount
            $transactionTo = AccountTransactionModel::create([
                'from_account_id' => $toUSDAccount->id,
                'to_account_id' => $fromUSDAccount->id,
                'person_name' => $this->create_person_name,
                'get_fee' => $this->create_get_fee,
                'transaction_type' => $this->create_transaction_type,
                'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name,
                'sender_phone' => $this->sender_phone,
                'receiver_name' => $this->receiver_name,
                'receiver_phone' => $this->receiver_phone
            ]);

            if($this->create_transaction_fee - $this->create_get_fee > 0 ){
                // Create profit transaction
                $transactionProfit = AccountTransactionModel::create([
                    'from_account_id' => $profitAccount->id,
                    'to_account_id' => $fromUSDAccount->id,
                    'person_name' => $this->create_person_name,
                    'get_fee' => 0,
                    'transaction_type' => 'deposit',
                    'transaction_amount' => $this->create_transaction_fee - $this->create_get_fee,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name,
                    'sender_phone' => $this->sender_phone,
                    'receiver_name' => $this->receiver_name,
                    'receiver_phone' => $this->receiver_phone
                ]);
            }
        }elseif($this->moneyTransferOnly) {

        // For withoutCustomer or moneyTransferOnly mode:
            // fromUSDAccount is the system account
            $fromUSDAccount = $systemUSDAccount;
            
            // toUSDAccount is from filters['customer']
            $toUSDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
            
            if (!$toUSDAccount) {
                $this->alert('error', 'موشتەری حسابی بە دۆلار نیە');
                return;
            }
            
            // Create transaction from system account (fromUSDAccount)
            $transactionFrom = AccountTransactionModel::create([
                'from_account_id' => $fromUSDAccount->id,
                'to_account_id' => $toUSDAccount->id,
                'person_name' => $this->create_person_name,
                'get_fee' => $this->create_transaction_fee,
                'transaction_type' => $this->create_transaction_type,
                'transaction_amount' => $this->create_transaction_amount + $this->create_transaction_fee,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name,
                'sender_phone' => $this->sender_phone,
                'receiver_name' => $this->receiver_name,
                'receiver_phone' => $this->receiver_phone,
            ]);

            // Handle file upload
            if ($this->file) {
                try {
                    $filePath = $this->file->store('files', 'public');
                    $transactionFrom->files()->create([
                        'file_path' => $filePath,
                    ]);
                } catch (\Exception $e) {
                    dd($e->getMessage());
                }
            }

            // Create corresponding transaction for customer account (toUSDAccount)
            $transactionTo = AccountTransactionModel::create([
                'from_account_id' => $toUSDAccount->id,
                'to_account_id' => $fromUSDAccount->id,
                'person_name' => $this->create_person_name,
                'get_fee' => $this->create_get_fee,
                'transaction_type' => $this->create_transaction_type,
                'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name,
                'sender_phone' => $this->sender_phone,
                'receiver_name' => $this->receiver_name,
                'receiver_phone' => $this->receiver_phone,
                'moneyTransferOnly'=> 1
            ]);

            if($this->create_transaction_fee - $this->create_get_fee > 0 ){
                // Create profit transaction
                $transactionProfit = AccountTransactionModel::create([
                    'from_account_id' => $profitAccount->id,
                    'to_account_id' => $fromUSDAccount->id,
                    'person_name' => $this->create_person_name,
                    'get_fee' => 0,
                    'transaction_type' => 'deposit',
                    'transaction_amount' => $this->create_transaction_fee - $this->create_get_fee,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name,
                    'sender_phone' => $this->sender_phone,
                    'receiver_name' => $this->receiver_name,
                    'receiver_phone' => $this->receiver_phone
            ]);
        }
       } else {
            // For customer account mode:
            // Get the customer's USD account
            $customerUSDAccount = Account::where('customer_id', $this->selectedCustomerId)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();

            if (!$customerUSDAccount) {
                $this->alert('error', 'موشتەری حسابی بە دۆلار نیە');
                return;
            }
            
            // Get the target account from filters['customer']
            $targetUSDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
                
            if (!$targetUSDAccount) {
                $this->alert('error', 'موشتەری دووەم حسابی بە دۆلار نیە');
                return;
            }
            
            // Check if selected customer is the same as target customer
            if ($this->filters['customer'] == $this->selectedCustomerId) {
                // Only create one transaction when same customer is selected
                $transaction = AccountTransactionModel::create([
                    'from_account_id' => $customerUSDAccount->id,
                    'to_account_id' => $systemUSDAccount->id,
                    'person_name' => $this->create_person_name,
                    'get_fee' => $this->create_transaction_fee,
                    'transaction_type' => $this->create_transaction_type,
                    'transaction_amount' => $this->create_transaction_amount,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => $this->create_note,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name ?? '',
                    'sender_phone' => $this->sender_phone ?? '',
                    'receiver_name' => $this->receiver_name,
                    'receiver_phone' => $this->receiver_phone
                ]);
                
                // Handle file upload
                if ($this->file) {
                    try {
                        $filePath = $this->file->store('files', 'public');
                        $transaction->files()->create([
                            'file_path' => $filePath,
                        ]);
                    } catch (\Exception $e) {
                        dd($e->getMessage());
                    }
                }
                
                // Create profit transaction with create_get_fee as amount
                $transactionProfit = AccountTransactionModel::create([
                    'from_account_id' => $profitAccount->id,
                    'to_account_id' => $customerUSDAccount->id,
                    'person_name' => $this->create_person_name,
                    'get_fee' => 0,
                    'status' => 'pending',
                    'transaction_type' => 'deposit',
                    'transaction_amount' => $this->create_get_fee,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                    'user_id' => Auth::id(),
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name ?? '',
                    'sender_phone' => $this->sender_phone ?? '',
                    'receiver_name' => $this->receiver_name,
                    'receiver_phone' => $this->receiver_phone
                ]);
            } else {
                // Define transaction details based on transaction type
                if ($this->create_transaction_type === 'deposit') {
                    // DEPOSIT: Money goes from customer's account to target account
                    
                    // Create transaction from customer account (deposit)
                    $transactionFrom = AccountTransactionModel::create([
                        'from_account_id' => $customerUSDAccount->id,
                        'to_account_id' => $targetUSDAccount->id,
                        'person_name' => $this->create_person_name,
                        'get_fee' => $this->create_transaction_fee,
                        'transaction_type' => 'withdrawal',
                        'transaction_amount' => $this->create_transaction_amount + $this->create_transaction_fee,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => $this->create_note,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name ?? '',
                        'sender_phone' => $this->sender_phone ?? '',
                        'receiver_name' => $this->receiver_name,
                        'receiver_phone' => $this->receiver_phone
                    ]);
                    
                    // Handle file upload
                    if ($this->file) {
                        try {
                            $filePath = $this->file->store('files', 'public');
                            $transactionFrom->files()->create([
                                'file_path' => $filePath,
                            ]);
                        } catch (\Exception $e) {
                            dd($e->getMessage());
                        }
                    }
                    
                    // Create transaction for target account (withdrawal)
                    $transactionTo = AccountTransactionModel::create([
                        'from_account_id' => $targetUSDAccount->id,
                        'to_account_id' => $customerUSDAccount->id,
                        'person_name' => $this->create_person_name,
                        'get_fee' => $this->create_get_fee,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => $this->create_note,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name ?? '',
                        'sender_phone' => $this->sender_phone ?? '',
                        'receiver_name' => $this->receiver_name,
                        'receiver_phone' => $this->receiver_phone
                    ]);
                } else {
                    // WITHDRAWAL: Money goes from target account to customer's account
                    
                    // Create transaction from target account (deposit)
                    $transactionFrom = AccountTransactionModel::create([
                        'from_account_id' => $targetUSDAccount->id,
                        'to_account_id' => $customerUSDAccount->id,
                        'person_name' => $this->create_person_name,
                        'get_fee' => $this->create_transaction_fee,
                        'transaction_type' => 'withdrawal',
                        'transaction_amount' => $this->create_transaction_amount + $this->create_transaction_fee,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => $this->create_note,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name ?? '',
                        'sender_phone' => $this->sender_phone ?? '',
                        'receiver_name' => $this->receiver_name,
                        'receiver_phone' => $this->receiver_phone
                    ]);
                    
                    // Handle file upload
                    if ($this->file) {
                        try {
                            $filePath = $this->file->store('files', 'public');
                            $transactionFrom->files()->create([
                                'file_path' => $filePath,
                            ]);
                        } catch (\Exception $e) {
                            dd($e->getMessage());
                        }
                    }
                    
                    // Create transaction for customer account (withdrawal)
                    $transactionTo = AccountTransactionModel::create([
                        'from_account_id' => $customerUSDAccount->id,
                        'to_account_id' => $targetUSDAccount->id,
                        'person_name' => $this->create_person_name,
                        'get_fee' => $this->create_get_fee,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => $this->create_note,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name ?? '',
                        'sender_phone' => $this->sender_phone ?? '',
                        'receiver_name' => $this->receiver_name,
                        'receiver_phone' => $this->receiver_phone
                    ]);
                }

                // Create profit transaction for the fee
                $transactionProfit = AccountTransactionModel::create([
                    'from_account_id' => $profitAccount->id,
                    'to_account_id' => $targetUSDAccount->id,
                    'person_name' => $this->create_person_name,
                    'get_fee' => 0,
                    'transaction_type' => 'deposit',
                    'transaction_amount' => $this->create_transaction_fee - $this->create_get_fee,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name ?? '',
                    'sender_phone' => $this->sender_phone ?? '',
                    'receiver_name' => $this->receiver_name,
                    'receiver_phone' => $this->receiver_phone
                ]);
            }
        }

        // Send notification to the customer whose account was affected
        try {
            // Send notification for the main transaction (transactionFrom or transactionTo depending on context)
            if (isset($transactionFrom)) {
                $this->sendTransactionNotification($transactionFrom);
            } elseif (isset($transactionTo)) {
                $this->sendTransactionNotification($transactionTo);
            }
        } catch (\Exception $e) {
            Log::error('Failed to send transaction notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }

        // Flash success message
        $this->alert('success', 'مامەڵەکە بەسەرکەوتووی تۆمارکرا');

        // Reset form fields
        $this->reset([
            'create_note', 
            'create_transaction_fee', 
            'create_transaction_amount',
            'create_get_fee', 
            'create_person_name',
            'file', 
            'create_transaction_type',
            'sender_name',
            'sender_phone',
            'receiver_name',
            'receiver_phone',
            'selectedCustomerId',
            'searchCustomer',
            'moneyTransferOnly'
        ]);
    }

    public function saveCreateIQD()
    {
        // Clean numeric values
        $this->create_transaction_amount_IQD = str_replace(',', '', $this->create_transaction_amount_IQD);
        $this->create_transaction_fee_IQD = str_replace(',', '', $this->create_transaction_fee_IQD);
        $this->create_get_fee_IQD = str_replace(',', '', $this->create_get_fee_IQD);

        // Set up validation rules
        $validationRules = [
            'create_transaction_type_IQD' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount_IQD' => 'required|numeric',
            'create_transaction_fee_IQD' => 'required|numeric',
            'create_get_fee_IQD' => 'required|numeric',
            'create_person_name_IQD' => 'nullable|string',
            'create_note_IQD' => 'nullable|string',
            'file_IQD' => 'nullable|image|max:1048',
            'receiver_name_iqd' => 'required|string',
            'receiver_phone_iqd' => 'required|string',
            'selectedDigitalCurrency' => $this->showDigitalCurrency ? 'required' : 'nullable'
        ];

        // Always require customer selection like in regular transaction form
        $validationRules['filters.customer'] = 'required';
        
        // Add conditional validation based on withoutCustomerIQD or moneyTransferOnlyIQD
        if ($this->withoutCustomerIQD || $this->moneyTransferOnlyIQD) {
            $validationRules['sender_name_iqd'] = 'required|string';
            $validationRules['sender_phone_iqd'] = 'required|string';
        } else {
            $validationRules['selectedCustomerIdIQD'] = 'required'; // Validate the selectedCustomerId
        }

        // Validate the input fields
        $this->validate($validationRules);

        // Generate unique transaction group ID
        $transactionGroupId = Str::uuid();

        // Get current date and time
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        // Get current user and role
        $user = Auth::user();
        $userRole = $user->getRoleNames()->first(); // Assuming you're using spatie/laravel-permission

        // Determine the system IQD account based on user role
        if ($userRole === 'cashier') {
            // Get cashier's IQD account
            $systemIQDAccount = Account::where('customer_id', $user->customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();
            
            if (!$systemIQDAccount) {
                $this->alert('error', 'کاشیر حسابی بە دینار نیە');
                return;
            }
        } 
        elseif ($userRole === 'supervisor') {
            // Get supervisor's IQD account
            $systemIQDAccount = Account::where('customer_id', $user->customer->id)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();
            
            if (!$systemIQDAccount) {
                $this->alert('error', 'سوپەرڤایزەر حسابی بە دینار نیە');
                return;
            }
        }
        else {
            // Default behavior for super-admin or other roles
            $systemIQDAccount = Account::where('customer_id', 1)
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->where('account_type', 'savings')
                ->first();
                
            if (!$systemIQDAccount) {
                $this->alert('error', 'حسابی سەیڤینگ نەدۆزرایەوە');
                return;
            }
        }

        // Get profit account (always needed)
        $profitAccount = Account::where('customer_id', 1)
            ->whereHas('currency', function ($query) {
                $query->where('currency_name', 'IQD');
            })
            ->where('account_type', 'profit')
            ->first();

        if (!$profitAccount) {
            $this->alert('error', 'حسابی قازانج بە دینار نیە');
            return;
        }

        // Handle digital currency transactions if selected
        if ($this->showDigitalCurrency) {
            // Get digital currency account
            $digitalCurrencyAccount = Account::where('customer_id', 1)
                ->where('currency_id', $this->selectedDigitalCurrency)
                ->where('account_type', 'savings')
                ->first();
                
            if (!$digitalCurrencyAccount) {
                $this->alert('error', 'حسابی دیجیتاڵ نیە');
                return;
            }
            
            // Get customer's IQD account (from filters, not selected customer)
            $customerIQDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();
                
            if (!$customerIQDAccount) {
                $this->alert('error', 'موشتەری حسابی بە دینار نیە');
                return;
            }
            
            // Create transaction from customer IQD account to digital currency account
            $transactionFrom = AccountTransactionModel::create([
                'from_account_id' => $customerIQDAccount->id,
                'to_account_id' => $digitalCurrencyAccount->id,
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD,
                'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name_iqd ?? '',
                'sender_phone' => $this->sender_phone_iqd ?? '',
                'receiver_name' => $this->receiver_name_iqd,
                'receiver_phone' => $this->receiver_phone_iqd
            ]);
            
            // Handle file upload
            if ($this->file_IQD) {
                try {
                    $filePath = $this->file_IQD->store('files', 'public');
                    $transactionFrom->files()->create([
                        'file_path' => $filePath,
                    ]);
                } catch (\Exception $e) {
                    dd($e->getMessage());
                }
            }
            
            // Create corresponding transaction for digital currency account to customer
            $transactionToDigital = AccountTransactionModel::create([
                'from_account_id' => $digitalCurrencyAccount->id,
                'to_account_id' => $customerIQDAccount->id,
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD === 'deposit' ? 'deposit' : 'withdrawal',
                'transaction_amount' => $this->create_transaction_amount_IQD,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name_iqd ?? '',
                'sender_phone' => $this->sender_phone_iqd ?? '',
                'receiver_name' => $this->receiver_name_iqd,
                'receiver_phone' => $this->receiver_phone_iqd
            ]);
            
            // Create profit transaction (profit goes to digital currency account)
            $transactionProfit = AccountTransactionModel::create([
                'from_account_id' => $profitAccount->id,
                'to_account_id' => $digitalCurrencyAccount->id,
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => 0,
                'transaction_type' => 'deposit',
                'transaction_amount' => $this->create_get_fee_IQD,
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                'user_id' => Auth::id(),
                'status' => 'pending',
                'transaction_group_id' => $transactionGroupId,
                'sender_name' => $this->sender_name_iqd ?? '',
                'sender_phone' => $this->sender_phone_iqd ?? '',
                'receiver_name' => $this->receiver_name_iqd,
                'receiver_phone' => $this->receiver_phone_iqd
            ]);
        } else {
            if ($this->withoutCustomerIQD) {
                // For withoutCustomer or moneyTransferOnly mode:
                // fromIQDAccount is the system account
                $fromIQDAccount = $systemIQDAccount;
                
                // toIQDAccount is from filters['customer']
                $toIQDAccount = Account::where('customer_id', $this->filters['customer'])
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();
                
                if (!$toIQDAccount) {
                    $this->alert('error', 'موشتەری حسابی بە دینار نیە');
                    return;
                }
                
                // Create transaction from system account (fromIQDAccount)
                // System account receives: create_transaction_fee_IQD + create_transaction_amount_IQD
                $transactionFrom = AccountTransactionModel::create([
                    'from_account_id' => $fromIQDAccount->id,
                    'to_account_id' => $toIQDAccount->id,
                    'person_name' => $this->create_person_name_IQD,
                    'get_fee' => $this->create_transaction_fee_IQD,
                    'transaction_type' => $this->create_transaction_type_IQD,
                    'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_transaction_fee_IQD,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => $this->create_note_IQD,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name_iqd,
                    'sender_phone' => $this->sender_phone_iqd,
                    'receiver_name' => $this->receiver_name_iqd,
                    'receiver_phone' => $this->receiver_phone_iqd
                ]);

                // Handle file upload
                if ($this->file_IQD) {
                    try {
                        $filePath = $this->file_IQD->store('files', 'public');
                        $transactionFrom->files()->create([
                            'file_path' => $filePath,
                        ]);
                    } catch (\Exception $e) {
                        dd($e->getMessage());
                    }
                }

                // Create corresponding transaction for customer account (toIQDAccount)
                // Customer account receives: create_get_fee_IQD + create_transaction_amount_IQD
                $transactionTo = AccountTransactionModel::create([
                    'from_account_id' => $toIQDAccount->id,
                    'to_account_id' => $fromIQDAccount->id,
                    'person_name' => $this->create_person_name_IQD,
                    'get_fee' => $this->create_get_fee_IQD,
                    'transaction_type' => $this->create_transaction_type_IQD,
                    'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => $this->create_note_IQD,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name_iqd,
                    'sender_phone' => $this->sender_phone_iqd,
                    'receiver_name' => $this->receiver_name_iqd,
                    'receiver_phone' => $this->receiver_phone_iqd
                ]);

                if($this->create_transaction_fee_IQD - $this->create_get_fee_IQD > 0 ){
                    // Create profit transaction
                    $transactionProfit = AccountTransactionModel::create([
                        'from_account_id' => $profitAccount->id,
                        'to_account_id' => $fromIQDAccount->id,
                        'person_name' => $this->create_person_name_IQD,
                        'get_fee' => 0,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_transaction_fee_IQD - $this->create_get_fee_IQD ,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name_iqd,
                        'sender_phone' => $this->sender_phone_iqd,
                        'receiver_name' => $this->receiver_name_iqd,
                        'receiver_phone' => $this->receiver_phone_iqd
                    ]);
                }
            } elseif ($this->moneyTransferOnlyIQD) {
                
                // For withoutCustomer or moneyTransferOnly mode:
                // fromIQDAccount is the system account
                $fromIQDAccount = $systemIQDAccount;
                
                // toIQDAccount is from filters['customer']
                $toIQDAccount = Account::where('customer_id', $this->filters['customer'])
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();
                
                if (!$toIQDAccount) {
                    $this->alert('error', 'موشتەری حسابی بە دینار نیە');
                    return;
                }
                
                // Create transaction from system account (fromIQDAccount)
                $transactionFrom = AccountTransactionModel::create([
                    'from_account_id' => $fromIQDAccount->id,
                    'to_account_id' => $toIQDAccount->id,
                    'person_name' => $this->create_person_name_IQD,
                    'get_fee' => $this->create_transaction_fee_IQD,
                    'transaction_type' => $this->create_transaction_type_IQD,
                    'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_transaction_fee_IQD,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => $this->create_note_IQD,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name_iqd,
                    'sender_phone' => $this->sender_phone_iqd,
                    'receiver_name' => $this->receiver_name_iqd,
                    'receiver_phone' => $this->receiver_phone_iqd
                ]);

                // Handle file upload
                if ($this->file_IQD) {
                    try {
                        $filePath = $this->file_IQD->store('files', 'public');
                        $transactionFrom->files()->create([
                            'file_path' => $filePath,
                        ]);
                    } catch (\Exception $e) {
                        dd($e->getMessage());
                    }
                }

                // Create corresponding transaction for customer account (toIQDAccount)
                $transactionTo = AccountTransactionModel::create([
                    'from_account_id' => $toIQDAccount->id,
                    'to_account_id' => $fromIQDAccount->id,
                    'person_name' => $this->create_person_name_IQD,
                    'get_fee' => $this->create_get_fee_IQD,
                    'transaction_type' => $this->create_transaction_type_IQD,
                    'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD,
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => $this->create_note_IQD,
                    'user_id' => Auth::id(),
                    'status' => 'pending',
                    'transaction_group_id' => $transactionGroupId,
                    'sender_name' => $this->sender_name_iqd,
                    'sender_phone' => $this->sender_phone_iqd,
                    'receiver_name' => $this->receiver_name_iqd,
                    'moneyTransferOnly' => 1,
                    'receiver_phone' => $this->receiver_phone_iqd
                ]);

                if($this->create_transaction_fee_IQD - $this->create_get_fee_IQD > 0 ){
                    // Create profit transaction
                    $transactionProfit = AccountTransactionModel::create([
                        'from_account_id' => $profitAccount->id,
                        'to_account_id' => $fromIQDAccount->id,
                        'person_name' => $this->create_person_name_IQD,
                        'get_fee' => 0,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_transaction_fee_IQD - $this->create_get_fee_IQD ,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name_iqd,
                        'sender_phone' => $this->sender_phone_iqd,
                        'receiver_name' => $this->receiver_name_iqd,
                        'receiver_phone' => $this->receiver_phone_iqd
                    ]);
                }

            }else {
                // Get the customer's IQD account
                $customerIQDAccount = Account::where('customer_id', $this->selectedCustomerIdIQD)
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();

                if (!$customerIQDAccount) {
                    $this->alert('error', 'موشتەری حسابی بە دینار نیە');
                    return;
                }
                
                // Get the target account from filters['customer']
                $targetIQDAccount = Account::where('customer_id', $this->filters['customer'])
                    ->whereHas('currency', function ($query) {
                        $query->where('currency_name', 'IQD');
                    })
                    ->first();
                    
                if (!$targetIQDAccount) {
                    $this->alert('error', 'موشتەری دووەم حسابی بە دینار نیە');
                    return;
                }
                
                // Check if selected customer is the same as target customer
                if ($this->filters['customer'] == $this->selectedCustomerIdIQD) {
                    // Only create one transaction when same customer is selected
                    $transaction = AccountTransactionModel::create([
                        'from_account_id' => $customerIQDAccount->id,
                        'to_account_id' => $systemIQDAccount->id,
                        'person_name' => $this->create_person_name_IQD,
                        'get_fee' => $this->create_transaction_fee_IQD,
                        'transaction_type' => $this->create_transaction_type_IQD,
                        'transaction_amount' => $this->create_transaction_amount_IQD,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => $this->create_note_IQD,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name_iqd ?? '',
                        'sender_phone' => $this->sender_phone_iqd ?? '',
                        'receiver_name' => $this->receiver_name_iqd,
                        'receiver_phone' => $this->receiver_phone_iqd
                    ]);
                    
                    // Handle file upload
                    if ($this->file_IQD) {
                        try {
                            $filePath = $this->file_IQD->store('files', 'public');
                            $transaction->files()->create([
                                'file_path' => $filePath,
                            ]);
                        } catch (\Exception $e) {
                            dd($e->getMessage());
                        }
                    }
                    
                    // Create profit transaction with create_get_fee as amount
                    $transactionProfit = AccountTransactionModel::create([
                        'from_account_id' => $profitAccount->id,
                        'to_account_id' => $customerIQDAccount->id,
                        'person_name' => $this->create_person_name_IQD,
                        'get_fee' => 0,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_get_fee_IQD,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name_iqd ?? '',
                        'sender_phone' => $this->sender_phone_iqd ?? '',
                        'receiver_name' => $this->receiver_name_iqd,
                        'receiver_phone' => $this->receiver_phone_iqd
                    ]);
                } else {
                    // Define transaction details based on transaction type
                    if ($this->create_transaction_type_IQD === 'deposit') {
                        // DEPOSIT: Money goes from customer's account to target account
                        
                        // Create transaction from customer account (deposit)
                        $transactionFrom = AccountTransactionModel::create([
                            'from_account_id' => $customerIQDAccount->id,
                            'to_account_id' => $targetIQDAccount->id,
                            'person_name' => $this->create_person_name_IQD,
                            'get_fee' => $this->create_transaction_fee_IQD,
                            'transaction_type' => 'withdrawal',
                            'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_transaction_fee_IQD,
                            'transaction_date' => $currentDate,
                            'transaction_time' => $currentTime,
                            'note' => $this->create_note_IQD,
                            'user_id' => Auth::id(),
                            'status' => 'pending',
                            'transaction_group_id' => $transactionGroupId,
                            'sender_name' => $this->sender_name_iqd ?? '',
                            'sender_phone' => $this->sender_phone_iqd ?? '',
                            'receiver_name' => $this->receiver_name_iqd,
                            'receiver_phone' => $this->receiver_phone_iqd
                        ]);
                        
                        // Handle file upload
                        if ($this->file_IQD) {
                            try {
                                $filePath = $this->file_IQD->store('files', 'public');
                                $transactionFrom->files()->create([
                                    'file_path' => $filePath,
                                ]);
                            } catch (\Exception $e) {
                                dd($e->getMessage());
                            }
                        }
                        
                        // Create transaction for target account (withdrawal)
                        $transactionTo = AccountTransactionModel::create([
                            'from_account_id' => $targetIQDAccount->id,
                            'to_account_id' => $customerIQDAccount->id,
                            'person_name' => $this->create_person_name_IQD,
                            'get_fee' => $this->create_get_fee_IQD,
                            'transaction_type' => 'deposit',
                            'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD,
                            'transaction_date' => $currentDate,
                            'transaction_time' => $currentTime,
                            'note' => $this->create_note_IQD,
                            'user_id' => Auth::id(),
                            'status' => 'pending',
                            'transaction_group_id' => $transactionGroupId,
                            'sender_name' => $this->sender_name_iqd ?? '',
                            'sender_phone' => $this->sender_phone_iqd ?? '',
                            'receiver_name' => $this->receiver_name_iqd,
                            'receiver_phone' => $this->receiver_phone_iqd
                        ]);
                    } else {
                        // WITHDRAWAL: Money goes from target account to customer's account
                        
                        // Create transaction from target account (deposit)
                        $transactionFrom = AccountTransactionModel::create([
                            'from_account_id' => $targetIQDAccount->id,
                            'to_account_id' => $customerIQDAccount->id,
                            'person_name' => $this->create_person_name_IQD,
                            'get_fee' => $this->create_transaction_fee_IQD,
                            'transaction_type' => 'withdrawal',
                            'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_transaction_fee_IQD,
                            'transaction_date' => $currentDate,
                            'transaction_time' => $currentTime,
                            'note' => $this->create_note_IQD,
                            'user_id' => Auth::id(),
                            'status' => 'pending',
                            'transaction_group_id' => $transactionGroupId,
                            'sender_name' => $this->sender_name_iqd ?? '',
                            'sender_phone' => $this->sender_phone_iqd ?? '',
                            'receiver_name' => $this->receiver_name_iqd,
                            'receiver_phone' => $this->receiver_phone_iqd
                        ]);
                        
                        // Handle file upload
                        if ($this->file_IQD) {
                            try {
                                $filePath = $this->file_IQD->store('files', 'public');
                                $transactionFrom->files()->create([
                                    'file_path' => $filePath,
                                ]);
                            } catch (\Exception $e) {
                                dd($e->getMessage());
                            }
                        }
                        
                        // Create transaction for customer account (withdrawal)
                        $transactionTo = AccountTransactionModel::create([
                            'from_account_id' => $customerIQDAccount->id,
                            'to_account_id' => $targetIQDAccount->id,
                            'person_name' => $this->create_person_name_IQD,
                            'get_fee' => $this->create_get_fee_IQD,
                            'transaction_type' => 'deposit',
                            'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD,
                            'transaction_date' => $currentDate,
                            'transaction_time' => $currentTime,
                            'note' => $this->create_note_IQD,
                            'user_id' => Auth::id(),
                            'status' => 'pending',
                            'transaction_group_id' => $transactionGroupId,
                            'sender_name' => $this->sender_name_iqd ?? '',
                            'sender_phone' => $this->sender_phone_iqd ?? '',
                            'receiver_name' => $this->receiver_name_iqd,
                            'receiver_phone' => $this->receiver_phone_iqd
                        ]);
                    }

                    // Create profit transaction for the fee
                    $transactionProfit = AccountTransactionModel::create([
                        'from_account_id' => $profitAccount->id,
                        'to_account_id' => $targetIQDAccount->id,
                        'person_name' => $this->create_person_name_IQD,
                        'get_fee' => 0,
                        'transaction_type' => 'deposit',
                        'transaction_amount' => $this->create_transaction_fee_IQD - $this->create_get_fee_IQD,
                        'transaction_date' => $currentDate,
                        'transaction_time' => $currentTime,
                        'note' => 'قازانج لە لە حەوالە :  ' . $transactionGroupId,
                        'user_id' => Auth::id(),
                        'status' => 'pending',
                        'transaction_group_id' => $transactionGroupId,
                        'sender_name' => $this->sender_name_iqd ?? '',
                        'sender_phone' => $this->sender_phone_iqd ?? '',
                        'receiver_name' => $this->receiver_name_iqd,
                        'receiver_phone' => $this->receiver_phone_iqd
                    ]);
                }
            }
        }
        
        // Reset form fields
        $this->reset([
            'create_note_IQD', 
            'create_transaction_fee_IQD', 
            'create_transaction_amount_IQD',
            'create_get_fee_IQD', 
            'create_person_name_IQD',
            'file_IQD', 
            'create_transaction_type_IQD',
            'selectedDigitalCurrency', 
            'showDigitalCurrency',
            'sender_name_iqd',
            'sender_phone_iqd',
            'receiver_name_iqd',
            'receiver_phone_iqd',
            'selectedCustomerIdIQD',
            'searchCustomerIQD',
            'moneyTransferOnlyIQD'
        ]);

        // Send notification to the customer whose account was affected
        try {
            // Send notification for the main transaction (transactionFrom or transactionTo depending on context)
            if (isset($transactionFrom)) {
                $this->sendTransactionNotification($transactionFrom);
            } elseif (isset($transactionTo)) {
                $this->sendTransactionNotification($transactionTo);
            }
        } catch (\Exception $e) {
            Log::error('Failed to send transaction notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }

        // Flash success message
        $this->alert('success', 'مامەڵەکە بەسەرکەوتووی تۆمارکرا');
    }

    public function render()
    {
        $USDAccount = null;
        $IQDAccount = null;

        $this->selectedCustomer = null;
        $this->selectedCustomerName = null;

        if ($this->filters['customer']) {
            $customer = Customer::select('id', 'customer_name')->find($this->filters['customer']);
            if ($customer) {
                $this->selectedCustomer = $customer->id;
                $this->selectedCustomerName = $customer->customer_name;
            }

            // Fetch the first USD account for the selected customer
            $USDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();
                
            // Apply date filtering to USD account transactions
            if ($USDAccount) {
                $USDAccount->load(['pendingTransactions' => function ($query) {
                    $query->whereBetween('transaction_date', [$this->startDate, $this->endDate])
                        ->orderBy('transaction_date', 'desc')
                        ->orderBy('transaction_time', 'desc');
                }]);
            }
            
            $this->destinationUSDAccount = $USDAccount;

            // Fetch the first IQD account for the selected customer
            $IQDAccount = Account::where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();
                
            // Apply date filtering to IQD account transactions
            if ($IQDAccount) {
                $IQDAccount->load(['pendingTransactions' => function ($query) {
                    $query->whereBetween('transaction_date', [$this->startDate, $this->endDate])
                        ->orderBy('transaction_date', 'desc')
                        ->orderBy('transaction_time', 'desc');
                }]);
            }
            
            $this->destinationIQDAccount = $IQDAccount;
            $this->USDaccount = $USDAccount;
            $this->IQDaccount = $IQDAccount;
        }

        return view('livewire.account-transaction-form-transfer', [
            'USDAccount' => $USDAccount,
            'IQDAccount' => $IQDAccount,
        ]);
    }

    /**
     * Send notification for new transaction
     */
    private function sendTransactionNotification(AccountTransactionModel $transaction)
    {
        // Resolve the notification service when needed
        $notificationService = app(TransactionNotificationService::class);
        $notificationService->sendTransactionCreatedNotification($transaction);
    }
    
}
