<?php

namespace App\Livewire;

use App\Livewire\BaseComponent;
use App\Models\Account;
use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\Customer;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Livewire\WithFileUploads;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Illuminate\Support\Str;
use App\Services\TransactionNotificationService;

class AccountTransactionForm extends BaseComponent
{

    use LivewireAlert, WithPagination;
    use WithFileUploads;

    protected $paginationTheme = 'bootstrap';
    
    // Add UTF-8 encoding support
    protected $encoding = 'UTF-8';
    
    public $showConvertSellModal = false;
    public $showDigitalCurrency = false;
    public $selectedDigitalCurrency = null;
    public $digitalCurrencies = [];
    public $withoutCustomer = false, $withoutCustomerIQD = false;
    public $file, $IQDaccount, $USDaccount, $operationType, $destinationUSDAccount, $destinationIQDAccount;

    public $startDate;
    public $endDate;
    // Create modal properties for USD
    public $create_account_id, $create_person_name, $create_get_fee=0;
    public $create_transaction_type;
    public $create_transaction_amount;
    public $create_transaction_date;
    public $create_transaction_fee=0;
    public $create_transaction_time;
    public $create_note;

    // Create modal properties for IQD
    public $create_person_name_IQD, $create_get_fee_IQD=0;
    public $create_transaction_type_IQD;
    public $create_transaction_amount_IQD;
    public $create_transaction_date_IQD;
    public $create_transaction_fee_IQD=0;
    public $create_transaction_time_IQD;
    public $create_note_IQD;
    
    public $file_IQD;
    public $amountToConvert, $convertedAmount;

    public $filters;
    public $showModal = false, $showEditModal = false;

    public $convert_note, $rate;

    // Define available transaction types (enum)
    public $transactionTypes = ['Select Transaction Type','deposit', 'withdrawal'];
    public $accounts, $account;
    public $transaction_type;

    public $accountsUSD = [];
    public $accountsIQD = [];
    public $selectedCustomer;
    public $createTransaction = [
        'from_account_id' => null,
        'to_account_id' => null,
        'transaction_type' => '',
        'transaction_amount' => '',
        'transaction_date' => '',
        'transaction_time' => '',
        'note' => '',
        'get_fee' => '',
        'person_name' => '',
        'user_id' => null,
        'transaction_id' => '',
    ];
    public $customerOptions = [];
    
    public function openConvertSellModal()
    {
        $this->showConvertSellModal = true;
    }

    public function closeConvertSellModal()
    {
        $this->rate = '';
        $this->convert_note = '';
        $this->amountToConvert = null;
        $this->transaction_type = '';
        $this->operationType = '';
        $this->showConvertSellModal = false;
    }

    public function mount()
    {
       

        $this->filters = [
            'customer' => null,
        ];
        $this->digitalCurrencies = \App\Models\Currency::select('id', 'currency_name')
            ->where('type', 1)
            ->orderBy('currency_name')
            ->get();
        $this->customerOptions = Cache::remember(
            'account_transaction_form_customer_options',
            now()->addMinutes(10),
            function () {
                return Customer::select('id', 'customer_name')
                    ->whereNotIn('customer_name', ['Bank (Cashboxes)', 'Cashiers'])
                    ->orderBy('customer_name')
                    ->get()
                    ->mapWithKeys(function ($customer) {
                        return [$customer->id => "{$customer->id} - {$customer->customer_name}"];
                    })
                    ->toArray();
            }
        );
        $this->createTransaction['transaction_date'] = Carbon::now()->toDateString();
        $this->createTransaction['transaction_time'] = Carbon::now()->toTimeString();
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->toDateString();
    }

    public function updatedShowDigitalCurrency()
    {
        // Reset selected currency when unchecked
        if (!$this->showDigitalCurrency) {
            $this->selectedDigitalCurrency = null;
        }
    }

    public function updatedSelectedDigitalCurrency()
    {
        // Validate only if showDigitalCurrency is checked
        if ($this->showDigitalCurrency) {
            $this->validate([
                'selectedDigitalCurrency' => 'required',
            ]);
        }
    }


    protected $listeners = ['valueSelected'];

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            // It's two words separated by a dot
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
    }


    
    public function saveConvert()
    {
        $this->amountToConvert = str_replace(',', '', $this->amountToConvert);
        $this->rate = str_replace(',', '', $this->rate);

         // to store unique GroupId 
         $transactionGroupId = Str::uuid();

        // Validation
        $this->validate([
            'transaction_type' => 'required|in:buy,sell',
            'rate' => 'required',
            'amountToConvert' => 'required',
            'operationType' => 'required|in:multiply,divide',
        ]);

        if ($this->transaction_type === 'sell') {
            $fromMyaccountType = 'withdrawal';
            $toMyaccountType = 'deposit';
            $destinationAccountType = 'deposit';
            $todestinationAccountType = 'withdrawal';
        }else{
            $fromMyaccountType = 'deposit';
            $toMyaccountType = 'withdrawal';
            $destinationAccountType = 'withdrawal';
            $todestinationAccountType = 'deposit';
        }

        try{

        $CashBoxUSDAccount = Account::where('customer_id', 1)
            ->whereHas('currency', function ($query) {
                $query->where('currency_name', 'USD'); // Assuming 'currency_name' holds the currency name
            })
            ->first();

        // Fetch the first IQD account for the selected customer
        $CashBoxIQDAccount = Account::where('customer_id', 1)
            ->whereHas('currency', function ($query) {
                $query->where('currency_name', 'IQD'); // Assuming 'currency_name' holds the currency name
            })
            ->first();
            

        // Calculate conversion based on operation type
        $convertedAmount = $this->operationType === 'multiply'
            ? $this->amountToConvert * $this->rate
            : $this->amountToConvert / $this->rate;

        $this->convertedAmount = $convertedAmount;            
            
        // Create transaction records
        $currentDate = Carbon::now()->toDateString();
        $currentTime = Carbon::now()->toTimeString();

        // transaction for customer from account
        $transactionCustomerUSD = AccountTransactionModel::create([
            'from_account_id' => $CashBoxUSDAccount->id,
            'to_account_id' => $this->destinationUSDAccount->id,
            'person_name' => "شکانەوە لەسەر داواکاری خۆی",
            'get_fee' => $this->rate,
            'transaction_type' => $fromMyaccountType,
            'transaction_amount' => $this->amountToConvert,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => empty($this->convert_note) ? 'شکانەوە لەسەر داواکاری خۆی' : $this->convert_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);
        // transaction for customer to account (target)
        AccountTransactionModel::create([
            'from_account_id' => $this->destinationUSDAccount->id,
            'to_account_id' => $CashBoxUSDAccount->id,
            'person_name' => "شکانەوە لەسەر داواکاری خۆی",
            'get_fee' => $this->rate,
            'transaction_type' => $destinationAccountType,
            'transaction_amount' => $this->amountToConvert,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => empty($this->convert_note) ? 'شکانەوە لەسەر داواکاری خۆی' : $this->convert_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        ////////////////////// Cashbox Transactions /////////////////////////////

         // transaction for cashbox from account
         $transactionCustomerIQD = AccountTransactionModel::create([
            'from_account_id' => $CashBoxIQDAccount->id,
            'to_account_id' => $this->destinationIQDAccount->id,
            'person_name' => "شکانەوە لەسەر داواکاری خۆی",
            'get_fee' => $this->rate,
            'transaction_type' => $toMyaccountType,
            'transaction_amount' => $convertedAmount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => empty($this->convert_note) ? 'شکانەوە لەسەر داواکاری خۆی' : $this->convert_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);
        // transaction for cashbox to account
        AccountTransactionModel::create([
            'from_account_id' => $this->destinationIQDAccount->id,
            'to_account_id' => $CashBoxIQDAccount->id,
            'person_name' => "شکانەوە لەسەر داواکاری خۆی",
            'get_fee' => $this->rate,
            'transaction_type' => $todestinationAccountType,
            'transaction_amount' => $convertedAmount,
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => empty($this->convert_note) ? 'شکانەوە لەسەر داواکاری خۆی' : $this->convert_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Send notifications for both USD and IQD transactions
        try {
            $this->sendConvertNotification($transactionCustomerUSD, $transactionCustomerIQD);
        } catch (\Exception $e) {
            Log::error('Failed to send convert notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }

        $this->alert('success', 'شکانەوەکە بە سەرکەوتووی ئەنجام درا !');
        $this->closeConvertSellModal();

        } catch (\Exception $e) {
            $this->alert('error', $e->getMessage());
        }
    }

    public function saveCreateUSD()
    {
        $this->create_transaction_amount = str_replace(',', '', $this->create_transaction_amount);
        $this->create_transaction_fee = str_replace(',', '', $this->create_transaction_fee);
        // $this->create_get_fee = str_replace(',', '', $this->create_get_fee);

        // Validate the input fields
        $this->validate([
            'filters.customer' => 'required',
            'create_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount' => 'required|numeric',
            'create_transaction_fee' => 'required|numeric', // Validate the fee as well
            // 'create_get_fee' => 'required|numeric', // Validate the fee as well
            'create_person_name' => 'nullable|string',
            'create_note' => 'nullable|string',
            'file' => 'nullable|image|max:1048'
        ]);

        // to store unique GroupId 
        $transactionGroupId = Str::uuid();

        $USDAccount = Account::where('customer_id', $this->filters['customer'])
        ->whereHas('currency', function ($query) {
            $query->where('currency_name', 'USD'); // Assuming 'currency_name' holds the currency name
        })
        ->first();

        if (!$USDAccount) {
            $this->alert('error', 'موشتەری حسابی بە دۆلار نیە');
            return;
        }

        // Retrieve the from account (selected account)
        $fromAccount = Account::findOrFail($USDAccount->id);

        // Fetch the cashbox account (where customer_id is 1) and ensure it matches the currency of from_account
        $cashboxAccount = Account::where('customer_id', 1)
            ->where('currency_id', $fromAccount->currency_id)  // Ensure the currency matches
            ->where('account_type', 'profit')
            ->first();

        // Get current user
        $user = Auth::user();
        $userRole = $user->getRoleNames()->first(); // Assuming you're using spatie/laravel-permission
        // Determine the to_account_id based on user role
        if ($userRole === 'cashier') {
            // Get cashier's USD account
            $toAccount = Account::where('customer_id', $user->customer->id)
                ->where('currency_id', $fromAccount->currency_id)
                ->first();
            
            if (!$toAccount) {
                $this->alert('error', 'کاشیر حسابی بە دۆلار نیە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        } 
        elseif ($userRole === 'supervisor') {
            // Get supervisor's USD account
            $toAccount = Account::where('customer_id', $user->customer->id)
                ->where('currency_id', $fromAccount->currency_id)
                ->first();
            
            if (!$toAccount) {
                $this->alert('error', 'سوپەرڤایزەر حسابی بە دۆلار نیە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        }
        else if ($userRole === 'super-admin') {
            // Default behavior for super-admin or other roles
            $toAccount = Account::where('customer_id', 1)
                ->where('currency_id', $fromAccount->currency_id)
                ->where('account_type', 'savings')
                ->first();
                
            if (!$toAccount) {
                $this->alert('error', 'حسابی سەیڤینگ نەدۆزرایەوە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        }

        // Get current date and time
        $currentDate = Carbon::now()->toDateString(); // Gets current date in 'YYYY-MM-DD' format
        $currentTime = Carbon::now()->toTimeString(); // Gets current time in 'HH:MM:SS' format

        // Create the main transaction for the from_account
        $transactionFrom = AccountTransactionModel::create([
            'from_account_id' => $USDAccount->id, // The account initiating the transaction
            'to_account_id' => $toAccountId,
            'person_name' => $this->create_person_name,
            'get_fee' => $this->create_get_fee,
            'transaction_type' => $this->create_transaction_type,
            'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee, // Actual transaction amount
            'transaction_date' => $currentDate, // Use current date
            'transaction_time' => $currentTime, // Use current time
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Handle file upload and associate with the transaction
        try {
            if ($this->file) {
                $filePath = $this->file->store('files', 'public');
                $transactionFrom->files()->create([
                    'file_path' => $filePath,
                ]);
            }
        } catch (\Exception $e) {
            dd($e->getMessage());
        }

        // Create a corresponding transaction for the to_account
        $transactionTo = AccountTransactionModel::create([
            'from_account_id' => $toAccountId, // The receiving account
            'to_account_id' => $USDAccount->id, // The originating account
            'person_name' => $this->create_person_name,
            'get_fee' => $this->create_get_fee,
            'transaction_type' => $this->create_transaction_type, // Same transaction type
            'transaction_amount' => $this->create_transaction_amount + $this->create_get_fee, // Same amount
            'transaction_date' => $currentDate, // Use current date
            'transaction_time' => $currentTime, // Use current time
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Create the third transaction (Profit Account -> Fee Record)
        $transactionProfit = AccountTransactionModel::create([
            'from_account_id' => $cashboxAccount->id, // Fee is deducted from the originating account
            'to_account_id' => $USDAccount->id, // Fee is credited to the profit account
            'person_name' => $this->create_person_name,
            'get_fee' => 0,
            'transaction_type' => 'deposit',
            'transaction_amount' => $this->create_get_fee, // Amount of the fee
            'transaction_date' => $currentDate,
            'transaction_time' => $currentTime,
            'note' => 'Fee collected for ' . $this->create_transaction_type,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Send notification to the customer whose account was affected
        try {
            $this->sendTransactionNotification($transactionFrom);
        } catch (\Exception $e) {
            Log::error('Failed to send transaction notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }

        $this->reset('create_note', 'create_transaction_fee', 'create_transaction_amount',
        'create_get_fee', 'create_person_name',
        'file', 'create_transaction_type');

        // Flash success message
        $this->alert('success', 'مامەڵەکە بەسەرکەوتووی تۆمارکرا');
    } 

    public function saveCreateIQD()
    {
    
        $this->create_transaction_amount_IQD = str_replace(',', '', $this->create_transaction_amount_IQD);
        $this->create_transaction_fee_IQD = str_replace(',', '', $this->create_transaction_fee_IQD);
        // $this->create_get_fee_IQD = str_replace(',', '', $this->create_get_fee_IQD);

        // to store unique GroupId 
        $transactionGroupId = Str::uuid();

        // Validate the input fields
        $this->validate([
            'filters.customer' => 'required',
            'create_transaction_type_IQD' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount_IQD' => 'required|numeric',
            // 'create_transaction_fee_IQD' => 'required|numeric', // Validate the fee as well
            // 'create_get_fee_IQD' => 'required|numeric', // Validate the fee as well
            'create_person_name_IQD' => 'nullable|string',
            'create_note_IQD' => 'nullable|string',
            'file_IQD' => 'nullable|image|max:1048',
            'selectedDigitalCurrency' => $this->showDigitalCurrency ? 'required' : 'nullable',
        ]);

        try {
            
        $IQDAccount = Account::where('customer_id', $this->filters['customer'])
            ->whereHas('currency', function ($query) {
                $query->where('currency_name', 'IQD'); // Assuming 'currency_name' holds the currency name
            })
            ->first();

        if (!$IQDAccount) {
            $this->alert('error', 'موشتەری حسابی بە دینار نیە');
            return;
        }
        
        // Retrieve the from account (selected account)
        $fromAccount = Account::findOrFail($IQDAccount->id);
        
        // Fetch the cashbox account (where customer_id is 1) and ensure it matches the currency of from_account
        $cashboxAccount = Account::where('customer_id', 1)
            ->where('currency_id', $fromAccount->currency_id)  // Ensure the currency matches
            ->where('account_type', 'profit')
            ->first();

        // Get current user
        $user = Auth::user();
        $userRole = $user->getRoleNames()->first(); // Assuming you're using spatie/laravel-permission
        
        // Only determine user account if NOT using digital currency
        $toAccountId = null;
        if (!$this->showDigitalCurrency) {
            // Determine the to_account_id based on user role
            if ($userRole === 'cashier') {
                // Get cashier's USD account
                $toAccount = Account::where('customer_id', $user->customer->id)
                    ->where('currency_id', $fromAccount->currency_id)
                    ->first();
                
                if (!$toAccount) {
                    $this->alert('error', 'کاشیر حسابی بە دۆلار نیە');
                    return;
                }
                
                $toAccountId = $toAccount->id;
            } 
            elseif ($userRole === 'supervisor') {
                // Get supervisor's USD account
                $toAccount = Account::where('customer_id', $user->customer->id)
                    ->where('currency_id', $fromAccount->currency_id)
                    ->first();
                
                if (!$toAccount) {
                    $this->alert('error', 'سوپەرڤایزەر حسابی بە دۆلار نیە');
                    return;
                }
                
                $toAccountId = $toAccount->id;
            }
            else if ($userRole === 'super-admin') {
                // Default behavior for super-admin or other roles
                $toAccount = Account::where('customer_id', 1)
                    ->where('currency_id', $fromAccount->currency_id)
                    ->where('account_type', 'savings')
                    ->first();
                    
                if (!$toAccount) {
                    $this->alert('error', 'حسابی سەیڤینگ نەدۆزرایەوە');
                    return;
                }
                
                $toAccountId = $toAccount->id;
            }
        }

        // Get current date and time
        $currentDate = Carbon::now()->toDateString(); // Gets current date in 'YYYY-MM-DD' format
        $currentTime = Carbon::now()->toTimeString(); // Gets current time in 'HH:MM:SS' format

        if($this->showDigitalCurrency){

            // Fetch the cashbox account (where customer_id is 1) and ensure it matches the currency is digital
            $cashboxDigitalAccount = Account::where('customer_id', 1)
            ->where('currency_id', $this->selectedDigitalCurrency)  // Ensure the currency matches
            ->where('account_type', 'savings')
            ->first();

            if (!$cashboxDigitalAccount) {
                $this->alert('error', 'حسابی دیجیتاڵ نەدۆزرایەوە');
                return;
            }

            // Transaction 1: Customer IQD Account -> Digital Account
            $transactionFrom = AccountTransactionModel::create([
                'from_account_id' => $IQDAccount->id, // Customer's IQD account
                'to_account_id' => $cashboxDigitalAccount->id, // Digital currency account
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD,
                'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD, // Actual transaction amount
                'transaction_date' => $currentDate, // Use current date
                'transaction_time' => $currentTime, // Use current time
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'transaction_group_id' => $transactionGroupId,
            ]);

            // Handle file upload and associate with the transaction
            try {
                if ($this->file_IQD) {
                    $filePath = $this->file_IQD->store('files', 'public');
                    $transactionFrom->files()->create([
                        'file_path' => $filePath,
                    ]);
                }
            } catch (\Exception $e) {
                dd($e->getMessage());
            }

            // Transaction 2: Digital Account -> Customer IQD Account (reverse transaction)
            $transactionToDigital = AccountTransactionModel::create([
                'from_account_id' => $cashboxDigitalAccount->id, // Digital currency account
                'to_account_id' => $IQDAccount->id, // Customer's IQD account
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD, // Same transaction type
                'transaction_amount' => $this->create_transaction_amount_IQD, // Amount without fee
                'transaction_date' => $currentDate, // Use current date
                'transaction_time' => $currentTime, // Use current time
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'transaction_group_id' => $transactionGroupId,
            ]);

            // Transaction 3: Fee transaction (if there's a fee)
            if ($this->create_get_fee_IQD > 0) {
                $transactionFee = AccountTransactionModel::create([
                    'from_account_id' => $cashboxAccount->id, // Profit account gets the fee
                    'to_account_id' => $IQDAccount->id, // Customer's account
                    'person_name' => $this->create_person_name_IQD,
                    'get_fee' => 0,
                    'transaction_type' => 'deposit',
                    'transaction_amount' => $this->create_get_fee_IQD, // Fee amount
                    'transaction_date' => $currentDate,
                    'transaction_time' => $currentTime,
                    'note' => 'Fee collected for digital currency ' . $this->create_transaction_type_IQD,
                    'user_id' => Auth::id(),
                    'transaction_group_id' => $transactionGroupId,
                ]);
            }

        } else {

            // Regular transaction (non-digital currency)
            // Create the main transaction for the from_account
            $transactionFrom = AccountTransactionModel::create([
                'from_account_id' => $IQDAccount->id, // The account initiating the transaction
                'to_account_id' => $toAccountId,
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD,
                'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD, // Actual transaction amount
                'transaction_date' => $currentDate, // Use current date
                'transaction_time' => $currentTime, // Use current time
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'transaction_group_id' => $transactionGroupId,
            ]);

            // Handle file upload and associate with the transaction
            try {
                if ($this->file_IQD) {
                    $filePath = $this->file_IQD->store('files', 'public');
                    $transactionFrom->files()->create([
                        'file_path' => $filePath,
                    ]);
                }
            } catch (\Exception $e) {
                dd($e->getMessage());
            }

            // Create a corresponding transaction for the to_account
            $transactionTo = AccountTransactionModel::create([
                'from_account_id' => $toAccountId, // The receiving account
                'to_account_id' => $IQDAccount->id, // The originating account
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => $this->create_get_fee_IQD,
                'transaction_type' => $this->create_transaction_type_IQD, // Same transaction type
                'transaction_amount' => $this->create_transaction_amount_IQD + $this->create_get_fee_IQD, // Amount of the fee
                'transaction_date' => $currentDate, // Use current date
                'transaction_time' => $currentTime, // Use current time
                'note' => $this->create_note_IQD,
                'user_id' => Auth::id(),
                'transaction_group_id' => $transactionGroupId,
            ]);

            // Create the third transaction (Profit Account -> Fee Record)
            $transactionProfit = AccountTransactionModel::create([
                'from_account_id' => $cashboxAccount->id, // Fee is deducted from the originating account
                'to_account_id' => $IQDAccount->id, // Fee is credited to the profit account
                'person_name' => $this->create_person_name_IQD,
                'get_fee' => 0,
                'transaction_type' => 'deposit',
                'transaction_amount' => $this->create_get_fee_IQD, // Amount of the fee
                'transaction_date' => $currentDate,
                'transaction_time' => $currentTime,
                'note' => 'Fee collected for ' . $this->create_transaction_type_IQD,
                'user_id' => Auth::id(),
                'transaction_group_id' => $transactionGroupId,
            ]);
        }
        
        // Send notification to the customer whose account was affected
        try {
            $this->sendTransactionNotification($transactionFrom);
        } catch (\Exception $e) {
            Log::error('Failed to send transaction notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }
        
        $this->reset('create_note_IQD', 'create_transaction_fee_IQD', 'create_transaction_amount_IQD',
        'create_get_fee_IQD', 'create_person_name_IQD',
        'file_IQD', 'create_transaction_type_IQD', 'selectedDigitalCurrency', 'showDigitalCurrency' );

        // Flash success message
        $this->alert('success', 'مامەڵەکە بەسەرکەوتووی تۆمارکرا');
    } catch (\Exception $e) {
        $this->alert('error', $e->getMessage());
    }

    }
    public function render()
    {
        $USDAccount = null;
        $IQDAccount = null;

        if ($this->filters['customer']) {
            // Fetch the first USD account for the selected customer
            $USDAccount = Account::select('id', 'customer_id', 'currency_id')
                ->where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'USD');
                })
                ->first();

            if ($USDAccount) {
                $USDAccount->load(['relatedTransactions' => function ($query) {
                    $query->select(
                            'id',
                            'from_account_id',
                            'to_account_id',
                            'transaction_type',
                            'transaction_amount',
                            'transaction_date',
                            'transaction_time',
                            'note',
                            'get_fee'
                        )
                        ->where('transaction_type', '!=', 'fee')
                        ->where('status', 'approved')
                        ->when($this->startDate, function ($q) {
                            $q->whereDate('transaction_date', '>=', $this->startDate);
                        })
                        ->when($this->endDate, function ($q) {
                            $q->whereDate('transaction_date', '<=', $this->endDate);
                        })
                        ->orderBy('transaction_date', 'desc')
                        ->orderBy('transaction_time', 'desc')
                        ->with([
                            'fees:id,fee_amount,feeable_id,feeable_type',
                            'account.currency:id,currency_name'
                        ]);
                }]);
            }
                
            $this->destinationUSDAccount = $USDAccount;

            // Fetch the first IQD account for the selected customer
            $IQDAccount = Account::select('id', 'customer_id', 'currency_id')
                ->where('customer_id', $this->filters['customer'])
                ->whereHas('currency', function ($query) {
                    $query->where('currency_name', 'IQD');
                })
                ->first();

            if ($IQDAccount) {
                $IQDAccount->load(['relatedTransactions' => function ($query) {
                    $query->select(
                            'id',
                            'from_account_id',
                            'to_account_id',
                            'transaction_type',
                            'transaction_amount',
                            'transaction_date',
                            'transaction_time',
                            'note',
                            'get_fee'
                        )
                        ->where('transaction_type', '!=', 'fee')
                        ->where('status', 'approved')
                        ->when($this->startDate, function ($q) {
                            $q->whereDate('transaction_date', '>=', $this->startDate);
                        })
                        ->when($this->endDate, function ($q) {
                            $q->whereDate('transaction_date', '<=', $this->endDate);
                        })
                        ->orderBy('transaction_date', 'desc')
                        ->orderBy('transaction_time', 'desc')
                        ->with([
                            'fees:id,fee_amount,feeable_id,feeable_type',
                            'account.currency:id,currency_name'
                        ]);
                }]);
            }
                
            $this->destinationIQDAccount = $IQDAccount;
            $this->USDaccount = $USDAccount;
            $this->IQDaccount = $IQDAccount;
            $this->selectedCustomer = optional(Customer::select('id')->find($this->filters['customer']))->id;
        }

        return view('livewire.account-transaction-form', [
            'USDAccount' => $USDAccount,
            'IQDAccount' => $IQDAccount,
            'customerOptions' => $this->customerOptions,
        ]);
    }

    // Add method to reset date filter
    public function resetDateFilter()
    {
        $this->startDate = Carbon::now()->subDays(3)->toDateString();
        $this->endDate = Carbon::now()->toDateString();
    }

    /**
     * Send notification for new transaction
     */
    private function sendTransactionNotification(AccountTransactionModel $transaction)
    {
        // Resolve the notification service when needed
        $notificationService = app(TransactionNotificationService::class);
        $notificationService->sendTransactionCreatedNotification($transaction);
    }

    /**
     * Send notification for currency conversion
     */
    private function sendConvertNotification(AccountTransactionModel $transactionUSD, AccountTransactionModel $transactionIQD)
    {
        try {
            // Get the user through the account relationship
            $fromAccount = $transactionUSD->to_account; // Customer's USD account
            
            if (!$fromAccount || !$fromAccount->customer || !$fromAccount->customer->user) {
                Log::warning('Convert transaction has no associated user', [
                    'transaction_id' => $transactionUSD->id,
                ]);
                return;
            }

            $user = $fromAccount->customer->user;
            $userLanguage = $user->language_preference ?? 'en';
            
            // Check if user has notifications enabled
            if (!$user->notifications_enabled || !$user->transaction_notifications) {
                Log::info('Skipping convert notification - user has disabled notifications', [
                    'user_id' => $user->id,
                ]);
                return;
            }

            // Get currency names from transactions
            $fromCurrency = $transactionUSD->to_account->currency->currency_name ?? 'USD';
            $toCurrency = $transactionIQD->to_account->currency->currency_name ?? 'IQD';
            $originalAmount = number_format($transactionUSD->transaction_amount, 2);
            $resultAmount = number_format($transactionIQD->transaction_amount, 0);
            
            // Create custom message for conversion with details
            if ($userLanguage === 'en') {
                $message = "Currency conversion completed: {$originalAmount} {$fromCurrency} → {$resultAmount} {$toCurrency}";
            } elseif ($userLanguage === 'ar') {
                $message = "تم تحويل العملة: {$originalAmount} {$fromCurrency} ← {$resultAmount} {$toCurrency}";
            } else {
                // Kurdish (default)
                $message = "شکاندنەوەی دراو: {$originalAmount} {$fromCurrency} → {$resultAmount} {$toCurrency}";
            }

            // Prepare notification data
            $data = [
                'type' => 'conversion',
                'transaction_id' => $transactionUSD->id,
                'from_currency' => $fromCurrency,
                'to_currency' => $toCurrency,
                'original_amount' => $transactionUSD->transaction_amount,
                'result_amount' => $transactionIQD->transaction_amount,
                'rate' => $this->rate,
                'date' => $transactionUSD->transaction_date,
            ];

            // Store notification in database
            \App\Models\Notification::create([
                'type' => 'conversion',
                'title' => $userLanguage === 'en' ? 'Currency Conversion' : ($userLanguage === 'ar' ? 'تحويل العملة' : 'شکاندنەوەی دراو'),
                'body' => $message,
                'user_id' => $user->id,
                'data' => $data,
            ]);

            // Send push notification via OneSignal
            $oneSignalService = app(\App\Services\OneSignalService::class);
            $oneSignalService->sendNotificationToUsers([$user->id], $message, $data, $userLanguage);

            Log::info('Convert notification sent successfully', [
                'user_id' => $user->id,
                'transaction_usd_id' => $transactionUSD->id,
                'transaction_iqd_id' => $transactionIQD->id,
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send convert notification: ' . $e->getMessage(), [
                'transaction_usd_id' => $transactionUSD->id ?? null,
                'transaction_iqd_id' => $transactionIQD->id ?? null,
                'error' => $e->getMessage(),
            ]);
        }
    }
    
}
