<?php
namespace App\Livewire;

use App\Models\AccountTransaction as AccountTransactionModel;
use App\Models\Account;
use App\Models\File;
use App\Models\Customer;
use App\Models\TransactionFee;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Carbon\Carbon;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\WithPagination;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\WithFileUploads;
use Illuminate\Support\Str;
use App\Services\TransactionNotificationService;

class AccountTransaction extends Component
{
    use LivewireAlert, WithPagination;
    use WithFileUploads;

    protected $paginationTheme = 'bootstrap';
    
    protected $notificationService;

    public $transactions;
    public $editTransactionId;
    public $editTransactionType;
    public $editTransactionAmount;
    public $transactionGroupIdToEdit, $editTransactionFeeAmount;
    public $transactionFees = [];

    public $file;

    public $startDate;
    public $endDate;
    // Create modal properties
    public $create_account_id, $create_person_name, $create_get_fee;
    public $create_transaction_type;
    public $create_transaction_amount;
    public $create_transaction_date;
    public $create_transaction_fee;
    public $create_transaction_time;
    public $create_note;
    public $filters;
    public $showModal = false, $showEditModal = false;

    // Edit modal properties
    public $edit_account_id, $edit_person_name, $edit_get_fee;
    public $edit_transaction_type;
    public $edit_transaction_amount;
    public $edit_transaction_date;
    public $edit_transaction_time;
    public $edit_transaction_fee;
    public $edit_note;
    public $transactionId;

    // Define available transaction types (enum)
    public $transactionTypes = ['Select Transaction Type','deposit', 'withdrawal'];
    public $accounts;

    // Add a listener for pagination events
    protected $listeners = ['valueSelected', 'paginationChanged' => '$refresh'];

    public function mount()
    {
        // Retrieve all customer IDs with the name "Cashiers" - Cache this query
        $customer_cashiers_ids = Customer::where('customer_name', 'Cashiers')->pluck('id');

        $this->filters = [
            'account' => null,
            'customer_account' => null
        ];

        // Set default dates to today
        $this->startDate = Carbon::now()->toDateString();
        $this->endDate = Carbon::now()->toDateString();

        // Get accounts excluding customer_id 1 and the IDs of customers named "Cashiers"
        $this->accounts = Account::where('customer_id', '!=', 1)
            ->whereNotIn('customer_id', $customer_cashiers_ids)
            ->get();
    }

    public function openModal()
    {
        $this->showModal = true;
    }

    public function closeModal()
    {
        $this->create_account_id = '';
        $this->create_person_name = '';
        $this->create_get_fee = '';
        $this->create_transaction_type = '';
        $this->create_transaction_amount = '';
        $this->create_transaction_date = '';
        $this->create_transaction_time = '';
        $this->create_transaction_fee = '';
        $this->create_note = '';
        $this->showModal = false;
    }

    public function openEditModal()
    {
        $this->showEditModal = true;
    }

    public function closeEditModal()
    {
        $this->edit_person_name = '';
        $this->edit_account_id = '';
        $this->edit_transaction_type = '';
        $this->edit_transaction_amount = '';
        $this->edit_transaction_date = '';
        $this->edit_transaction_time = '';
        $this->edit_transaction_fee = '';
        $this->edit_note = '';
        $this->showEditModal = false;
    }

    public function valueSelected($selectedValue, $identifier)
    {
        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $this->$identifier = $selectedValue;
        }

        if (preg_match('/^[A-Za-z_][A-Za-z_0-9]*\.[A-Za-z_][A-Za-z_0-9]*$/', $identifier)) {
            $parts = explode('.', $identifier);
            $array = $parts[0];
            $index = $parts[1];
            $this->$array[$index] = $selectedValue;
        }
    }

    // OPTIMIZED: Load transactions for editing with eager loading and selective queries
    public function editTransaction($transactionGroupId)
    {
        $this->transactionGroupIdToEdit = $transactionGroupId;
     
        // OPTIMIZATION 1: Use eager loading to reduce database queries
        $this->transactions = AccountTransactionModel::with([
            'from_account.customer:id,customer_name',
            'to_account.customer:id,customer_name',
            'fees:id,fee_amount,feeable_id,feeable_type'
        ])
        ->where('transaction_group_id', $transactionGroupId)
        ->select(['id', 'transaction_type', 'transaction_amount', 'from_account_id', 'to_account_id'])
        ->get();
     
        if ($this->transactions->isNotEmpty()) {
            $firstTransaction = $this->transactions->first();
            $this->editTransactionType = $firstTransaction->transaction_type;
            $this->editTransactionAmount = $firstTransaction->transaction_amount;
     
            // OPTIMIZATION 2: Get fees directly from the already loaded relationship
            $this->transactionFees = $firstTransaction->fees;
     
            if ($this->transactionFees->isNotEmpty()) {
                $this->editTransactionFeeAmount = $this->transactionFees->first()->fee_amount; 
            } else {
                $this->editTransactionFeeAmount = null;
            }
        }
     
        $this->showEditModal = true;
    }

    // OPTIMIZED: Update Transaction with batch operations
    public function updateTransaction()
    {
        $this->editTransactionAmount = str_replace(',', '', $this->editTransactionAmount);
        $this->editTransactionFeeAmount = str_replace(',', '', $this->editTransactionFeeAmount);

        $this->validate([
            'editTransactionType' => 'required|in:deposit,withdrawal,transfer',
            'editTransactionAmount' => 'required|numeric',
            'editTransactionFeeAmount' => 'nullable|numeric',
        ]);

        try {
            DB::beginTransaction();

            // OPTIMIZATION 3: Use bulk update instead of individual updates
            AccountTransactionModel::where('transaction_group_id', $this->transactionGroupIdToEdit)
                ->update([
                    'transaction_type' => $this->editTransactionType,
                    'transaction_amount' => $this->editTransactionAmount,
                    'updated_at' => now()
                ]);

            // OPTIMIZATION 4: Update fees efficiently if they exist
            if (!empty($this->transactionFees) && count($this->transactionFees) > 0 && $this->editTransactionFeeAmount !== null) {
                TransactionFee::where('feeable_type', AccountTransactionModel::class)
                    ->whereIn('feeable_id', $this->transactions->pluck('id'))
                    ->update([
                        'fee_amount' => $this->editTransactionFeeAmount,
                        'updated_at' => now()
                    ]);
            }

            DB::commit();

            // Send notification for transaction update
            try {
                $updatedTransaction = AccountTransactionModel::where('transaction_group_id', $this->transactionGroupIdToEdit)->first();
                if ($updatedTransaction) {
                    $this->sendTransactionUpdateNotification($updatedTransaction);
                }
            } catch (\Exception $e) {
                Log::error('Failed to send transaction update notification: ' . $e->getMessage());
                // Don't fail the update if notification fails
            }

            $this->showEditModal = false;
            session()->flash('success', 'جولەکان بەسەرکەوتووی نوێکرایەوە');

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Transaction update failed: ' . $e->getMessage());
            session()->flash('error', 'هەڵەیەک ڕوویدا لە کاتی نوێکردنەوەدا');
        }
    }

    public function edit($id)
    {
        $transaction = AccountTransactionModel::findOrFail($id);
        $this->transactionId = $transaction->id;
        $this->edit_account_id = $transaction->from_account_id;
        $this->edit_person_name = $transaction->person_name;
        $this->edit_get_fee = $transaction->get_fee;
        $this->edit_transaction_type = $transaction->transaction_type;
        $this->edit_transaction_amount = $transaction->transaction_amount;
        $this->edit_transaction_date = $transaction->transaction_date;
        $this->edit_transaction_time = $transaction->transaction_time;
        $this->edit_note = $transaction->note;
        $this->showEditModal = true;
    }

    public function saveEdit()
    {
        $this->validate([
            'edit_account_id' => 'required|exists:accounts,id',
            'edit_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'edit_transaction_amount' => 'required|numeric|min:0',
            'edit_note' => 'nullable|string',
            'edit_person_name' => 'nullable|string',
        ]);

        $transaction = AccountTransactionModel::findOrFail($this->transactionId);

        $oldAmount = $transaction->transaction_amount;
        $newAmount = $this->edit_transaction_amount;
        $difference = $newAmount - $oldAmount;

        $transaction->update([
            'account_id' => $this->edit_account_id,
            'transaction_type' => $this->edit_transaction_type,
            'transaction_amount' => $this->edit_transaction_amount,
            'note' => $this->edit_note,
            'person_name' => $this->edit_person_name,
        ]);

        $account = Account::findOrFail($this->edit_account_id);
        if ($transaction->transaction_type == 'deposit') {
            $account->balance -= $oldAmount;
            $account->balance += $newAmount;
        } elseif ($transaction->transaction_type == 'withdrawal') {
            $account->balance += $oldAmount;
            $account->balance -= $newAmount;
        }

        $account->save();
        $this->resetEditFields();

        $this->alert('success', 'Transaction updated successfully!');
        $this->showEditModal = false;
    }

    private function resetCreateFields()
    {
        $this->create_account_id = '';
        $this->create_person_name = '';
        $this->create_get_fee = '';
        $this->create_transaction_type = '';
        $this->create_transaction_amount = '';
        $this->create_transaction_date = '';
        $this->create_transaction_time = '';
        $this->create_transaction_fee = '';
        $this->create_note = '';
    }

    private function resetEditFields()
    {
        $this->edit_account_id = '';
        $this->edit_person_name = '';
        $this->edit_transaction_type = '';
        $this->edit_transaction_amount = '';
        $this->edit_transaction_date = '';
        $this->edit_transaction_fee = '';
        $this->edit_transaction_time = '';
        $this->edit_note = '';
        $this->transactionId = null;
    }

    public function updatingFilters($value, $key = null)
    {
        $this->resetPage();
    }

    public function updatingStartDate($value)
    {
        $this->resetPage();
    }

    public function updatingEndDate($value)
    {
        $this->resetPage();
    }

    /**
     * Validate filter inputs
     */
    public function validateFilters()
    {
        $errors = [];
        
        // Validate date range
        if ($this->startDate && $this->endDate) {
            if ($this->startDate > $this->endDate) {
                $errors[] = 'Start date cannot be after end date';
            }
        }
        
        // Validate date format
        if ($this->startDate && !$this->isValidDate($this->startDate)) {
            $errors[] = 'Invalid start date format';
        }
        
        if ($this->endDate && !$this->isValidDate($this->endDate)) {
            $errors[] = 'Invalid end date format';
        }
        
        return $errors;
    }

    /**
     * Check if date is valid
     */
    private function isValidDate($date)
    {
        try {
            Carbon::createFromFormat('Y-m-d', $date);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function resetFilters()
    {
        $this->reset(['startDate', 'endDate', 'filters']);
        
        // Reset filters to default state
        $this->filters = [
            'account' => null,
            'customer_account' => null
        ];
        
        // Set default dates to today
        $this->startDate = Carbon::now()->toDateString();
        $this->endDate = Carbon::now()->toDateString();
        
        $this->resetPage();
    }

    
    public function saveCreate()
    {
        $this->create_transaction_amount = str_replace(',', '', $this->create_transaction_amount);
        $this->create_transaction_fee = str_replace(',', '', $this->create_transaction_fee);
        $this->create_get_fee = str_replace(',', '', $this->create_get_fee);

        // Validate the input fields
        $this->validate([
            'filters.account' => 'required|exists:accounts,id',
            'create_transaction_type' => 'required|in:deposit,withdrawal,transfer',
            'create_transaction_amount' => 'required|numeric',
            'create_transaction_fee' => 'required|numeric', // Validate the fee as well
            'create_get_fee' => 'required|numeric', // Validate the fee as well
            'create_person_name' => 'nullable|string',
            'create_note' => 'nullable|string',
            'file' => 'nullable|image|max:1048'
        ]);

        // to store unique GroupId 
        $transactionGroupId = Str::uuid();

        // Retrieve the from account (selected account)
        $fromAccount = Account::findOrFail($this->filters['account']);

        // Fetch the cashbox account (where customer_id is 1) and ensure it matches the currency of from_account
        $cashboxAccount = Account::where('customer_id', 1)
            ->where('currency_id', $fromAccount->currency_id)  // Ensure the currency matches
            ->where('account_type', 'profit')
            ->first();
    
        if (!$cashboxAccount) {
            session()->flash('error', 'Cashbox account with matching currency not found.');
            return;
        }

        // Get current user
        $user = Auth::user();
        $userRole = $user->getRoleNames()->first(); // Assuming you're using spatie/laravel-permission
        
        // Determine the to_account_id based on user role
        if ($userRole === 'cashier') {
            // Get cashier's USD account
            $toAccount = Account::where('customer_id', $user->customer->id)
                ->where('currency_id', $fromAccount->currency_id)
                ->first();
            
            if (!$toAccount) {
                $this->alert('error', 'کاشیر حسابی بە دۆلار نیە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        } 
        elseif ($userRole === 'supervisor') {
            // Get supervisor's USD account
            $toAccount = Account::where('customer_id', $user->customer->id)
                ->where('currency_id', $fromAccount->currency_id)
                ->first();
            
            if (!$toAccount) {
                $this->alert('error', 'سوپەرڤایزەر حسابی بە دۆلار نیە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        }
        else if ($userRole === 'super-admin') {
            // Default behavior for super-admin or other roles
            $toAccount = Account::where('customer_id', 1)
                ->where('currency_id', $fromAccount->currency_id)
                ->where('account_type', 'savings')
                ->first();
                
            if (!$toAccount) {
                $this->alert('error', 'حسابی سەیڤینگ نەدۆزرایەوە');
                return;
            }
            
            $toAccountId = $toAccount->id;
        }
            
        // Get current date and time
        $currentDate = Carbon::now()->toDateString(); // Gets current date in 'YYYY-MM-DD' format
        $currentTime = Carbon::now()->toTimeString(); // Gets current time in 'HH:MM:SS' format
    
        // Create the main transaction for the from_account
        $transactionFrom = AccountTransactionModel::create([
            'from_account_id' => $this->filters['account'], // The account initiating the transaction
            'to_account_id' => $toAccountId,
            'person_name' => $this->create_person_name,
            'get_fee' => $this->create_transaction_fee,
            'transaction_type' => $this->create_transaction_type,
            'transaction_amount' => $this->create_transaction_amount, // Actual transaction amount
            'transaction_date' => $currentDate, // Use current date
            'transaction_time' => $currentTime, // Use current time
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

        // Handle file upload and associate with the transaction
        try {
            if ($this->file) {
                $filePath = $this->file->store('files', 'public');
                $transactionFrom->files()->create([
                    'file_path' => $filePath,
                ]);
            }
        } catch (\Exception $e) {
            dd($e->getMessage());
        }

        // Create a corresponding transaction for the to_account
        $transactionTo = AccountTransactionModel::create([
            'from_account_id' => $toAccountId, // The receiving account
            'to_account_id' => $this->filters['account'], // The originating account
            'person_name' => $this->create_person_name,
            'get_fee' => $this->create_transaction_fee,
            'transaction_type' => $this->create_transaction_type, // Same transaction type
            'transaction_amount' => $this->create_transaction_amount, // Same amount
            'transaction_date' => $currentDate, // Use current date
            'transaction_time' => $currentTime, // Use current time
            'note' => $this->create_note,
            'user_id' => Auth::id(),
            'transaction_group_id' => $transactionGroupId,
        ]);

    
        // Update account balances for both from_account and cashboxAccount
        $fromAccount = Account::findOrFail($this->filters['account']); // Refresh the from account
        // $cashboxAccount = Account::where('customer_id', 1)
    
        if ($this->create_transaction_type == 'deposit') {
            // Deposit: customer receives total amount (transaction amount + fee)
            $fromAccount->balance += ($this->create_transaction_amount + $this->create_get_fee);
            // Cashbox gains the fee amount
            $cashboxAccount->balance += $this->create_get_fee;
    
        } elseif ($this->create_transaction_type == 'withdrawal') {
            // Withdrawal: customer pays total amount (transaction amount + fee)
            $fromAccount->balance -= ($this->create_transaction_amount + $this->create_get_fee);
            // Cashbox gains the fee amount
            $cashboxAccount->balance += $this->create_get_fee;
        }
    
        // Save the updated balances
        $fromAccount->save();
        $cashboxAccount->save(); // Make sure the cashbox is being updated

        
        // if ($this->create_get_fee > 0) {
        $transactionFrom->fees()->create([
            'fee_amount' => $this->create_get_fee,
            'fee_type' => $this->create_transaction_type . '_fee', // E.g., 'deposit_fee' or 'withdrawal_fee'
        ]);
    
        $transactionTo->fees()->create([
            'fee_amount' => $this->create_get_fee,
            'fee_type' => $this->create_transaction_type . '_fee', // E.g., 'deposit_fee' or 'withdrawal_fee'
        ]);
        // }
        
        // Send notification to the customer whose account was affected
        try {
            $this->sendTransactionNotification($transactionFrom);
        } catch (\Exception $e) {
            Log::error('Failed to send transaction notification: ' . $e->getMessage());
            // Don't fail the transaction if notification fails
        }

        // Flash success message
        $this->alert('success', 'Transaction and fee processed successfully!');

        // Reset input fields and close modal
        $this->resetCreateFields();
        $this->showModal = false; // Close the modal
    }    

    // OPTIMIZED: Render method with better query optimization
    public function render()
    {
        // Validate filters first
        $filterErrors = $this->validateFilters();
        if (!empty($filterErrors)) {
            foreach ($filterErrors as $error) {
                $this->alert('error', $error);
            }
        }

        // OPTIMIZATION 5: Use eager loading and select only necessary columns
        $query = AccountTransactionModel::with([
            'from_account.customer:id,customer_name',
            'from_account.currency:id,currency_name',
            'to_account.customer:id,customer_name',
            'to_account.currency:id,currency_name', // Added this for complete data
            'fees:id,fee_amount,feeable_id,feeable_type'
        ])
        ->select([
            'id', 'transaction_type', 'transaction_amount', 'transaction_date', 
            'transaction_time', 'person_name', 'note', 'from_account_id', 
            'to_account_id', 'transaction_group_id', 'status', 'created_at', 'get_fee'
        ])
        ->where('transaction_type', '!=', 'fee')
        ->where('status', '!=', 'rejected');
        
        // Fix customer account filtering logic
        if (!empty($this->filters['customer_account'])) {
            // Show all transactions related to the selected customer account
            $query->where(function($q) {
                $q->where('from_account_id', $this->filters['customer_account'])
                  ->orWhere('to_account_id', $this->filters['customer_account']);
            });
        } else {
            // When no specific account is selected, show all customer transactions
            // Exclude system accounts (customer_id = 1) and cashier accounts
            $customer_cashiers_ids = Customer::where('customer_name', 'Cashiers')->pluck('id');
            
            $query->where(function($q) use ($customer_cashiers_ids) {
                $q->whereHas('from_account', function($subQ) use ($customer_cashiers_ids) {
                    $subQ->where('customer_id', '!=', 1)
                         ->whereNotIn('customer_id', $customer_cashiers_ids);
                })
                ->orWhereHas('to_account', function($subQ) use ($customer_cashiers_ids) {
                    $subQ->where('customer_id', '!=', 1)
                         ->whereNotIn('customer_id', $customer_cashiers_ids);
                });
            });
        }

        // Fix date filtering logic
        if ($this->startDate && $this->endDate) {
            // If both dates are the same, filter for that specific day
            if ($this->startDate === $this->endDate) {
                $query->whereDate('transaction_date', $this->startDate);
            } else {
                // Date range filtering
                $query->whereBetween('transaction_date', [$this->startDate, $this->endDate]);
            }
        } elseif ($this->startDate) {
            // Only start date provided
            $query->whereDate('transaction_date', '>=', $this->startDate);
        } elseif ($this->endDate) {
            // Only end date provided
            $query->whereDate('transaction_date', '<=', $this->endDate);
        }

        $account_transactions = $query->orderBy('created_at', 'desc')
            ->paginate(1000, ['*'], 'page')
            ->onEachSide(1);

        return view('livewire.account-transaction', [
            'account_transactions' => $account_transactions
        ]);
    }

    // OPTIMIZATION 6: Add method to delete transactions efficiently
    public function deleteTransaction($transactionGroupId)
    {
        try {
            DB::beginTransaction();
            
            // Delete related fees first
            $transactionIds = AccountTransactionModel::where('transaction_group_id', $transactionGroupId)
                ->pluck('id');
                
            TransactionFee::where('feeable_type', AccountTransactionModel::class)
                ->whereIn('feeable_id', $transactionIds)
                ->delete();
            
            // Delete transactions
            AccountTransactionModel::where('transaction_group_id', $transactionGroupId)->delete();
            
            DB::commit();
            
            session()->flash('success', 'جولەکان بەسەرکەوتووی سڕایەوە');
            
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Transaction deletion failed: ' . $e->getMessage());
            session()->flash('error', 'هەڵەیەک ڕوویدا لە کاتی سڕینەوەدا');
        }
    }

    /**
     * Send notification for new transaction
     */
    private function sendTransactionNotification(AccountTransactionModel $transaction)
    {
        // Resolve the notification service when needed
        $notificationService = app(TransactionNotificationService::class);
        $notificationService->sendTransactionCreatedNotification($transaction);
    }

    /**
     * Send notification for transaction update
     */
    private function sendTransactionUpdateNotification(AccountTransactionModel $transaction)
    {
        // Resolve the notification service when needed
        $notificationService = app(TransactionNotificationService::class);
        $notificationService->sendTransactionCreatedNotification($transaction);
    }
}